var __defProp = Object.defineProperty;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __publicField = (obj, key, value) => __defNormalProp(obj, typeof key !== "symbol" ? key + "" : key, value);
class Result {
  constructor(value, error) {
    this.value = value;
    this.error = error;
  }
  unwrap() {
    if (this.error) {
      throw this.error;
    }
    return this.value;
  }
  isOk() {
    return !this.error;
  }
  isErr() {
    return !!this.error;
  }
  static ok(value) {
    return new OK$1(value);
  }
  static err(error) {
    return new Err(error);
  }
  map(fn) {
    if (this.isOk()) {
      try {
        const res = fn(this.value);
        return Result.ok(res);
      } catch (ex) {
        return Result.err(ex);
      }
    }
    return this;
  }
  mapErr(fn) {
    if (this.isErr()) {
      return Result.err(fn(this.error));
    }
    return this;
  }
  andThen(fn) {
    if (this.isOk()) {
      return fn(this.value);
    }
    return this;
  }
  flatMap(fn) {
    if (this.isOk()) {
      try {
        if (Array.isArray(this.value)) {
          const result = this.value.flatMap(fn);
          return Result.ok(result);
        } else {
          return Result.ok([]);
        }
      } catch (ex) {
        return Result.err(ex);
      }
    }
    return this;
  }
  toAsyncResult() {
    return AsyncResult.fromResult(this);
  }
}
class AsyncResult extends Result {
  constructor(value, error, isloading) {
    super(value, error);
    this.value = value;
    this.error = error;
    this.isloading = isloading;
  }
  isLoading() {
    return !!this.isloading;
  }
  isOk() {
    return !this.error && !this.isloading;
  }
  isErr() {
    return !!this.error;
  }
  static ok(value) {
    return new OK$1(value);
  }
  static err(error) {
    return new Err(error);
  }
  static loading() {
    return new Loading();
  }
  static fromResult(result) {
    if (result.isOk()) {
      return AsyncResult.ok(result.value);
    } else if (result.isErr()) {
      return AsyncResult.err(result.error);
    } else {
      throw new Error("Invalid result");
    }
  }
  flatMap(fn) {
    if (this.isOk()) {
      try {
        if (Array.isArray(this.value)) {
          const result = this.value.flatMap(fn);
          return AsyncResult.ok(result);
        } else {
          return AsyncResult.ok([]);
        }
      } catch (ex) {
        return AsyncResult.err(ex);
      }
    }
    return this;
  }
}
let OK$1 = class OK extends AsyncResult {
  constructor(value) {
    super(value, void 0, false);
    this.value = value;
  }
};
class Err extends AsyncResult {
  constructor(error) {
    super(void 0, error, false);
    this.error = error;
  }
}
class Loading extends AsyncResult {
  constructor() {
    super(void 0, void 0, true);
  }
}
function genId() {
  return Date.now().toString(36) + Math.random().toString(36).substring(2);
}
var util;
(function(util2) {
  util2.assertEqual = (val) => val;
  function assertIs(_arg) {
  }
  util2.assertIs = assertIs;
  function assertNever(_x) {
    throw new Error();
  }
  util2.assertNever = assertNever;
  util2.arrayToEnum = (items) => {
    const obj = {};
    for (const item of items) {
      obj[item] = item;
    }
    return obj;
  };
  util2.getValidEnumValues = (obj) => {
    const validKeys = util2.objectKeys(obj).filter((k) => typeof obj[obj[k]] !== "number");
    const filtered = {};
    for (const k of validKeys) {
      filtered[k] = obj[k];
    }
    return util2.objectValues(filtered);
  };
  util2.objectValues = (obj) => {
    return util2.objectKeys(obj).map(function(e) {
      return obj[e];
    });
  };
  util2.objectKeys = typeof Object.keys === "function" ? (obj) => Object.keys(obj) : (object) => {
    const keys = [];
    for (const key in object) {
      if (Object.prototype.hasOwnProperty.call(object, key)) {
        keys.push(key);
      }
    }
    return keys;
  };
  util2.find = (arr, checker) => {
    for (const item of arr) {
      if (checker(item))
        return item;
    }
    return void 0;
  };
  util2.isInteger = typeof Number.isInteger === "function" ? (val) => Number.isInteger(val) : (val) => typeof val === "number" && isFinite(val) && Math.floor(val) === val;
  function joinValues(array, separator = " | ") {
    return array.map((val) => typeof val === "string" ? `'${val}'` : val).join(separator);
  }
  util2.joinValues = joinValues;
  util2.jsonStringifyReplacer = (_, value) => {
    if (typeof value === "bigint") {
      return value.toString();
    }
    return value;
  };
})(util || (util = {}));
var objectUtil;
(function(objectUtil2) {
  objectUtil2.mergeShapes = (first, second) => {
    return {
      ...first,
      ...second
      // second overwrites first
    };
  };
})(objectUtil || (objectUtil = {}));
const ZodParsedType = util.arrayToEnum([
  "string",
  "nan",
  "number",
  "integer",
  "float",
  "boolean",
  "date",
  "bigint",
  "symbol",
  "function",
  "undefined",
  "null",
  "array",
  "object",
  "unknown",
  "promise",
  "void",
  "never",
  "map",
  "set"
]);
const getParsedType = (data) => {
  const t = typeof data;
  switch (t) {
    case "undefined":
      return ZodParsedType.undefined;
    case "string":
      return ZodParsedType.string;
    case "number":
      return isNaN(data) ? ZodParsedType.nan : ZodParsedType.number;
    case "boolean":
      return ZodParsedType.boolean;
    case "function":
      return ZodParsedType.function;
    case "bigint":
      return ZodParsedType.bigint;
    case "symbol":
      return ZodParsedType.symbol;
    case "object":
      if (Array.isArray(data)) {
        return ZodParsedType.array;
      }
      if (data === null) {
        return ZodParsedType.null;
      }
      if (data.then && typeof data.then === "function" && data.catch && typeof data.catch === "function") {
        return ZodParsedType.promise;
      }
      if (typeof Map !== "undefined" && data instanceof Map) {
        return ZodParsedType.map;
      }
      if (typeof Set !== "undefined" && data instanceof Set) {
        return ZodParsedType.set;
      }
      if (typeof Date !== "undefined" && data instanceof Date) {
        return ZodParsedType.date;
      }
      return ZodParsedType.object;
    default:
      return ZodParsedType.unknown;
  }
};
const ZodIssueCode = util.arrayToEnum([
  "invalid_type",
  "invalid_literal",
  "custom",
  "invalid_union",
  "invalid_union_discriminator",
  "invalid_enum_value",
  "unrecognized_keys",
  "invalid_arguments",
  "invalid_return_type",
  "invalid_date",
  "invalid_string",
  "too_small",
  "too_big",
  "invalid_intersection_types",
  "not_multiple_of",
  "not_finite"
]);
const quotelessJson = (obj) => {
  const json = JSON.stringify(obj, null, 2);
  return json.replace(/"([^"]+)":/g, "$1:");
};
class ZodError extends Error {
  get errors() {
    return this.issues;
  }
  constructor(issues) {
    super();
    this.issues = [];
    this.addIssue = (sub) => {
      this.issues = [...this.issues, sub];
    };
    this.addIssues = (subs = []) => {
      this.issues = [...this.issues, ...subs];
    };
    const actualProto = new.target.prototype;
    if (Object.setPrototypeOf) {
      Object.setPrototypeOf(this, actualProto);
    } else {
      this.__proto__ = actualProto;
    }
    this.name = "ZodError";
    this.issues = issues;
  }
  format(_mapper) {
    const mapper = _mapper || function(issue) {
      return issue.message;
    };
    const fieldErrors = { _errors: [] };
    const processError = (error) => {
      for (const issue of error.issues) {
        if (issue.code === "invalid_union") {
          issue.unionErrors.map(processError);
        } else if (issue.code === "invalid_return_type") {
          processError(issue.returnTypeError);
        } else if (issue.code === "invalid_arguments") {
          processError(issue.argumentsError);
        } else if (issue.path.length === 0) {
          fieldErrors._errors.push(mapper(issue));
        } else {
          let curr = fieldErrors;
          let i = 0;
          while (i < issue.path.length) {
            const el = issue.path[i];
            const terminal = i === issue.path.length - 1;
            if (!terminal) {
              curr[el] = curr[el] || { _errors: [] };
            } else {
              curr[el] = curr[el] || { _errors: [] };
              curr[el]._errors.push(mapper(issue));
            }
            curr = curr[el];
            i++;
          }
        }
      }
    };
    processError(this);
    return fieldErrors;
  }
  static assert(value) {
    if (!(value instanceof ZodError)) {
      throw new Error(`Not a ZodError: ${value}`);
    }
  }
  toString() {
    return this.message;
  }
  get message() {
    return JSON.stringify(this.issues, util.jsonStringifyReplacer, 2);
  }
  get isEmpty() {
    return this.issues.length === 0;
  }
  flatten(mapper = (issue) => issue.message) {
    const fieldErrors = {};
    const formErrors = [];
    for (const sub of this.issues) {
      if (sub.path.length > 0) {
        fieldErrors[sub.path[0]] = fieldErrors[sub.path[0]] || [];
        fieldErrors[sub.path[0]].push(mapper(sub));
      } else {
        formErrors.push(mapper(sub));
      }
    }
    return { formErrors, fieldErrors };
  }
  get formErrors() {
    return this.flatten();
  }
}
ZodError.create = (issues) => {
  const error = new ZodError(issues);
  return error;
};
const errorMap = (issue, _ctx) => {
  let message;
  switch (issue.code) {
    case ZodIssueCode.invalid_type:
      if (issue.received === ZodParsedType.undefined) {
        message = "Required";
      } else {
        message = `Expected ${issue.expected}, received ${issue.received}`;
      }
      break;
    case ZodIssueCode.invalid_literal:
      message = `Invalid literal value, expected ${JSON.stringify(issue.expected, util.jsonStringifyReplacer)}`;
      break;
    case ZodIssueCode.unrecognized_keys:
      message = `Unrecognized key(s) in object: ${util.joinValues(issue.keys, ", ")}`;
      break;
    case ZodIssueCode.invalid_union:
      message = `Invalid input`;
      break;
    case ZodIssueCode.invalid_union_discriminator:
      message = `Invalid discriminator value. Expected ${util.joinValues(issue.options)}`;
      break;
    case ZodIssueCode.invalid_enum_value:
      message = `Invalid enum value. Expected ${util.joinValues(issue.options)}, received '${issue.received}'`;
      break;
    case ZodIssueCode.invalid_arguments:
      message = `Invalid function arguments`;
      break;
    case ZodIssueCode.invalid_return_type:
      message = `Invalid function return type`;
      break;
    case ZodIssueCode.invalid_date:
      message = `Invalid date`;
      break;
    case ZodIssueCode.invalid_string:
      if (typeof issue.validation === "object") {
        if ("includes" in issue.validation) {
          message = `Invalid input: must include "${issue.validation.includes}"`;
          if (typeof issue.validation.position === "number") {
            message = `${message} at one or more positions greater than or equal to ${issue.validation.position}`;
          }
        } else if ("startsWith" in issue.validation) {
          message = `Invalid input: must start with "${issue.validation.startsWith}"`;
        } else if ("endsWith" in issue.validation) {
          message = `Invalid input: must end with "${issue.validation.endsWith}"`;
        } else {
          util.assertNever(issue.validation);
        }
      } else if (issue.validation !== "regex") {
        message = `Invalid ${issue.validation}`;
      } else {
        message = "Invalid";
      }
      break;
    case ZodIssueCode.too_small:
      if (issue.type === "array")
        message = `Array must contain ${issue.exact ? "exactly" : issue.inclusive ? `at least` : `more than`} ${issue.minimum} element(s)`;
      else if (issue.type === "string")
        message = `String must contain ${issue.exact ? "exactly" : issue.inclusive ? `at least` : `over`} ${issue.minimum} character(s)`;
      else if (issue.type === "number")
        message = `Number must be ${issue.exact ? `exactly equal to ` : issue.inclusive ? `greater than or equal to ` : `greater than `}${issue.minimum}`;
      else if (issue.type === "date")
        message = `Date must be ${issue.exact ? `exactly equal to ` : issue.inclusive ? `greater than or equal to ` : `greater than `}${new Date(Number(issue.minimum))}`;
      else
        message = "Invalid input";
      break;
    case ZodIssueCode.too_big:
      if (issue.type === "array")
        message = `Array must contain ${issue.exact ? `exactly` : issue.inclusive ? `at most` : `less than`} ${issue.maximum} element(s)`;
      else if (issue.type === "string")
        message = `String must contain ${issue.exact ? `exactly` : issue.inclusive ? `at most` : `under`} ${issue.maximum} character(s)`;
      else if (issue.type === "number")
        message = `Number must be ${issue.exact ? `exactly` : issue.inclusive ? `less than or equal to` : `less than`} ${issue.maximum}`;
      else if (issue.type === "bigint")
        message = `BigInt must be ${issue.exact ? `exactly` : issue.inclusive ? `less than or equal to` : `less than`} ${issue.maximum}`;
      else if (issue.type === "date")
        message = `Date must be ${issue.exact ? `exactly` : issue.inclusive ? `smaller than or equal to` : `smaller than`} ${new Date(Number(issue.maximum))}`;
      else
        message = "Invalid input";
      break;
    case ZodIssueCode.custom:
      message = `Invalid input`;
      break;
    case ZodIssueCode.invalid_intersection_types:
      message = `Intersection results could not be merged`;
      break;
    case ZodIssueCode.not_multiple_of:
      message = `Number must be a multiple of ${issue.multipleOf}`;
      break;
    case ZodIssueCode.not_finite:
      message = "Number must be finite";
      break;
    default:
      message = _ctx.defaultError;
      util.assertNever(issue);
  }
  return { message };
};
let overrideErrorMap = errorMap;
function setErrorMap(map) {
  overrideErrorMap = map;
}
function getErrorMap() {
  return overrideErrorMap;
}
const makeIssue = (params) => {
  const { data, path, errorMaps, issueData } = params;
  const fullPath = [...path, ...issueData.path || []];
  const fullIssue = {
    ...issueData,
    path: fullPath
  };
  if (issueData.message !== void 0) {
    return {
      ...issueData,
      path: fullPath,
      message: issueData.message
    };
  }
  let errorMessage = "";
  const maps = errorMaps.filter((m) => !!m).slice().reverse();
  for (const map of maps) {
    errorMessage = map(fullIssue, { data, defaultError: errorMessage }).message;
  }
  return {
    ...issueData,
    path: fullPath,
    message: errorMessage
  };
};
const EMPTY_PATH = [];
function addIssueToContext(ctx, issueData) {
  const overrideMap = getErrorMap();
  const issue = makeIssue({
    issueData,
    data: ctx.data,
    path: ctx.path,
    errorMaps: [
      ctx.common.contextualErrorMap,
      // contextual error map is first priority
      ctx.schemaErrorMap,
      // then schema-bound map if available
      overrideMap,
      // then global override map
      overrideMap === errorMap ? void 0 : errorMap
      // then global default map
    ].filter((x) => !!x)
  });
  ctx.common.issues.push(issue);
}
class ParseStatus {
  constructor() {
    this.value = "valid";
  }
  dirty() {
    if (this.value === "valid")
      this.value = "dirty";
  }
  abort() {
    if (this.value !== "aborted")
      this.value = "aborted";
  }
  static mergeArray(status, results) {
    const arrayValue = [];
    for (const s of results) {
      if (s.status === "aborted")
        return INVALID;
      if (s.status === "dirty")
        status.dirty();
      arrayValue.push(s.value);
    }
    return { status: status.value, value: arrayValue };
  }
  static async mergeObjectAsync(status, pairs) {
    const syncPairs = [];
    for (const pair of pairs) {
      const key = await pair.key;
      const value = await pair.value;
      syncPairs.push({
        key,
        value
      });
    }
    return ParseStatus.mergeObjectSync(status, syncPairs);
  }
  static mergeObjectSync(status, pairs) {
    const finalObject = {};
    for (const pair of pairs) {
      const { key, value } = pair;
      if (key.status === "aborted")
        return INVALID;
      if (value.status === "aborted")
        return INVALID;
      if (key.status === "dirty")
        status.dirty();
      if (value.status === "dirty")
        status.dirty();
      if (key.value !== "__proto__" && (typeof value.value !== "undefined" || pair.alwaysSet)) {
        finalObject[key.value] = value.value;
      }
    }
    return { status: status.value, value: finalObject };
  }
}
const INVALID = Object.freeze({
  status: "aborted"
});
const DIRTY = (value) => ({ status: "dirty", value });
const OK2 = (value) => ({ status: "valid", value });
const isAborted = (x) => x.status === "aborted";
const isDirty = (x) => x.status === "dirty";
const isValid = (x) => x.status === "valid";
const isAsync = (x) => typeof Promise !== "undefined" && x instanceof Promise;
function __classPrivateFieldGet(receiver, state, kind, f) {
  if (typeof state === "function" ? receiver !== state || true : !state.has(receiver)) throw new TypeError("Cannot read private member from an object whose class did not declare it");
  return state.get(receiver);
}
function __classPrivateFieldSet(receiver, state, value, kind, f) {
  if (typeof state === "function" ? receiver !== state || true : !state.has(receiver)) throw new TypeError("Cannot write private member to an object whose class did not declare it");
  return state.set(receiver, value), value;
}
typeof SuppressedError === "function" ? SuppressedError : function(error, suppressed, message) {
  var e = new Error(message);
  return e.name = "SuppressedError", e.error = error, e.suppressed = suppressed, e;
};
var errorUtil;
(function(errorUtil2) {
  errorUtil2.errToObj = (message) => typeof message === "string" ? { message } : message || {};
  errorUtil2.toString = (message) => typeof message === "string" ? message : message === null || message === void 0 ? void 0 : message.message;
})(errorUtil || (errorUtil = {}));
var _ZodEnum_cache, _ZodNativeEnum_cache;
class ParseInputLazyPath {
  constructor(parent, value, path, key) {
    this._cachedPath = [];
    this.parent = parent;
    this.data = value;
    this._path = path;
    this._key = key;
  }
  get path() {
    if (!this._cachedPath.length) {
      if (this._key instanceof Array) {
        this._cachedPath.push(...this._path, ...this._key);
      } else {
        this._cachedPath.push(...this._path, this._key);
      }
    }
    return this._cachedPath;
  }
}
const handleResult = (ctx, result) => {
  if (isValid(result)) {
    return { success: true, data: result.value };
  } else {
    if (!ctx.common.issues.length) {
      throw new Error("Validation failed but no issues detected.");
    }
    return {
      success: false,
      get error() {
        if (this._error)
          return this._error;
        const error = new ZodError(ctx.common.issues);
        this._error = error;
        return this._error;
      }
    };
  }
};
function processCreateParams(params) {
  if (!params)
    return {};
  const { errorMap: errorMap2, invalid_type_error, required_error, description } = params;
  if (errorMap2 && (invalid_type_error || required_error)) {
    throw new Error(`Can't use "invalid_type_error" or "required_error" in conjunction with custom error map.`);
  }
  if (errorMap2)
    return { errorMap: errorMap2, description };
  const customMap = (iss, ctx) => {
    var _a, _b;
    const { message } = params;
    if (iss.code === "invalid_enum_value") {
      return { message: message !== null && message !== void 0 ? message : ctx.defaultError };
    }
    if (typeof ctx.data === "undefined") {
      return { message: (_a = message !== null && message !== void 0 ? message : required_error) !== null && _a !== void 0 ? _a : ctx.defaultError };
    }
    if (iss.code !== "invalid_type")
      return { message: ctx.defaultError };
    return { message: (_b = message !== null && message !== void 0 ? message : invalid_type_error) !== null && _b !== void 0 ? _b : ctx.defaultError };
  };
  return { errorMap: customMap, description };
}
class ZodType {
  get description() {
    return this._def.description;
  }
  _getType(input) {
    return getParsedType(input.data);
  }
  _getOrReturnCtx(input, ctx) {
    return ctx || {
      common: input.parent.common,
      data: input.data,
      parsedType: getParsedType(input.data),
      schemaErrorMap: this._def.errorMap,
      path: input.path,
      parent: input.parent
    };
  }
  _processInputParams(input) {
    return {
      status: new ParseStatus(),
      ctx: {
        common: input.parent.common,
        data: input.data,
        parsedType: getParsedType(input.data),
        schemaErrorMap: this._def.errorMap,
        path: input.path,
        parent: input.parent
      }
    };
  }
  _parseSync(input) {
    const result = this._parse(input);
    if (isAsync(result)) {
      throw new Error("Synchronous parse encountered promise.");
    }
    return result;
  }
  _parseAsync(input) {
    const result = this._parse(input);
    return Promise.resolve(result);
  }
  parse(data, params) {
    const result = this.safeParse(data, params);
    if (result.success)
      return result.data;
    throw result.error;
  }
  safeParse(data, params) {
    var _a;
    const ctx = {
      common: {
        issues: [],
        async: (_a = params === null || params === void 0 ? void 0 : params.async) !== null && _a !== void 0 ? _a : false,
        contextualErrorMap: params === null || params === void 0 ? void 0 : params.errorMap
      },
      path: (params === null || params === void 0 ? void 0 : params.path) || [],
      schemaErrorMap: this._def.errorMap,
      parent: null,
      data,
      parsedType: getParsedType(data)
    };
    const result = this._parseSync({ data, path: ctx.path, parent: ctx });
    return handleResult(ctx, result);
  }
  "~validate"(data) {
    var _a, _b;
    const ctx = {
      common: {
        issues: [],
        async: !!this["~standard"].async
      },
      path: [],
      schemaErrorMap: this._def.errorMap,
      parent: null,
      data,
      parsedType: getParsedType(data)
    };
    if (!this["~standard"].async) {
      try {
        const result = this._parseSync({ data, path: [], parent: ctx });
        return isValid(result) ? {
          value: result.value
        } : {
          issues: ctx.common.issues
        };
      } catch (err) {
        if ((_b = (_a = err === null || err === void 0 ? void 0 : err.message) === null || _a === void 0 ? void 0 : _a.toLowerCase()) === null || _b === void 0 ? void 0 : _b.includes("encountered")) {
          this["~standard"].async = true;
        }
        ctx.common = {
          issues: [],
          async: true
        };
      }
    }
    return this._parseAsync({ data, path: [], parent: ctx }).then((result) => isValid(result) ? {
      value: result.value
    } : {
      issues: ctx.common.issues
    });
  }
  async parseAsync(data, params) {
    const result = await this.safeParseAsync(data, params);
    if (result.success)
      return result.data;
    throw result.error;
  }
  async safeParseAsync(data, params) {
    const ctx = {
      common: {
        issues: [],
        contextualErrorMap: params === null || params === void 0 ? void 0 : params.errorMap,
        async: true
      },
      path: (params === null || params === void 0 ? void 0 : params.path) || [],
      schemaErrorMap: this._def.errorMap,
      parent: null,
      data,
      parsedType: getParsedType(data)
    };
    const maybeAsyncResult = this._parse({ data, path: ctx.path, parent: ctx });
    const result = await (isAsync(maybeAsyncResult) ? maybeAsyncResult : Promise.resolve(maybeAsyncResult));
    return handleResult(ctx, result);
  }
  refine(check, message) {
    const getIssueProperties = (val) => {
      if (typeof message === "string" || typeof message === "undefined") {
        return { message };
      } else if (typeof message === "function") {
        return message(val);
      } else {
        return message;
      }
    };
    return this._refinement((val, ctx) => {
      const result = check(val);
      const setError = () => ctx.addIssue({
        code: ZodIssueCode.custom,
        ...getIssueProperties(val)
      });
      if (typeof Promise !== "undefined" && result instanceof Promise) {
        return result.then((data) => {
          if (!data) {
            setError();
            return false;
          } else {
            return true;
          }
        });
      }
      if (!result) {
        setError();
        return false;
      } else {
        return true;
      }
    });
  }
  refinement(check, refinementData) {
    return this._refinement((val, ctx) => {
      if (!check(val)) {
        ctx.addIssue(typeof refinementData === "function" ? refinementData(val, ctx) : refinementData);
        return false;
      } else {
        return true;
      }
    });
  }
  _refinement(refinement) {
    return new ZodEffects({
      schema: this,
      typeName: ZodFirstPartyTypeKind.ZodEffects,
      effect: { type: "refinement", refinement }
    });
  }
  superRefine(refinement) {
    return this._refinement(refinement);
  }
  constructor(def) {
    this.spa = this.safeParseAsync;
    this._def = def;
    this.parse = this.parse.bind(this);
    this.safeParse = this.safeParse.bind(this);
    this.parseAsync = this.parseAsync.bind(this);
    this.safeParseAsync = this.safeParseAsync.bind(this);
    this.spa = this.spa.bind(this);
    this.refine = this.refine.bind(this);
    this.refinement = this.refinement.bind(this);
    this.superRefine = this.superRefine.bind(this);
    this.optional = this.optional.bind(this);
    this.nullable = this.nullable.bind(this);
    this.nullish = this.nullish.bind(this);
    this.array = this.array.bind(this);
    this.promise = this.promise.bind(this);
    this.or = this.or.bind(this);
    this.and = this.and.bind(this);
    this.transform = this.transform.bind(this);
    this.brand = this.brand.bind(this);
    this.default = this.default.bind(this);
    this.catch = this.catch.bind(this);
    this.describe = this.describe.bind(this);
    this.pipe = this.pipe.bind(this);
    this.readonly = this.readonly.bind(this);
    this.isNullable = this.isNullable.bind(this);
    this.isOptional = this.isOptional.bind(this);
    this["~standard"] = {
      version: 1,
      vendor: "zod",
      validate: (data) => this["~validate"](data)
    };
  }
  optional() {
    return ZodOptional.create(this, this._def);
  }
  nullable() {
    return ZodNullable.create(this, this._def);
  }
  nullish() {
    return this.nullable().optional();
  }
  array() {
    return ZodArray.create(this);
  }
  promise() {
    return ZodPromise.create(this, this._def);
  }
  or(option) {
    return ZodUnion.create([this, option], this._def);
  }
  and(incoming) {
    return ZodIntersection.create(this, incoming, this._def);
  }
  transform(transform) {
    return new ZodEffects({
      ...processCreateParams(this._def),
      schema: this,
      typeName: ZodFirstPartyTypeKind.ZodEffects,
      effect: { type: "transform", transform }
    });
  }
  default(def) {
    const defaultValueFunc = typeof def === "function" ? def : () => def;
    return new ZodDefault({
      ...processCreateParams(this._def),
      innerType: this,
      defaultValue: defaultValueFunc,
      typeName: ZodFirstPartyTypeKind.ZodDefault
    });
  }
  brand() {
    return new ZodBranded({
      typeName: ZodFirstPartyTypeKind.ZodBranded,
      type: this,
      ...processCreateParams(this._def)
    });
  }
  catch(def) {
    const catchValueFunc = typeof def === "function" ? def : () => def;
    return new ZodCatch({
      ...processCreateParams(this._def),
      innerType: this,
      catchValue: catchValueFunc,
      typeName: ZodFirstPartyTypeKind.ZodCatch
    });
  }
  describe(description) {
    const This = this.constructor;
    return new This({
      ...this._def,
      description
    });
  }
  pipe(target) {
    return ZodPipeline.create(this, target);
  }
  readonly() {
    return ZodReadonly.create(this);
  }
  isOptional() {
    return this.safeParse(void 0).success;
  }
  isNullable() {
    return this.safeParse(null).success;
  }
}
const cuidRegex = /^c[^\s-]{8,}$/i;
const cuid2Regex = /^[0-9a-z]+$/;
const ulidRegex = /^[0-9A-HJKMNP-TV-Z]{26}$/i;
const uuidRegex = /^[0-9a-fA-F]{8}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{12}$/i;
const nanoidRegex = /^[a-z0-9_-]{21}$/i;
const jwtRegex = /^[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+\.[A-Za-z0-9-_]*$/;
const durationRegex = /^[-+]?P(?!$)(?:(?:[-+]?\d+Y)|(?:[-+]?\d+[.,]\d+Y$))?(?:(?:[-+]?\d+M)|(?:[-+]?\d+[.,]\d+M$))?(?:(?:[-+]?\d+W)|(?:[-+]?\d+[.,]\d+W$))?(?:(?:[-+]?\d+D)|(?:[-+]?\d+[.,]\d+D$))?(?:T(?=[\d+-])(?:(?:[-+]?\d+H)|(?:[-+]?\d+[.,]\d+H$))?(?:(?:[-+]?\d+M)|(?:[-+]?\d+[.,]\d+M$))?(?:[-+]?\d+(?:[.,]\d+)?S)?)??$/;
const emailRegex = /^(?!\.)(?!.*\.\.)([A-Z0-9_'+\-\.]*)[A-Z0-9_+-]@([A-Z0-9][A-Z0-9\-]*\.)+[A-Z]{2,}$/i;
const _emojiRegex = `^(\\p{Extended_Pictographic}|\\p{Emoji_Component})+$`;
let emojiRegex;
const ipv4Regex = /^(?:(?:25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\.){3}(?:25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])$/;
const ipv4CidrRegex = /^(?:(?:25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\.){3}(?:25[0-5]|2[0-4][0-9]|1[0-9][0-9]|[1-9][0-9]|[0-9])\/(3[0-2]|[12]?[0-9])$/;
const ipv6Regex = /^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$/;
const ipv6CidrRegex = /^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))\/(12[0-8]|1[01][0-9]|[1-9]?[0-9])$/;
const base64Regex = /^([0-9a-zA-Z+/]{4})*(([0-9a-zA-Z+/]{2}==)|([0-9a-zA-Z+/]{3}=))?$/;
const base64urlRegex = /^([0-9a-zA-Z-_]{4})*(([0-9a-zA-Z-_]{2}(==)?)|([0-9a-zA-Z-_]{3}(=)?))?$/;
const dateRegexSource = `((\\d\\d[2468][048]|\\d\\d[13579][26]|\\d\\d0[48]|[02468][048]00|[13579][26]00)-02-29|\\d{4}-((0[13578]|1[02])-(0[1-9]|[12]\\d|3[01])|(0[469]|11)-(0[1-9]|[12]\\d|30)|(02)-(0[1-9]|1\\d|2[0-8])))`;
const dateRegex = new RegExp(`^${dateRegexSource}$`);
function timeRegexSource(args) {
  let secondsRegexSource = `[0-5]\\d`;
  if (args.precision) {
    secondsRegexSource = `${secondsRegexSource}\\.\\d{${args.precision}}`;
  } else if (args.precision == null) {
    secondsRegexSource = `${secondsRegexSource}(\\.\\d+)?`;
  }
  const secondsQuantifier = args.precision ? "+" : "?";
  return `([01]\\d|2[0-3]):[0-5]\\d(:${secondsRegexSource})${secondsQuantifier}`;
}
function timeRegex(args) {
  return new RegExp(`^${timeRegexSource(args)}$`);
}
function datetimeRegex(args) {
  let regex = `${dateRegexSource}T${timeRegexSource(args)}`;
  const opts = [];
  opts.push(args.local ? `Z?` : `Z`);
  if (args.offset)
    opts.push(`([+-]\\d{2}:?\\d{2})`);
  regex = `${regex}(${opts.join("|")})`;
  return new RegExp(`^${regex}$`);
}
function isValidIP(ip, version) {
  if ((version === "v4" || !version) && ipv4Regex.test(ip)) {
    return true;
  }
  if ((version === "v6" || !version) && ipv6Regex.test(ip)) {
    return true;
  }
  return false;
}
function isValidJWT(jwt, alg) {
  if (!jwtRegex.test(jwt))
    return false;
  try {
    const [header] = jwt.split(".");
    const base64 = header.replace(/-/g, "+").replace(/_/g, "/").padEnd(header.length + (4 - header.length % 4) % 4, "=");
    const decoded = JSON.parse(atob(base64));
    if (typeof decoded !== "object" || decoded === null)
      return false;
    if (!decoded.typ || !decoded.alg)
      return false;
    if (alg && decoded.alg !== alg)
      return false;
    return true;
  } catch (_a) {
    return false;
  }
}
function isValidCidr(ip, version) {
  if ((version === "v4" || !version) && ipv4CidrRegex.test(ip)) {
    return true;
  }
  if ((version === "v6" || !version) && ipv6CidrRegex.test(ip)) {
    return true;
  }
  return false;
}
class ZodString extends ZodType {
  _parse(input) {
    if (this._def.coerce) {
      input.data = String(input.data);
    }
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.string) {
      const ctx2 = this._getOrReturnCtx(input);
      addIssueToContext(ctx2, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.string,
        received: ctx2.parsedType
      });
      return INVALID;
    }
    const status = new ParseStatus();
    let ctx = void 0;
    for (const check of this._def.checks) {
      if (check.kind === "min") {
        if (input.data.length < check.value) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_small,
            minimum: check.value,
            type: "string",
            inclusive: true,
            exact: false,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "max") {
        if (input.data.length > check.value) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_big,
            maximum: check.value,
            type: "string",
            inclusive: true,
            exact: false,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "length") {
        const tooBig = input.data.length > check.value;
        const tooSmall = input.data.length < check.value;
        if (tooBig || tooSmall) {
          ctx = this._getOrReturnCtx(input, ctx);
          if (tooBig) {
            addIssueToContext(ctx, {
              code: ZodIssueCode.too_big,
              maximum: check.value,
              type: "string",
              inclusive: true,
              exact: true,
              message: check.message
            });
          } else if (tooSmall) {
            addIssueToContext(ctx, {
              code: ZodIssueCode.too_small,
              minimum: check.value,
              type: "string",
              inclusive: true,
              exact: true,
              message: check.message
            });
          }
          status.dirty();
        }
      } else if (check.kind === "email") {
        if (!emailRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "email",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "emoji") {
        if (!emojiRegex) {
          emojiRegex = new RegExp(_emojiRegex, "u");
        }
        if (!emojiRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "emoji",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "uuid") {
        if (!uuidRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "uuid",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "nanoid") {
        if (!nanoidRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "nanoid",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "cuid") {
        if (!cuidRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "cuid",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "cuid2") {
        if (!cuid2Regex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "cuid2",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "ulid") {
        if (!ulidRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "ulid",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "url") {
        try {
          new URL(input.data);
        } catch (_a) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "url",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "regex") {
        check.regex.lastIndex = 0;
        const testResult = check.regex.test(input.data);
        if (!testResult) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "regex",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "trim") {
        input.data = input.data.trim();
      } else if (check.kind === "includes") {
        if (!input.data.includes(check.value, check.position)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.invalid_string,
            validation: { includes: check.value, position: check.position },
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "toLowerCase") {
        input.data = input.data.toLowerCase();
      } else if (check.kind === "toUpperCase") {
        input.data = input.data.toUpperCase();
      } else if (check.kind === "startsWith") {
        if (!input.data.startsWith(check.value)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.invalid_string,
            validation: { startsWith: check.value },
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "endsWith") {
        if (!input.data.endsWith(check.value)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.invalid_string,
            validation: { endsWith: check.value },
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "datetime") {
        const regex = datetimeRegex(check);
        if (!regex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.invalid_string,
            validation: "datetime",
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "date") {
        const regex = dateRegex;
        if (!regex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.invalid_string,
            validation: "date",
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "time") {
        const regex = timeRegex(check);
        if (!regex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.invalid_string,
            validation: "time",
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "duration") {
        if (!durationRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "duration",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "ip") {
        if (!isValidIP(input.data, check.version)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "ip",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "jwt") {
        if (!isValidJWT(input.data, check.alg)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "jwt",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "cidr") {
        if (!isValidCidr(input.data, check.version)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "cidr",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "base64") {
        if (!base64Regex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "base64",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "base64url") {
        if (!base64urlRegex.test(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            validation: "base64url",
            code: ZodIssueCode.invalid_string,
            message: check.message
          });
          status.dirty();
        }
      } else {
        util.assertNever(check);
      }
    }
    return { status: status.value, value: input.data };
  }
  _regex(regex, validation, message) {
    return this.refinement((data) => regex.test(data), {
      validation,
      code: ZodIssueCode.invalid_string,
      ...errorUtil.errToObj(message)
    });
  }
  _addCheck(check) {
    return new ZodString({
      ...this._def,
      checks: [...this._def.checks, check]
    });
  }
  email(message) {
    return this._addCheck({ kind: "email", ...errorUtil.errToObj(message) });
  }
  url(message) {
    return this._addCheck({ kind: "url", ...errorUtil.errToObj(message) });
  }
  emoji(message) {
    return this._addCheck({ kind: "emoji", ...errorUtil.errToObj(message) });
  }
  uuid(message) {
    return this._addCheck({ kind: "uuid", ...errorUtil.errToObj(message) });
  }
  nanoid(message) {
    return this._addCheck({ kind: "nanoid", ...errorUtil.errToObj(message) });
  }
  cuid(message) {
    return this._addCheck({ kind: "cuid", ...errorUtil.errToObj(message) });
  }
  cuid2(message) {
    return this._addCheck({ kind: "cuid2", ...errorUtil.errToObj(message) });
  }
  ulid(message) {
    return this._addCheck({ kind: "ulid", ...errorUtil.errToObj(message) });
  }
  base64(message) {
    return this._addCheck({ kind: "base64", ...errorUtil.errToObj(message) });
  }
  base64url(message) {
    return this._addCheck({
      kind: "base64url",
      ...errorUtil.errToObj(message)
    });
  }
  jwt(options) {
    return this._addCheck({ kind: "jwt", ...errorUtil.errToObj(options) });
  }
  ip(options) {
    return this._addCheck({ kind: "ip", ...errorUtil.errToObj(options) });
  }
  cidr(options) {
    return this._addCheck({ kind: "cidr", ...errorUtil.errToObj(options) });
  }
  datetime(options) {
    var _a, _b;
    if (typeof options === "string") {
      return this._addCheck({
        kind: "datetime",
        precision: null,
        offset: false,
        local: false,
        message: options
      });
    }
    return this._addCheck({
      kind: "datetime",
      precision: typeof (options === null || options === void 0 ? void 0 : options.precision) === "undefined" ? null : options === null || options === void 0 ? void 0 : options.precision,
      offset: (_a = options === null || options === void 0 ? void 0 : options.offset) !== null && _a !== void 0 ? _a : false,
      local: (_b = options === null || options === void 0 ? void 0 : options.local) !== null && _b !== void 0 ? _b : false,
      ...errorUtil.errToObj(options === null || options === void 0 ? void 0 : options.message)
    });
  }
  date(message) {
    return this._addCheck({ kind: "date", message });
  }
  time(options) {
    if (typeof options === "string") {
      return this._addCheck({
        kind: "time",
        precision: null,
        message: options
      });
    }
    return this._addCheck({
      kind: "time",
      precision: typeof (options === null || options === void 0 ? void 0 : options.precision) === "undefined" ? null : options === null || options === void 0 ? void 0 : options.precision,
      ...errorUtil.errToObj(options === null || options === void 0 ? void 0 : options.message)
    });
  }
  duration(message) {
    return this._addCheck({ kind: "duration", ...errorUtil.errToObj(message) });
  }
  regex(regex, message) {
    return this._addCheck({
      kind: "regex",
      regex,
      ...errorUtil.errToObj(message)
    });
  }
  includes(value, options) {
    return this._addCheck({
      kind: "includes",
      value,
      position: options === null || options === void 0 ? void 0 : options.position,
      ...errorUtil.errToObj(options === null || options === void 0 ? void 0 : options.message)
    });
  }
  startsWith(value, message) {
    return this._addCheck({
      kind: "startsWith",
      value,
      ...errorUtil.errToObj(message)
    });
  }
  endsWith(value, message) {
    return this._addCheck({
      kind: "endsWith",
      value,
      ...errorUtil.errToObj(message)
    });
  }
  min(minLength, message) {
    return this._addCheck({
      kind: "min",
      value: minLength,
      ...errorUtil.errToObj(message)
    });
  }
  max(maxLength, message) {
    return this._addCheck({
      kind: "max",
      value: maxLength,
      ...errorUtil.errToObj(message)
    });
  }
  length(len, message) {
    return this._addCheck({
      kind: "length",
      value: len,
      ...errorUtil.errToObj(message)
    });
  }
  /**
   * Equivalent to `.min(1)`
   */
  nonempty(message) {
    return this.min(1, errorUtil.errToObj(message));
  }
  trim() {
    return new ZodString({
      ...this._def,
      checks: [...this._def.checks, { kind: "trim" }]
    });
  }
  toLowerCase() {
    return new ZodString({
      ...this._def,
      checks: [...this._def.checks, { kind: "toLowerCase" }]
    });
  }
  toUpperCase() {
    return new ZodString({
      ...this._def,
      checks: [...this._def.checks, { kind: "toUpperCase" }]
    });
  }
  get isDatetime() {
    return !!this._def.checks.find((ch) => ch.kind === "datetime");
  }
  get isDate() {
    return !!this._def.checks.find((ch) => ch.kind === "date");
  }
  get isTime() {
    return !!this._def.checks.find((ch) => ch.kind === "time");
  }
  get isDuration() {
    return !!this._def.checks.find((ch) => ch.kind === "duration");
  }
  get isEmail() {
    return !!this._def.checks.find((ch) => ch.kind === "email");
  }
  get isURL() {
    return !!this._def.checks.find((ch) => ch.kind === "url");
  }
  get isEmoji() {
    return !!this._def.checks.find((ch) => ch.kind === "emoji");
  }
  get isUUID() {
    return !!this._def.checks.find((ch) => ch.kind === "uuid");
  }
  get isNANOID() {
    return !!this._def.checks.find((ch) => ch.kind === "nanoid");
  }
  get isCUID() {
    return !!this._def.checks.find((ch) => ch.kind === "cuid");
  }
  get isCUID2() {
    return !!this._def.checks.find((ch) => ch.kind === "cuid2");
  }
  get isULID() {
    return !!this._def.checks.find((ch) => ch.kind === "ulid");
  }
  get isIP() {
    return !!this._def.checks.find((ch) => ch.kind === "ip");
  }
  get isCIDR() {
    return !!this._def.checks.find((ch) => ch.kind === "cidr");
  }
  get isBase64() {
    return !!this._def.checks.find((ch) => ch.kind === "base64");
  }
  get isBase64url() {
    return !!this._def.checks.find((ch) => ch.kind === "base64url");
  }
  get minLength() {
    let min = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "min") {
        if (min === null || ch.value > min)
          min = ch.value;
      }
    }
    return min;
  }
  get maxLength() {
    let max = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "max") {
        if (max === null || ch.value < max)
          max = ch.value;
      }
    }
    return max;
  }
}
ZodString.create = (params) => {
  var _a;
  return new ZodString({
    checks: [],
    typeName: ZodFirstPartyTypeKind.ZodString,
    coerce: (_a = params === null || params === void 0 ? void 0 : params.coerce) !== null && _a !== void 0 ? _a : false,
    ...processCreateParams(params)
  });
};
function floatSafeRemainder(val, step) {
  const valDecCount = (val.toString().split(".")[1] || "").length;
  const stepDecCount = (step.toString().split(".")[1] || "").length;
  const decCount = valDecCount > stepDecCount ? valDecCount : stepDecCount;
  const valInt = parseInt(val.toFixed(decCount).replace(".", ""));
  const stepInt = parseInt(step.toFixed(decCount).replace(".", ""));
  return valInt % stepInt / Math.pow(10, decCount);
}
class ZodNumber extends ZodType {
  constructor() {
    super(...arguments);
    this.min = this.gte;
    this.max = this.lte;
    this.step = this.multipleOf;
  }
  _parse(input) {
    if (this._def.coerce) {
      input.data = Number(input.data);
    }
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.number) {
      const ctx2 = this._getOrReturnCtx(input);
      addIssueToContext(ctx2, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.number,
        received: ctx2.parsedType
      });
      return INVALID;
    }
    let ctx = void 0;
    const status = new ParseStatus();
    for (const check of this._def.checks) {
      if (check.kind === "int") {
        if (!util.isInteger(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.invalid_type,
            expected: "integer",
            received: "float",
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "min") {
        const tooSmall = check.inclusive ? input.data < check.value : input.data <= check.value;
        if (tooSmall) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_small,
            minimum: check.value,
            type: "number",
            inclusive: check.inclusive,
            exact: false,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "max") {
        const tooBig = check.inclusive ? input.data > check.value : input.data >= check.value;
        if (tooBig) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_big,
            maximum: check.value,
            type: "number",
            inclusive: check.inclusive,
            exact: false,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "multipleOf") {
        if (floatSafeRemainder(input.data, check.value) !== 0) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.not_multiple_of,
            multipleOf: check.value,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "finite") {
        if (!Number.isFinite(input.data)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.not_finite,
            message: check.message
          });
          status.dirty();
        }
      } else {
        util.assertNever(check);
      }
    }
    return { status: status.value, value: input.data };
  }
  gte(value, message) {
    return this.setLimit("min", value, true, errorUtil.toString(message));
  }
  gt(value, message) {
    return this.setLimit("min", value, false, errorUtil.toString(message));
  }
  lte(value, message) {
    return this.setLimit("max", value, true, errorUtil.toString(message));
  }
  lt(value, message) {
    return this.setLimit("max", value, false, errorUtil.toString(message));
  }
  setLimit(kind, value, inclusive, message) {
    return new ZodNumber({
      ...this._def,
      checks: [
        ...this._def.checks,
        {
          kind,
          value,
          inclusive,
          message: errorUtil.toString(message)
        }
      ]
    });
  }
  _addCheck(check) {
    return new ZodNumber({
      ...this._def,
      checks: [...this._def.checks, check]
    });
  }
  int(message) {
    return this._addCheck({
      kind: "int",
      message: errorUtil.toString(message)
    });
  }
  positive(message) {
    return this._addCheck({
      kind: "min",
      value: 0,
      inclusive: false,
      message: errorUtil.toString(message)
    });
  }
  negative(message) {
    return this._addCheck({
      kind: "max",
      value: 0,
      inclusive: false,
      message: errorUtil.toString(message)
    });
  }
  nonpositive(message) {
    return this._addCheck({
      kind: "max",
      value: 0,
      inclusive: true,
      message: errorUtil.toString(message)
    });
  }
  nonnegative(message) {
    return this._addCheck({
      kind: "min",
      value: 0,
      inclusive: true,
      message: errorUtil.toString(message)
    });
  }
  multipleOf(value, message) {
    return this._addCheck({
      kind: "multipleOf",
      value,
      message: errorUtil.toString(message)
    });
  }
  finite(message) {
    return this._addCheck({
      kind: "finite",
      message: errorUtil.toString(message)
    });
  }
  safe(message) {
    return this._addCheck({
      kind: "min",
      inclusive: true,
      value: Number.MIN_SAFE_INTEGER,
      message: errorUtil.toString(message)
    })._addCheck({
      kind: "max",
      inclusive: true,
      value: Number.MAX_SAFE_INTEGER,
      message: errorUtil.toString(message)
    });
  }
  get minValue() {
    let min = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "min") {
        if (min === null || ch.value > min)
          min = ch.value;
      }
    }
    return min;
  }
  get maxValue() {
    let max = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "max") {
        if (max === null || ch.value < max)
          max = ch.value;
      }
    }
    return max;
  }
  get isInt() {
    return !!this._def.checks.find((ch) => ch.kind === "int" || ch.kind === "multipleOf" && util.isInteger(ch.value));
  }
  get isFinite() {
    let max = null, min = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "finite" || ch.kind === "int" || ch.kind === "multipleOf") {
        return true;
      } else if (ch.kind === "min") {
        if (min === null || ch.value > min)
          min = ch.value;
      } else if (ch.kind === "max") {
        if (max === null || ch.value < max)
          max = ch.value;
      }
    }
    return Number.isFinite(min) && Number.isFinite(max);
  }
}
ZodNumber.create = (params) => {
  return new ZodNumber({
    checks: [],
    typeName: ZodFirstPartyTypeKind.ZodNumber,
    coerce: (params === null || params === void 0 ? void 0 : params.coerce) || false,
    ...processCreateParams(params)
  });
};
class ZodBigInt extends ZodType {
  constructor() {
    super(...arguments);
    this.min = this.gte;
    this.max = this.lte;
  }
  _parse(input) {
    if (this._def.coerce) {
      try {
        input.data = BigInt(input.data);
      } catch (_a) {
        return this._getInvalidInput(input);
      }
    }
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.bigint) {
      return this._getInvalidInput(input);
    }
    let ctx = void 0;
    const status = new ParseStatus();
    for (const check of this._def.checks) {
      if (check.kind === "min") {
        const tooSmall = check.inclusive ? input.data < check.value : input.data <= check.value;
        if (tooSmall) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_small,
            type: "bigint",
            minimum: check.value,
            inclusive: check.inclusive,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "max") {
        const tooBig = check.inclusive ? input.data > check.value : input.data >= check.value;
        if (tooBig) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_big,
            type: "bigint",
            maximum: check.value,
            inclusive: check.inclusive,
            message: check.message
          });
          status.dirty();
        }
      } else if (check.kind === "multipleOf") {
        if (input.data % check.value !== BigInt(0)) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.not_multiple_of,
            multipleOf: check.value,
            message: check.message
          });
          status.dirty();
        }
      } else {
        util.assertNever(check);
      }
    }
    return { status: status.value, value: input.data };
  }
  _getInvalidInput(input) {
    const ctx = this._getOrReturnCtx(input);
    addIssueToContext(ctx, {
      code: ZodIssueCode.invalid_type,
      expected: ZodParsedType.bigint,
      received: ctx.parsedType
    });
    return INVALID;
  }
  gte(value, message) {
    return this.setLimit("min", value, true, errorUtil.toString(message));
  }
  gt(value, message) {
    return this.setLimit("min", value, false, errorUtil.toString(message));
  }
  lte(value, message) {
    return this.setLimit("max", value, true, errorUtil.toString(message));
  }
  lt(value, message) {
    return this.setLimit("max", value, false, errorUtil.toString(message));
  }
  setLimit(kind, value, inclusive, message) {
    return new ZodBigInt({
      ...this._def,
      checks: [
        ...this._def.checks,
        {
          kind,
          value,
          inclusive,
          message: errorUtil.toString(message)
        }
      ]
    });
  }
  _addCheck(check) {
    return new ZodBigInt({
      ...this._def,
      checks: [...this._def.checks, check]
    });
  }
  positive(message) {
    return this._addCheck({
      kind: "min",
      value: BigInt(0),
      inclusive: false,
      message: errorUtil.toString(message)
    });
  }
  negative(message) {
    return this._addCheck({
      kind: "max",
      value: BigInt(0),
      inclusive: false,
      message: errorUtil.toString(message)
    });
  }
  nonpositive(message) {
    return this._addCheck({
      kind: "max",
      value: BigInt(0),
      inclusive: true,
      message: errorUtil.toString(message)
    });
  }
  nonnegative(message) {
    return this._addCheck({
      kind: "min",
      value: BigInt(0),
      inclusive: true,
      message: errorUtil.toString(message)
    });
  }
  multipleOf(value, message) {
    return this._addCheck({
      kind: "multipleOf",
      value,
      message: errorUtil.toString(message)
    });
  }
  get minValue() {
    let min = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "min") {
        if (min === null || ch.value > min)
          min = ch.value;
      }
    }
    return min;
  }
  get maxValue() {
    let max = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "max") {
        if (max === null || ch.value < max)
          max = ch.value;
      }
    }
    return max;
  }
}
ZodBigInt.create = (params) => {
  var _a;
  return new ZodBigInt({
    checks: [],
    typeName: ZodFirstPartyTypeKind.ZodBigInt,
    coerce: (_a = params === null || params === void 0 ? void 0 : params.coerce) !== null && _a !== void 0 ? _a : false,
    ...processCreateParams(params)
  });
};
class ZodBoolean extends ZodType {
  _parse(input) {
    if (this._def.coerce) {
      input.data = Boolean(input.data);
    }
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.boolean) {
      const ctx = this._getOrReturnCtx(input);
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.boolean,
        received: ctx.parsedType
      });
      return INVALID;
    }
    return OK2(input.data);
  }
}
ZodBoolean.create = (params) => {
  return new ZodBoolean({
    typeName: ZodFirstPartyTypeKind.ZodBoolean,
    coerce: (params === null || params === void 0 ? void 0 : params.coerce) || false,
    ...processCreateParams(params)
  });
};
class ZodDate extends ZodType {
  _parse(input) {
    if (this._def.coerce) {
      input.data = new Date(input.data);
    }
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.date) {
      const ctx2 = this._getOrReturnCtx(input);
      addIssueToContext(ctx2, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.date,
        received: ctx2.parsedType
      });
      return INVALID;
    }
    if (isNaN(input.data.getTime())) {
      const ctx2 = this._getOrReturnCtx(input);
      addIssueToContext(ctx2, {
        code: ZodIssueCode.invalid_date
      });
      return INVALID;
    }
    const status = new ParseStatus();
    let ctx = void 0;
    for (const check of this._def.checks) {
      if (check.kind === "min") {
        if (input.data.getTime() < check.value) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_small,
            message: check.message,
            inclusive: true,
            exact: false,
            minimum: check.value,
            type: "date"
          });
          status.dirty();
        }
      } else if (check.kind === "max") {
        if (input.data.getTime() > check.value) {
          ctx = this._getOrReturnCtx(input, ctx);
          addIssueToContext(ctx, {
            code: ZodIssueCode.too_big,
            message: check.message,
            inclusive: true,
            exact: false,
            maximum: check.value,
            type: "date"
          });
          status.dirty();
        }
      } else {
        util.assertNever(check);
      }
    }
    return {
      status: status.value,
      value: new Date(input.data.getTime())
    };
  }
  _addCheck(check) {
    return new ZodDate({
      ...this._def,
      checks: [...this._def.checks, check]
    });
  }
  min(minDate, message) {
    return this._addCheck({
      kind: "min",
      value: minDate.getTime(),
      message: errorUtil.toString(message)
    });
  }
  max(maxDate, message) {
    return this._addCheck({
      kind: "max",
      value: maxDate.getTime(),
      message: errorUtil.toString(message)
    });
  }
  get minDate() {
    let min = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "min") {
        if (min === null || ch.value > min)
          min = ch.value;
      }
    }
    return min != null ? new Date(min) : null;
  }
  get maxDate() {
    let max = null;
    for (const ch of this._def.checks) {
      if (ch.kind === "max") {
        if (max === null || ch.value < max)
          max = ch.value;
      }
    }
    return max != null ? new Date(max) : null;
  }
}
ZodDate.create = (params) => {
  return new ZodDate({
    checks: [],
    coerce: (params === null || params === void 0 ? void 0 : params.coerce) || false,
    typeName: ZodFirstPartyTypeKind.ZodDate,
    ...processCreateParams(params)
  });
};
class ZodSymbol extends ZodType {
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.symbol) {
      const ctx = this._getOrReturnCtx(input);
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.symbol,
        received: ctx.parsedType
      });
      return INVALID;
    }
    return OK2(input.data);
  }
}
ZodSymbol.create = (params) => {
  return new ZodSymbol({
    typeName: ZodFirstPartyTypeKind.ZodSymbol,
    ...processCreateParams(params)
  });
};
class ZodUndefined extends ZodType {
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.undefined) {
      const ctx = this._getOrReturnCtx(input);
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.undefined,
        received: ctx.parsedType
      });
      return INVALID;
    }
    return OK2(input.data);
  }
}
ZodUndefined.create = (params) => {
  return new ZodUndefined({
    typeName: ZodFirstPartyTypeKind.ZodUndefined,
    ...processCreateParams(params)
  });
};
class ZodNull extends ZodType {
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.null) {
      const ctx = this._getOrReturnCtx(input);
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.null,
        received: ctx.parsedType
      });
      return INVALID;
    }
    return OK2(input.data);
  }
}
ZodNull.create = (params) => {
  return new ZodNull({
    typeName: ZodFirstPartyTypeKind.ZodNull,
    ...processCreateParams(params)
  });
};
class ZodAny extends ZodType {
  constructor() {
    super(...arguments);
    this._any = true;
  }
  _parse(input) {
    return OK2(input.data);
  }
}
ZodAny.create = (params) => {
  return new ZodAny({
    typeName: ZodFirstPartyTypeKind.ZodAny,
    ...processCreateParams(params)
  });
};
class ZodUnknown extends ZodType {
  constructor() {
    super(...arguments);
    this._unknown = true;
  }
  _parse(input) {
    return OK2(input.data);
  }
}
ZodUnknown.create = (params) => {
  return new ZodUnknown({
    typeName: ZodFirstPartyTypeKind.ZodUnknown,
    ...processCreateParams(params)
  });
};
class ZodNever extends ZodType {
  _parse(input) {
    const ctx = this._getOrReturnCtx(input);
    addIssueToContext(ctx, {
      code: ZodIssueCode.invalid_type,
      expected: ZodParsedType.never,
      received: ctx.parsedType
    });
    return INVALID;
  }
}
ZodNever.create = (params) => {
  return new ZodNever({
    typeName: ZodFirstPartyTypeKind.ZodNever,
    ...processCreateParams(params)
  });
};
class ZodVoid extends ZodType {
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.undefined) {
      const ctx = this._getOrReturnCtx(input);
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.void,
        received: ctx.parsedType
      });
      return INVALID;
    }
    return OK2(input.data);
  }
}
ZodVoid.create = (params) => {
  return new ZodVoid({
    typeName: ZodFirstPartyTypeKind.ZodVoid,
    ...processCreateParams(params)
  });
};
class ZodArray extends ZodType {
  _parse(input) {
    const { ctx, status } = this._processInputParams(input);
    const def = this._def;
    if (ctx.parsedType !== ZodParsedType.array) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.array,
        received: ctx.parsedType
      });
      return INVALID;
    }
    if (def.exactLength !== null) {
      const tooBig = ctx.data.length > def.exactLength.value;
      const tooSmall = ctx.data.length < def.exactLength.value;
      if (tooBig || tooSmall) {
        addIssueToContext(ctx, {
          code: tooBig ? ZodIssueCode.too_big : ZodIssueCode.too_small,
          minimum: tooSmall ? def.exactLength.value : void 0,
          maximum: tooBig ? def.exactLength.value : void 0,
          type: "array",
          inclusive: true,
          exact: true,
          message: def.exactLength.message
        });
        status.dirty();
      }
    }
    if (def.minLength !== null) {
      if (ctx.data.length < def.minLength.value) {
        addIssueToContext(ctx, {
          code: ZodIssueCode.too_small,
          minimum: def.minLength.value,
          type: "array",
          inclusive: true,
          exact: false,
          message: def.minLength.message
        });
        status.dirty();
      }
    }
    if (def.maxLength !== null) {
      if (ctx.data.length > def.maxLength.value) {
        addIssueToContext(ctx, {
          code: ZodIssueCode.too_big,
          maximum: def.maxLength.value,
          type: "array",
          inclusive: true,
          exact: false,
          message: def.maxLength.message
        });
        status.dirty();
      }
    }
    if (ctx.common.async) {
      return Promise.all([...ctx.data].map((item, i) => {
        return def.type._parseAsync(new ParseInputLazyPath(ctx, item, ctx.path, i));
      })).then((result2) => {
        return ParseStatus.mergeArray(status, result2);
      });
    }
    const result = [...ctx.data].map((item, i) => {
      return def.type._parseSync(new ParseInputLazyPath(ctx, item, ctx.path, i));
    });
    return ParseStatus.mergeArray(status, result);
  }
  get element() {
    return this._def.type;
  }
  min(minLength, message) {
    return new ZodArray({
      ...this._def,
      minLength: { value: minLength, message: errorUtil.toString(message) }
    });
  }
  max(maxLength, message) {
    return new ZodArray({
      ...this._def,
      maxLength: { value: maxLength, message: errorUtil.toString(message) }
    });
  }
  length(len, message) {
    return new ZodArray({
      ...this._def,
      exactLength: { value: len, message: errorUtil.toString(message) }
    });
  }
  nonempty(message) {
    return this.min(1, message);
  }
}
ZodArray.create = (schema, params) => {
  return new ZodArray({
    type: schema,
    minLength: null,
    maxLength: null,
    exactLength: null,
    typeName: ZodFirstPartyTypeKind.ZodArray,
    ...processCreateParams(params)
  });
};
function deepPartialify(schema) {
  if (schema instanceof ZodObject) {
    const newShape = {};
    for (const key in schema.shape) {
      const fieldSchema = schema.shape[key];
      newShape[key] = ZodOptional.create(deepPartialify(fieldSchema));
    }
    return new ZodObject({
      ...schema._def,
      shape: () => newShape
    });
  } else if (schema instanceof ZodArray) {
    return new ZodArray({
      ...schema._def,
      type: deepPartialify(schema.element)
    });
  } else if (schema instanceof ZodOptional) {
    return ZodOptional.create(deepPartialify(schema.unwrap()));
  } else if (schema instanceof ZodNullable) {
    return ZodNullable.create(deepPartialify(schema.unwrap()));
  } else if (schema instanceof ZodTuple) {
    return ZodTuple.create(schema.items.map((item) => deepPartialify(item)));
  } else {
    return schema;
  }
}
class ZodObject extends ZodType {
  constructor() {
    super(...arguments);
    this._cached = null;
    this.nonstrict = this.passthrough;
    this.augment = this.extend;
  }
  _getCached() {
    if (this._cached !== null)
      return this._cached;
    const shape = this._def.shape();
    const keys = util.objectKeys(shape);
    return this._cached = { shape, keys };
  }
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.object) {
      const ctx2 = this._getOrReturnCtx(input);
      addIssueToContext(ctx2, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.object,
        received: ctx2.parsedType
      });
      return INVALID;
    }
    const { status, ctx } = this._processInputParams(input);
    const { shape, keys: shapeKeys } = this._getCached();
    const extraKeys = [];
    if (!(this._def.catchall instanceof ZodNever && this._def.unknownKeys === "strip")) {
      for (const key in ctx.data) {
        if (!shapeKeys.includes(key)) {
          extraKeys.push(key);
        }
      }
    }
    const pairs = [];
    for (const key of shapeKeys) {
      const keyValidator = shape[key];
      const value = ctx.data[key];
      pairs.push({
        key: { status: "valid", value: key },
        value: keyValidator._parse(new ParseInputLazyPath(ctx, value, ctx.path, key)),
        alwaysSet: key in ctx.data
      });
    }
    if (this._def.catchall instanceof ZodNever) {
      const unknownKeys = this._def.unknownKeys;
      if (unknownKeys === "passthrough") {
        for (const key of extraKeys) {
          pairs.push({
            key: { status: "valid", value: key },
            value: { status: "valid", value: ctx.data[key] }
          });
        }
      } else if (unknownKeys === "strict") {
        if (extraKeys.length > 0) {
          addIssueToContext(ctx, {
            code: ZodIssueCode.unrecognized_keys,
            keys: extraKeys
          });
          status.dirty();
        }
      } else if (unknownKeys === "strip") ;
      else {
        throw new Error(`Internal ZodObject error: invalid unknownKeys value.`);
      }
    } else {
      const catchall = this._def.catchall;
      for (const key of extraKeys) {
        const value = ctx.data[key];
        pairs.push({
          key: { status: "valid", value: key },
          value: catchall._parse(
            new ParseInputLazyPath(ctx, value, ctx.path, key)
            //, ctx.child(key), value, getParsedType(value)
          ),
          alwaysSet: key in ctx.data
        });
      }
    }
    if (ctx.common.async) {
      return Promise.resolve().then(async () => {
        const syncPairs = [];
        for (const pair of pairs) {
          const key = await pair.key;
          const value = await pair.value;
          syncPairs.push({
            key,
            value,
            alwaysSet: pair.alwaysSet
          });
        }
        return syncPairs;
      }).then((syncPairs) => {
        return ParseStatus.mergeObjectSync(status, syncPairs);
      });
    } else {
      return ParseStatus.mergeObjectSync(status, pairs);
    }
  }
  get shape() {
    return this._def.shape();
  }
  strict(message) {
    errorUtil.errToObj;
    return new ZodObject({
      ...this._def,
      unknownKeys: "strict",
      ...message !== void 0 ? {
        errorMap: (issue, ctx) => {
          var _a, _b, _c, _d;
          const defaultError = (_c = (_b = (_a = this._def).errorMap) === null || _b === void 0 ? void 0 : _b.call(_a, issue, ctx).message) !== null && _c !== void 0 ? _c : ctx.defaultError;
          if (issue.code === "unrecognized_keys")
            return {
              message: (_d = errorUtil.errToObj(message).message) !== null && _d !== void 0 ? _d : defaultError
            };
          return {
            message: defaultError
          };
        }
      } : {}
    });
  }
  strip() {
    return new ZodObject({
      ...this._def,
      unknownKeys: "strip"
    });
  }
  passthrough() {
    return new ZodObject({
      ...this._def,
      unknownKeys: "passthrough"
    });
  }
  // const AugmentFactory =
  //   <Def extends ZodObjectDef>(def: Def) =>
  //   <Augmentation extends ZodRawShape>(
  //     augmentation: Augmentation
  //   ): ZodObject<
  //     extendShape<ReturnType<Def["shape"]>, Augmentation>,
  //     Def["unknownKeys"],
  //     Def["catchall"]
  //   > => {
  //     return new ZodObject({
  //       ...def,
  //       shape: () => ({
  //         ...def.shape(),
  //         ...augmentation,
  //       }),
  //     }) as any;
  //   };
  extend(augmentation) {
    return new ZodObject({
      ...this._def,
      shape: () => ({
        ...this._def.shape(),
        ...augmentation
      })
    });
  }
  /**
   * Prior to zod@1.0.12 there was a bug in the
   * inferred type of merged objects. Please
   * upgrade if you are experiencing issues.
   */
  merge(merging) {
    const merged = new ZodObject({
      unknownKeys: merging._def.unknownKeys,
      catchall: merging._def.catchall,
      shape: () => ({
        ...this._def.shape(),
        ...merging._def.shape()
      }),
      typeName: ZodFirstPartyTypeKind.ZodObject
    });
    return merged;
  }
  // merge<
  //   Incoming extends AnyZodObject,
  //   Augmentation extends Incoming["shape"],
  //   NewOutput extends {
  //     [k in keyof Augmentation | keyof Output]: k extends keyof Augmentation
  //       ? Augmentation[k]["_output"]
  //       : k extends keyof Output
  //       ? Output[k]
  //       : never;
  //   },
  //   NewInput extends {
  //     [k in keyof Augmentation | keyof Input]: k extends keyof Augmentation
  //       ? Augmentation[k]["_input"]
  //       : k extends keyof Input
  //       ? Input[k]
  //       : never;
  //   }
  // >(
  //   merging: Incoming
  // ): ZodObject<
  //   extendShape<T, ReturnType<Incoming["_def"]["shape"]>>,
  //   Incoming["_def"]["unknownKeys"],
  //   Incoming["_def"]["catchall"],
  //   NewOutput,
  //   NewInput
  // > {
  //   const merged: any = new ZodObject({
  //     unknownKeys: merging._def.unknownKeys,
  //     catchall: merging._def.catchall,
  //     shape: () =>
  //       objectUtil.mergeShapes(this._def.shape(), merging._def.shape()),
  //     typeName: ZodFirstPartyTypeKind.ZodObject,
  //   }) as any;
  //   return merged;
  // }
  setKey(key, schema) {
    return this.augment({ [key]: schema });
  }
  // merge<Incoming extends AnyZodObject>(
  //   merging: Incoming
  // ): //ZodObject<T & Incoming["_shape"], UnknownKeys, Catchall> = (merging) => {
  // ZodObject<
  //   extendShape<T, ReturnType<Incoming["_def"]["shape"]>>,
  //   Incoming["_def"]["unknownKeys"],
  //   Incoming["_def"]["catchall"]
  // > {
  //   // const mergedShape = objectUtil.mergeShapes(
  //   //   this._def.shape(),
  //   //   merging._def.shape()
  //   // );
  //   const merged: any = new ZodObject({
  //     unknownKeys: merging._def.unknownKeys,
  //     catchall: merging._def.catchall,
  //     shape: () =>
  //       objectUtil.mergeShapes(this._def.shape(), merging._def.shape()),
  //     typeName: ZodFirstPartyTypeKind.ZodObject,
  //   }) as any;
  //   return merged;
  // }
  catchall(index) {
    return new ZodObject({
      ...this._def,
      catchall: index
    });
  }
  pick(mask) {
    const shape = {};
    util.objectKeys(mask).forEach((key) => {
      if (mask[key] && this.shape[key]) {
        shape[key] = this.shape[key];
      }
    });
    return new ZodObject({
      ...this._def,
      shape: () => shape
    });
  }
  omit(mask) {
    const shape = {};
    util.objectKeys(this.shape).forEach((key) => {
      if (!mask[key]) {
        shape[key] = this.shape[key];
      }
    });
    return new ZodObject({
      ...this._def,
      shape: () => shape
    });
  }
  /**
   * @deprecated
   */
  deepPartial() {
    return deepPartialify(this);
  }
  partial(mask) {
    const newShape = {};
    util.objectKeys(this.shape).forEach((key) => {
      const fieldSchema = this.shape[key];
      if (mask && !mask[key]) {
        newShape[key] = fieldSchema;
      } else {
        newShape[key] = fieldSchema.optional();
      }
    });
    return new ZodObject({
      ...this._def,
      shape: () => newShape
    });
  }
  required(mask) {
    const newShape = {};
    util.objectKeys(this.shape).forEach((key) => {
      if (mask && !mask[key]) {
        newShape[key] = this.shape[key];
      } else {
        const fieldSchema = this.shape[key];
        let newField = fieldSchema;
        while (newField instanceof ZodOptional) {
          newField = newField._def.innerType;
        }
        newShape[key] = newField;
      }
    });
    return new ZodObject({
      ...this._def,
      shape: () => newShape
    });
  }
  keyof() {
    return createZodEnum(util.objectKeys(this.shape));
  }
}
ZodObject.create = (shape, params) => {
  return new ZodObject({
    shape: () => shape,
    unknownKeys: "strip",
    catchall: ZodNever.create(),
    typeName: ZodFirstPartyTypeKind.ZodObject,
    ...processCreateParams(params)
  });
};
ZodObject.strictCreate = (shape, params) => {
  return new ZodObject({
    shape: () => shape,
    unknownKeys: "strict",
    catchall: ZodNever.create(),
    typeName: ZodFirstPartyTypeKind.ZodObject,
    ...processCreateParams(params)
  });
};
ZodObject.lazycreate = (shape, params) => {
  return new ZodObject({
    shape,
    unknownKeys: "strip",
    catchall: ZodNever.create(),
    typeName: ZodFirstPartyTypeKind.ZodObject,
    ...processCreateParams(params)
  });
};
class ZodUnion extends ZodType {
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    const options = this._def.options;
    function handleResults(results) {
      for (const result of results) {
        if (result.result.status === "valid") {
          return result.result;
        }
      }
      for (const result of results) {
        if (result.result.status === "dirty") {
          ctx.common.issues.push(...result.ctx.common.issues);
          return result.result;
        }
      }
      const unionErrors = results.map((result) => new ZodError(result.ctx.common.issues));
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_union,
        unionErrors
      });
      return INVALID;
    }
    if (ctx.common.async) {
      return Promise.all(options.map(async (option) => {
        const childCtx = {
          ...ctx,
          common: {
            ...ctx.common,
            issues: []
          },
          parent: null
        };
        return {
          result: await option._parseAsync({
            data: ctx.data,
            path: ctx.path,
            parent: childCtx
          }),
          ctx: childCtx
        };
      })).then(handleResults);
    } else {
      let dirty = void 0;
      const issues = [];
      for (const option of options) {
        const childCtx = {
          ...ctx,
          common: {
            ...ctx.common,
            issues: []
          },
          parent: null
        };
        const result = option._parseSync({
          data: ctx.data,
          path: ctx.path,
          parent: childCtx
        });
        if (result.status === "valid") {
          return result;
        } else if (result.status === "dirty" && !dirty) {
          dirty = { result, ctx: childCtx };
        }
        if (childCtx.common.issues.length) {
          issues.push(childCtx.common.issues);
        }
      }
      if (dirty) {
        ctx.common.issues.push(...dirty.ctx.common.issues);
        return dirty.result;
      }
      const unionErrors = issues.map((issues2) => new ZodError(issues2));
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_union,
        unionErrors
      });
      return INVALID;
    }
  }
  get options() {
    return this._def.options;
  }
}
ZodUnion.create = (types2, params) => {
  return new ZodUnion({
    options: types2,
    typeName: ZodFirstPartyTypeKind.ZodUnion,
    ...processCreateParams(params)
  });
};
const getDiscriminator = (type) => {
  if (type instanceof ZodLazy) {
    return getDiscriminator(type.schema);
  } else if (type instanceof ZodEffects) {
    return getDiscriminator(type.innerType());
  } else if (type instanceof ZodLiteral) {
    return [type.value];
  } else if (type instanceof ZodEnum) {
    return type.options;
  } else if (type instanceof ZodNativeEnum) {
    return util.objectValues(type.enum);
  } else if (type instanceof ZodDefault) {
    return getDiscriminator(type._def.innerType);
  } else if (type instanceof ZodUndefined) {
    return [void 0];
  } else if (type instanceof ZodNull) {
    return [null];
  } else if (type instanceof ZodOptional) {
    return [void 0, ...getDiscriminator(type.unwrap())];
  } else if (type instanceof ZodNullable) {
    return [null, ...getDiscriminator(type.unwrap())];
  } else if (type instanceof ZodBranded) {
    return getDiscriminator(type.unwrap());
  } else if (type instanceof ZodReadonly) {
    return getDiscriminator(type.unwrap());
  } else if (type instanceof ZodCatch) {
    return getDiscriminator(type._def.innerType);
  } else {
    return [];
  }
};
class ZodDiscriminatedUnion extends ZodType {
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    if (ctx.parsedType !== ZodParsedType.object) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.object,
        received: ctx.parsedType
      });
      return INVALID;
    }
    const discriminator = this.discriminator;
    const discriminatorValue = ctx.data[discriminator];
    const option = this.optionsMap.get(discriminatorValue);
    if (!option) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_union_discriminator,
        options: Array.from(this.optionsMap.keys()),
        path: [discriminator]
      });
      return INVALID;
    }
    if (ctx.common.async) {
      return option._parseAsync({
        data: ctx.data,
        path: ctx.path,
        parent: ctx
      });
    } else {
      return option._parseSync({
        data: ctx.data,
        path: ctx.path,
        parent: ctx
      });
    }
  }
  get discriminator() {
    return this._def.discriminator;
  }
  get options() {
    return this._def.options;
  }
  get optionsMap() {
    return this._def.optionsMap;
  }
  /**
   * The constructor of the discriminated union schema. Its behaviour is very similar to that of the normal z.union() constructor.
   * However, it only allows a union of objects, all of which need to share a discriminator property. This property must
   * have a different value for each object in the union.
   * @param discriminator the name of the discriminator property
   * @param types an array of object schemas
   * @param params
   */
  static create(discriminator, options, params) {
    const optionsMap = /* @__PURE__ */ new Map();
    for (const type of options) {
      const discriminatorValues = getDiscriminator(type.shape[discriminator]);
      if (!discriminatorValues.length) {
        throw new Error(`A discriminator value for key \`${discriminator}\` could not be extracted from all schema options`);
      }
      for (const value of discriminatorValues) {
        if (optionsMap.has(value)) {
          throw new Error(`Discriminator property ${String(discriminator)} has duplicate value ${String(value)}`);
        }
        optionsMap.set(value, type);
      }
    }
    return new ZodDiscriminatedUnion({
      typeName: ZodFirstPartyTypeKind.ZodDiscriminatedUnion,
      discriminator,
      options,
      optionsMap,
      ...processCreateParams(params)
    });
  }
}
function mergeValues(a, b) {
  const aType = getParsedType(a);
  const bType = getParsedType(b);
  if (a === b) {
    return { valid: true, data: a };
  } else if (aType === ZodParsedType.object && bType === ZodParsedType.object) {
    const bKeys = util.objectKeys(b);
    const sharedKeys = util.objectKeys(a).filter((key) => bKeys.indexOf(key) !== -1);
    const newObj = { ...a, ...b };
    for (const key of sharedKeys) {
      const sharedValue = mergeValues(a[key], b[key]);
      if (!sharedValue.valid) {
        return { valid: false };
      }
      newObj[key] = sharedValue.data;
    }
    return { valid: true, data: newObj };
  } else if (aType === ZodParsedType.array && bType === ZodParsedType.array) {
    if (a.length !== b.length) {
      return { valid: false };
    }
    const newArray = [];
    for (let index = 0; index < a.length; index++) {
      const itemA = a[index];
      const itemB = b[index];
      const sharedValue = mergeValues(itemA, itemB);
      if (!sharedValue.valid) {
        return { valid: false };
      }
      newArray.push(sharedValue.data);
    }
    return { valid: true, data: newArray };
  } else if (aType === ZodParsedType.date && bType === ZodParsedType.date && +a === +b) {
    return { valid: true, data: a };
  } else {
    return { valid: false };
  }
}
class ZodIntersection extends ZodType {
  _parse(input) {
    const { status, ctx } = this._processInputParams(input);
    const handleParsed = (parsedLeft, parsedRight) => {
      if (isAborted(parsedLeft) || isAborted(parsedRight)) {
        return INVALID;
      }
      const merged = mergeValues(parsedLeft.value, parsedRight.value);
      if (!merged.valid) {
        addIssueToContext(ctx, {
          code: ZodIssueCode.invalid_intersection_types
        });
        return INVALID;
      }
      if (isDirty(parsedLeft) || isDirty(parsedRight)) {
        status.dirty();
      }
      return { status: status.value, value: merged.data };
    };
    if (ctx.common.async) {
      return Promise.all([
        this._def.left._parseAsync({
          data: ctx.data,
          path: ctx.path,
          parent: ctx
        }),
        this._def.right._parseAsync({
          data: ctx.data,
          path: ctx.path,
          parent: ctx
        })
      ]).then(([left, right]) => handleParsed(left, right));
    } else {
      return handleParsed(this._def.left._parseSync({
        data: ctx.data,
        path: ctx.path,
        parent: ctx
      }), this._def.right._parseSync({
        data: ctx.data,
        path: ctx.path,
        parent: ctx
      }));
    }
  }
}
ZodIntersection.create = (left, right, params) => {
  return new ZodIntersection({
    left,
    right,
    typeName: ZodFirstPartyTypeKind.ZodIntersection,
    ...processCreateParams(params)
  });
};
class ZodTuple extends ZodType {
  _parse(input) {
    const { status, ctx } = this._processInputParams(input);
    if (ctx.parsedType !== ZodParsedType.array) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.array,
        received: ctx.parsedType
      });
      return INVALID;
    }
    if (ctx.data.length < this._def.items.length) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.too_small,
        minimum: this._def.items.length,
        inclusive: true,
        exact: false,
        type: "array"
      });
      return INVALID;
    }
    const rest = this._def.rest;
    if (!rest && ctx.data.length > this._def.items.length) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.too_big,
        maximum: this._def.items.length,
        inclusive: true,
        exact: false,
        type: "array"
      });
      status.dirty();
    }
    const items = [...ctx.data].map((item, itemIndex) => {
      const schema = this._def.items[itemIndex] || this._def.rest;
      if (!schema)
        return null;
      return schema._parse(new ParseInputLazyPath(ctx, item, ctx.path, itemIndex));
    }).filter((x) => !!x);
    if (ctx.common.async) {
      return Promise.all(items).then((results) => {
        return ParseStatus.mergeArray(status, results);
      });
    } else {
      return ParseStatus.mergeArray(status, items);
    }
  }
  get items() {
    return this._def.items;
  }
  rest(rest) {
    return new ZodTuple({
      ...this._def,
      rest
    });
  }
}
ZodTuple.create = (schemas, params) => {
  if (!Array.isArray(schemas)) {
    throw new Error("You must pass an array of schemas to z.tuple([ ... ])");
  }
  return new ZodTuple({
    items: schemas,
    typeName: ZodFirstPartyTypeKind.ZodTuple,
    rest: null,
    ...processCreateParams(params)
  });
};
class ZodRecord extends ZodType {
  get keySchema() {
    return this._def.keyType;
  }
  get valueSchema() {
    return this._def.valueType;
  }
  _parse(input) {
    const { status, ctx } = this._processInputParams(input);
    if (ctx.parsedType !== ZodParsedType.object) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.object,
        received: ctx.parsedType
      });
      return INVALID;
    }
    const pairs = [];
    const keyType = this._def.keyType;
    const valueType = this._def.valueType;
    for (const key in ctx.data) {
      pairs.push({
        key: keyType._parse(new ParseInputLazyPath(ctx, key, ctx.path, key)),
        value: valueType._parse(new ParseInputLazyPath(ctx, ctx.data[key], ctx.path, key)),
        alwaysSet: key in ctx.data
      });
    }
    if (ctx.common.async) {
      return ParseStatus.mergeObjectAsync(status, pairs);
    } else {
      return ParseStatus.mergeObjectSync(status, pairs);
    }
  }
  get element() {
    return this._def.valueType;
  }
  static create(first, second, third) {
    if (second instanceof ZodType) {
      return new ZodRecord({
        keyType: first,
        valueType: second,
        typeName: ZodFirstPartyTypeKind.ZodRecord,
        ...processCreateParams(third)
      });
    }
    return new ZodRecord({
      keyType: ZodString.create(),
      valueType: first,
      typeName: ZodFirstPartyTypeKind.ZodRecord,
      ...processCreateParams(second)
    });
  }
}
class ZodMap extends ZodType {
  get keySchema() {
    return this._def.keyType;
  }
  get valueSchema() {
    return this._def.valueType;
  }
  _parse(input) {
    const { status, ctx } = this._processInputParams(input);
    if (ctx.parsedType !== ZodParsedType.map) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.map,
        received: ctx.parsedType
      });
      return INVALID;
    }
    const keyType = this._def.keyType;
    const valueType = this._def.valueType;
    const pairs = [...ctx.data.entries()].map(([key, value], index) => {
      return {
        key: keyType._parse(new ParseInputLazyPath(ctx, key, ctx.path, [index, "key"])),
        value: valueType._parse(new ParseInputLazyPath(ctx, value, ctx.path, [index, "value"]))
      };
    });
    if (ctx.common.async) {
      const finalMap = /* @__PURE__ */ new Map();
      return Promise.resolve().then(async () => {
        for (const pair of pairs) {
          const key = await pair.key;
          const value = await pair.value;
          if (key.status === "aborted" || value.status === "aborted") {
            return INVALID;
          }
          if (key.status === "dirty" || value.status === "dirty") {
            status.dirty();
          }
          finalMap.set(key.value, value.value);
        }
        return { status: status.value, value: finalMap };
      });
    } else {
      const finalMap = /* @__PURE__ */ new Map();
      for (const pair of pairs) {
        const key = pair.key;
        const value = pair.value;
        if (key.status === "aborted" || value.status === "aborted") {
          return INVALID;
        }
        if (key.status === "dirty" || value.status === "dirty") {
          status.dirty();
        }
        finalMap.set(key.value, value.value);
      }
      return { status: status.value, value: finalMap };
    }
  }
}
ZodMap.create = (keyType, valueType, params) => {
  return new ZodMap({
    valueType,
    keyType,
    typeName: ZodFirstPartyTypeKind.ZodMap,
    ...processCreateParams(params)
  });
};
class ZodSet extends ZodType {
  _parse(input) {
    const { status, ctx } = this._processInputParams(input);
    if (ctx.parsedType !== ZodParsedType.set) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.set,
        received: ctx.parsedType
      });
      return INVALID;
    }
    const def = this._def;
    if (def.minSize !== null) {
      if (ctx.data.size < def.minSize.value) {
        addIssueToContext(ctx, {
          code: ZodIssueCode.too_small,
          minimum: def.minSize.value,
          type: "set",
          inclusive: true,
          exact: false,
          message: def.minSize.message
        });
        status.dirty();
      }
    }
    if (def.maxSize !== null) {
      if (ctx.data.size > def.maxSize.value) {
        addIssueToContext(ctx, {
          code: ZodIssueCode.too_big,
          maximum: def.maxSize.value,
          type: "set",
          inclusive: true,
          exact: false,
          message: def.maxSize.message
        });
        status.dirty();
      }
    }
    const valueType = this._def.valueType;
    function finalizeSet(elements2) {
      const parsedSet = /* @__PURE__ */ new Set();
      for (const element of elements2) {
        if (element.status === "aborted")
          return INVALID;
        if (element.status === "dirty")
          status.dirty();
        parsedSet.add(element.value);
      }
      return { status: status.value, value: parsedSet };
    }
    const elements = [...ctx.data.values()].map((item, i) => valueType._parse(new ParseInputLazyPath(ctx, item, ctx.path, i)));
    if (ctx.common.async) {
      return Promise.all(elements).then((elements2) => finalizeSet(elements2));
    } else {
      return finalizeSet(elements);
    }
  }
  min(minSize, message) {
    return new ZodSet({
      ...this._def,
      minSize: { value: minSize, message: errorUtil.toString(message) }
    });
  }
  max(maxSize, message) {
    return new ZodSet({
      ...this._def,
      maxSize: { value: maxSize, message: errorUtil.toString(message) }
    });
  }
  size(size, message) {
    return this.min(size, message).max(size, message);
  }
  nonempty(message) {
    return this.min(1, message);
  }
}
ZodSet.create = (valueType, params) => {
  return new ZodSet({
    valueType,
    minSize: null,
    maxSize: null,
    typeName: ZodFirstPartyTypeKind.ZodSet,
    ...processCreateParams(params)
  });
};
class ZodFunction extends ZodType {
  constructor() {
    super(...arguments);
    this.validate = this.implement;
  }
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    if (ctx.parsedType !== ZodParsedType.function) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.function,
        received: ctx.parsedType
      });
      return INVALID;
    }
    function makeArgsIssue(args, error) {
      return makeIssue({
        data: args,
        path: ctx.path,
        errorMaps: [
          ctx.common.contextualErrorMap,
          ctx.schemaErrorMap,
          getErrorMap(),
          errorMap
        ].filter((x) => !!x),
        issueData: {
          code: ZodIssueCode.invalid_arguments,
          argumentsError: error
        }
      });
    }
    function makeReturnsIssue(returns, error) {
      return makeIssue({
        data: returns,
        path: ctx.path,
        errorMaps: [
          ctx.common.contextualErrorMap,
          ctx.schemaErrorMap,
          getErrorMap(),
          errorMap
        ].filter((x) => !!x),
        issueData: {
          code: ZodIssueCode.invalid_return_type,
          returnTypeError: error
        }
      });
    }
    const params = { errorMap: ctx.common.contextualErrorMap };
    const fn = ctx.data;
    if (this._def.returns instanceof ZodPromise) {
      const me = this;
      return OK2(async function(...args) {
        const error = new ZodError([]);
        const parsedArgs = await me._def.args.parseAsync(args, params).catch((e) => {
          error.addIssue(makeArgsIssue(args, e));
          throw error;
        });
        const result = await Reflect.apply(fn, this, parsedArgs);
        const parsedReturns = await me._def.returns._def.type.parseAsync(result, params).catch((e) => {
          error.addIssue(makeReturnsIssue(result, e));
          throw error;
        });
        return parsedReturns;
      });
    } else {
      const me = this;
      return OK2(function(...args) {
        const parsedArgs = me._def.args.safeParse(args, params);
        if (!parsedArgs.success) {
          throw new ZodError([makeArgsIssue(args, parsedArgs.error)]);
        }
        const result = Reflect.apply(fn, this, parsedArgs.data);
        const parsedReturns = me._def.returns.safeParse(result, params);
        if (!parsedReturns.success) {
          throw new ZodError([makeReturnsIssue(result, parsedReturns.error)]);
        }
        return parsedReturns.data;
      });
    }
  }
  parameters() {
    return this._def.args;
  }
  returnType() {
    return this._def.returns;
  }
  args(...items) {
    return new ZodFunction({
      ...this._def,
      args: ZodTuple.create(items).rest(ZodUnknown.create())
    });
  }
  returns(returnType) {
    return new ZodFunction({
      ...this._def,
      returns: returnType
    });
  }
  implement(func) {
    const validatedFunc = this.parse(func);
    return validatedFunc;
  }
  strictImplement(func) {
    const validatedFunc = this.parse(func);
    return validatedFunc;
  }
  static create(args, returns, params) {
    return new ZodFunction({
      args: args ? args : ZodTuple.create([]).rest(ZodUnknown.create()),
      returns: returns || ZodUnknown.create(),
      typeName: ZodFirstPartyTypeKind.ZodFunction,
      ...processCreateParams(params)
    });
  }
}
class ZodLazy extends ZodType {
  get schema() {
    return this._def.getter();
  }
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    const lazySchema = this._def.getter();
    return lazySchema._parse({ data: ctx.data, path: ctx.path, parent: ctx });
  }
}
ZodLazy.create = (getter, params) => {
  return new ZodLazy({
    getter,
    typeName: ZodFirstPartyTypeKind.ZodLazy,
    ...processCreateParams(params)
  });
};
class ZodLiteral extends ZodType {
  _parse(input) {
    if (input.data !== this._def.value) {
      const ctx = this._getOrReturnCtx(input);
      addIssueToContext(ctx, {
        received: ctx.data,
        code: ZodIssueCode.invalid_literal,
        expected: this._def.value
      });
      return INVALID;
    }
    return { status: "valid", value: input.data };
  }
  get value() {
    return this._def.value;
  }
}
ZodLiteral.create = (value, params) => {
  return new ZodLiteral({
    value,
    typeName: ZodFirstPartyTypeKind.ZodLiteral,
    ...processCreateParams(params)
  });
};
function createZodEnum(values, params) {
  return new ZodEnum({
    values,
    typeName: ZodFirstPartyTypeKind.ZodEnum,
    ...processCreateParams(params)
  });
}
class ZodEnum extends ZodType {
  constructor() {
    super(...arguments);
    _ZodEnum_cache.set(this, void 0);
  }
  _parse(input) {
    if (typeof input.data !== "string") {
      const ctx = this._getOrReturnCtx(input);
      const expectedValues = this._def.values;
      addIssueToContext(ctx, {
        expected: util.joinValues(expectedValues),
        received: ctx.parsedType,
        code: ZodIssueCode.invalid_type
      });
      return INVALID;
    }
    if (!__classPrivateFieldGet(this, _ZodEnum_cache)) {
      __classPrivateFieldSet(this, _ZodEnum_cache, new Set(this._def.values));
    }
    if (!__classPrivateFieldGet(this, _ZodEnum_cache).has(input.data)) {
      const ctx = this._getOrReturnCtx(input);
      const expectedValues = this._def.values;
      addIssueToContext(ctx, {
        received: ctx.data,
        code: ZodIssueCode.invalid_enum_value,
        options: expectedValues
      });
      return INVALID;
    }
    return OK2(input.data);
  }
  get options() {
    return this._def.values;
  }
  get enum() {
    const enumValues = {};
    for (const val of this._def.values) {
      enumValues[val] = val;
    }
    return enumValues;
  }
  get Values() {
    const enumValues = {};
    for (const val of this._def.values) {
      enumValues[val] = val;
    }
    return enumValues;
  }
  get Enum() {
    const enumValues = {};
    for (const val of this._def.values) {
      enumValues[val] = val;
    }
    return enumValues;
  }
  extract(values, newDef = this._def) {
    return ZodEnum.create(values, {
      ...this._def,
      ...newDef
    });
  }
  exclude(values, newDef = this._def) {
    return ZodEnum.create(this.options.filter((opt) => !values.includes(opt)), {
      ...this._def,
      ...newDef
    });
  }
}
_ZodEnum_cache = /* @__PURE__ */ new WeakMap();
ZodEnum.create = createZodEnum;
class ZodNativeEnum extends ZodType {
  constructor() {
    super(...arguments);
    _ZodNativeEnum_cache.set(this, void 0);
  }
  _parse(input) {
    const nativeEnumValues = util.getValidEnumValues(this._def.values);
    const ctx = this._getOrReturnCtx(input);
    if (ctx.parsedType !== ZodParsedType.string && ctx.parsedType !== ZodParsedType.number) {
      const expectedValues = util.objectValues(nativeEnumValues);
      addIssueToContext(ctx, {
        expected: util.joinValues(expectedValues),
        received: ctx.parsedType,
        code: ZodIssueCode.invalid_type
      });
      return INVALID;
    }
    if (!__classPrivateFieldGet(this, _ZodNativeEnum_cache)) {
      __classPrivateFieldSet(this, _ZodNativeEnum_cache, new Set(util.getValidEnumValues(this._def.values)));
    }
    if (!__classPrivateFieldGet(this, _ZodNativeEnum_cache).has(input.data)) {
      const expectedValues = util.objectValues(nativeEnumValues);
      addIssueToContext(ctx, {
        received: ctx.data,
        code: ZodIssueCode.invalid_enum_value,
        options: expectedValues
      });
      return INVALID;
    }
    return OK2(input.data);
  }
  get enum() {
    return this._def.values;
  }
}
_ZodNativeEnum_cache = /* @__PURE__ */ new WeakMap();
ZodNativeEnum.create = (values, params) => {
  return new ZodNativeEnum({
    values,
    typeName: ZodFirstPartyTypeKind.ZodNativeEnum,
    ...processCreateParams(params)
  });
};
class ZodPromise extends ZodType {
  unwrap() {
    return this._def.type;
  }
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    if (ctx.parsedType !== ZodParsedType.promise && ctx.common.async === false) {
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.promise,
        received: ctx.parsedType
      });
      return INVALID;
    }
    const promisified = ctx.parsedType === ZodParsedType.promise ? ctx.data : Promise.resolve(ctx.data);
    return OK2(promisified.then((data) => {
      return this._def.type.parseAsync(data, {
        path: ctx.path,
        errorMap: ctx.common.contextualErrorMap
      });
    }));
  }
}
ZodPromise.create = (schema, params) => {
  return new ZodPromise({
    type: schema,
    typeName: ZodFirstPartyTypeKind.ZodPromise,
    ...processCreateParams(params)
  });
};
class ZodEffects extends ZodType {
  innerType() {
    return this._def.schema;
  }
  sourceType() {
    return this._def.schema._def.typeName === ZodFirstPartyTypeKind.ZodEffects ? this._def.schema.sourceType() : this._def.schema;
  }
  _parse(input) {
    const { status, ctx } = this._processInputParams(input);
    const effect = this._def.effect || null;
    const checkCtx = {
      addIssue: (arg) => {
        addIssueToContext(ctx, arg);
        if (arg.fatal) {
          status.abort();
        } else {
          status.dirty();
        }
      },
      get path() {
        return ctx.path;
      }
    };
    checkCtx.addIssue = checkCtx.addIssue.bind(checkCtx);
    if (effect.type === "preprocess") {
      const processed = effect.transform(ctx.data, checkCtx);
      if (ctx.common.async) {
        return Promise.resolve(processed).then(async (processed2) => {
          if (status.value === "aborted")
            return INVALID;
          const result = await this._def.schema._parseAsync({
            data: processed2,
            path: ctx.path,
            parent: ctx
          });
          if (result.status === "aborted")
            return INVALID;
          if (result.status === "dirty")
            return DIRTY(result.value);
          if (status.value === "dirty")
            return DIRTY(result.value);
          return result;
        });
      } else {
        if (status.value === "aborted")
          return INVALID;
        const result = this._def.schema._parseSync({
          data: processed,
          path: ctx.path,
          parent: ctx
        });
        if (result.status === "aborted")
          return INVALID;
        if (result.status === "dirty")
          return DIRTY(result.value);
        if (status.value === "dirty")
          return DIRTY(result.value);
        return result;
      }
    }
    if (effect.type === "refinement") {
      const executeRefinement = (acc) => {
        const result = effect.refinement(acc, checkCtx);
        if (ctx.common.async) {
          return Promise.resolve(result);
        }
        if (result instanceof Promise) {
          throw new Error("Async refinement encountered during synchronous parse operation. Use .parseAsync instead.");
        }
        return acc;
      };
      if (ctx.common.async === false) {
        const inner = this._def.schema._parseSync({
          data: ctx.data,
          path: ctx.path,
          parent: ctx
        });
        if (inner.status === "aborted")
          return INVALID;
        if (inner.status === "dirty")
          status.dirty();
        executeRefinement(inner.value);
        return { status: status.value, value: inner.value };
      } else {
        return this._def.schema._parseAsync({ data: ctx.data, path: ctx.path, parent: ctx }).then((inner) => {
          if (inner.status === "aborted")
            return INVALID;
          if (inner.status === "dirty")
            status.dirty();
          return executeRefinement(inner.value).then(() => {
            return { status: status.value, value: inner.value };
          });
        });
      }
    }
    if (effect.type === "transform") {
      if (ctx.common.async === false) {
        const base = this._def.schema._parseSync({
          data: ctx.data,
          path: ctx.path,
          parent: ctx
        });
        if (!isValid(base))
          return base;
        const result = effect.transform(base.value, checkCtx);
        if (result instanceof Promise) {
          throw new Error(`Asynchronous transform encountered during synchronous parse operation. Use .parseAsync instead.`);
        }
        return { status: status.value, value: result };
      } else {
        return this._def.schema._parseAsync({ data: ctx.data, path: ctx.path, parent: ctx }).then((base) => {
          if (!isValid(base))
            return base;
          return Promise.resolve(effect.transform(base.value, checkCtx)).then((result) => ({ status: status.value, value: result }));
        });
      }
    }
    util.assertNever(effect);
  }
}
ZodEffects.create = (schema, effect, params) => {
  return new ZodEffects({
    schema,
    typeName: ZodFirstPartyTypeKind.ZodEffects,
    effect,
    ...processCreateParams(params)
  });
};
ZodEffects.createWithPreprocess = (preprocess, schema, params) => {
  return new ZodEffects({
    schema,
    effect: { type: "preprocess", transform: preprocess },
    typeName: ZodFirstPartyTypeKind.ZodEffects,
    ...processCreateParams(params)
  });
};
class ZodOptional extends ZodType {
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType === ZodParsedType.undefined) {
      return OK2(void 0);
    }
    return this._def.innerType._parse(input);
  }
  unwrap() {
    return this._def.innerType;
  }
}
ZodOptional.create = (type, params) => {
  return new ZodOptional({
    innerType: type,
    typeName: ZodFirstPartyTypeKind.ZodOptional,
    ...processCreateParams(params)
  });
};
class ZodNullable extends ZodType {
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType === ZodParsedType.null) {
      return OK2(null);
    }
    return this._def.innerType._parse(input);
  }
  unwrap() {
    return this._def.innerType;
  }
}
ZodNullable.create = (type, params) => {
  return new ZodNullable({
    innerType: type,
    typeName: ZodFirstPartyTypeKind.ZodNullable,
    ...processCreateParams(params)
  });
};
class ZodDefault extends ZodType {
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    let data = ctx.data;
    if (ctx.parsedType === ZodParsedType.undefined) {
      data = this._def.defaultValue();
    }
    return this._def.innerType._parse({
      data,
      path: ctx.path,
      parent: ctx
    });
  }
  removeDefault() {
    return this._def.innerType;
  }
}
ZodDefault.create = (type, params) => {
  return new ZodDefault({
    innerType: type,
    typeName: ZodFirstPartyTypeKind.ZodDefault,
    defaultValue: typeof params.default === "function" ? params.default : () => params.default,
    ...processCreateParams(params)
  });
};
class ZodCatch extends ZodType {
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    const newCtx = {
      ...ctx,
      common: {
        ...ctx.common,
        issues: []
      }
    };
    const result = this._def.innerType._parse({
      data: newCtx.data,
      path: newCtx.path,
      parent: {
        ...newCtx
      }
    });
    if (isAsync(result)) {
      return result.then((result2) => {
        return {
          status: "valid",
          value: result2.status === "valid" ? result2.value : this._def.catchValue({
            get error() {
              return new ZodError(newCtx.common.issues);
            },
            input: newCtx.data
          })
        };
      });
    } else {
      return {
        status: "valid",
        value: result.status === "valid" ? result.value : this._def.catchValue({
          get error() {
            return new ZodError(newCtx.common.issues);
          },
          input: newCtx.data
        })
      };
    }
  }
  removeCatch() {
    return this._def.innerType;
  }
}
ZodCatch.create = (type, params) => {
  return new ZodCatch({
    innerType: type,
    typeName: ZodFirstPartyTypeKind.ZodCatch,
    catchValue: typeof params.catch === "function" ? params.catch : () => params.catch,
    ...processCreateParams(params)
  });
};
class ZodNaN extends ZodType {
  _parse(input) {
    const parsedType = this._getType(input);
    if (parsedType !== ZodParsedType.nan) {
      const ctx = this._getOrReturnCtx(input);
      addIssueToContext(ctx, {
        code: ZodIssueCode.invalid_type,
        expected: ZodParsedType.nan,
        received: ctx.parsedType
      });
      return INVALID;
    }
    return { status: "valid", value: input.data };
  }
}
ZodNaN.create = (params) => {
  return new ZodNaN({
    typeName: ZodFirstPartyTypeKind.ZodNaN,
    ...processCreateParams(params)
  });
};
const BRAND = Symbol("zod_brand");
class ZodBranded extends ZodType {
  _parse(input) {
    const { ctx } = this._processInputParams(input);
    const data = ctx.data;
    return this._def.type._parse({
      data,
      path: ctx.path,
      parent: ctx
    });
  }
  unwrap() {
    return this._def.type;
  }
}
class ZodPipeline extends ZodType {
  _parse(input) {
    const { status, ctx } = this._processInputParams(input);
    if (ctx.common.async) {
      const handleAsync = async () => {
        const inResult = await this._def.in._parseAsync({
          data: ctx.data,
          path: ctx.path,
          parent: ctx
        });
        if (inResult.status === "aborted")
          return INVALID;
        if (inResult.status === "dirty") {
          status.dirty();
          return DIRTY(inResult.value);
        } else {
          return this._def.out._parseAsync({
            data: inResult.value,
            path: ctx.path,
            parent: ctx
          });
        }
      };
      return handleAsync();
    } else {
      const inResult = this._def.in._parseSync({
        data: ctx.data,
        path: ctx.path,
        parent: ctx
      });
      if (inResult.status === "aborted")
        return INVALID;
      if (inResult.status === "dirty") {
        status.dirty();
        return {
          status: "dirty",
          value: inResult.value
        };
      } else {
        return this._def.out._parseSync({
          data: inResult.value,
          path: ctx.path,
          parent: ctx
        });
      }
    }
  }
  static create(a, b) {
    return new ZodPipeline({
      in: a,
      out: b,
      typeName: ZodFirstPartyTypeKind.ZodPipeline
    });
  }
}
class ZodReadonly extends ZodType {
  _parse(input) {
    const result = this._def.innerType._parse(input);
    const freeze2 = (data) => {
      if (isValid(data)) {
        data.value = Object.freeze(data.value);
      }
      return data;
    };
    return isAsync(result) ? result.then((data) => freeze2(data)) : freeze2(result);
  }
  unwrap() {
    return this._def.innerType;
  }
}
ZodReadonly.create = (type, params) => {
  return new ZodReadonly({
    innerType: type,
    typeName: ZodFirstPartyTypeKind.ZodReadonly,
    ...processCreateParams(params)
  });
};
function cleanParams(params, data) {
  const p = typeof params === "function" ? params(data) : typeof params === "string" ? { message: params } : params;
  const p2 = typeof p === "string" ? { message: p } : p;
  return p2;
}
function custom(check, _params = {}, fatal) {
  if (check)
    return ZodAny.create().superRefine((data, ctx) => {
      var _a, _b;
      const r = check(data);
      if (r instanceof Promise) {
        return r.then((r2) => {
          var _a2, _b2;
          if (!r2) {
            const params = cleanParams(_params, data);
            const _fatal = (_b2 = (_a2 = params.fatal) !== null && _a2 !== void 0 ? _a2 : fatal) !== null && _b2 !== void 0 ? _b2 : true;
            ctx.addIssue({ code: "custom", ...params, fatal: _fatal });
          }
        });
      }
      if (!r) {
        const params = cleanParams(_params, data);
        const _fatal = (_b = (_a = params.fatal) !== null && _a !== void 0 ? _a : fatal) !== null && _b !== void 0 ? _b : true;
        ctx.addIssue({ code: "custom", ...params, fatal: _fatal });
      }
      return;
    });
  return ZodAny.create();
}
const late = {
  object: ZodObject.lazycreate
};
var ZodFirstPartyTypeKind;
(function(ZodFirstPartyTypeKind2) {
  ZodFirstPartyTypeKind2["ZodString"] = "ZodString";
  ZodFirstPartyTypeKind2["ZodNumber"] = "ZodNumber";
  ZodFirstPartyTypeKind2["ZodNaN"] = "ZodNaN";
  ZodFirstPartyTypeKind2["ZodBigInt"] = "ZodBigInt";
  ZodFirstPartyTypeKind2["ZodBoolean"] = "ZodBoolean";
  ZodFirstPartyTypeKind2["ZodDate"] = "ZodDate";
  ZodFirstPartyTypeKind2["ZodSymbol"] = "ZodSymbol";
  ZodFirstPartyTypeKind2["ZodUndefined"] = "ZodUndefined";
  ZodFirstPartyTypeKind2["ZodNull"] = "ZodNull";
  ZodFirstPartyTypeKind2["ZodAny"] = "ZodAny";
  ZodFirstPartyTypeKind2["ZodUnknown"] = "ZodUnknown";
  ZodFirstPartyTypeKind2["ZodNever"] = "ZodNever";
  ZodFirstPartyTypeKind2["ZodVoid"] = "ZodVoid";
  ZodFirstPartyTypeKind2["ZodArray"] = "ZodArray";
  ZodFirstPartyTypeKind2["ZodObject"] = "ZodObject";
  ZodFirstPartyTypeKind2["ZodUnion"] = "ZodUnion";
  ZodFirstPartyTypeKind2["ZodDiscriminatedUnion"] = "ZodDiscriminatedUnion";
  ZodFirstPartyTypeKind2["ZodIntersection"] = "ZodIntersection";
  ZodFirstPartyTypeKind2["ZodTuple"] = "ZodTuple";
  ZodFirstPartyTypeKind2["ZodRecord"] = "ZodRecord";
  ZodFirstPartyTypeKind2["ZodMap"] = "ZodMap";
  ZodFirstPartyTypeKind2["ZodSet"] = "ZodSet";
  ZodFirstPartyTypeKind2["ZodFunction"] = "ZodFunction";
  ZodFirstPartyTypeKind2["ZodLazy"] = "ZodLazy";
  ZodFirstPartyTypeKind2["ZodLiteral"] = "ZodLiteral";
  ZodFirstPartyTypeKind2["ZodEnum"] = "ZodEnum";
  ZodFirstPartyTypeKind2["ZodEffects"] = "ZodEffects";
  ZodFirstPartyTypeKind2["ZodNativeEnum"] = "ZodNativeEnum";
  ZodFirstPartyTypeKind2["ZodOptional"] = "ZodOptional";
  ZodFirstPartyTypeKind2["ZodNullable"] = "ZodNullable";
  ZodFirstPartyTypeKind2["ZodDefault"] = "ZodDefault";
  ZodFirstPartyTypeKind2["ZodCatch"] = "ZodCatch";
  ZodFirstPartyTypeKind2["ZodPromise"] = "ZodPromise";
  ZodFirstPartyTypeKind2["ZodBranded"] = "ZodBranded";
  ZodFirstPartyTypeKind2["ZodPipeline"] = "ZodPipeline";
  ZodFirstPartyTypeKind2["ZodReadonly"] = "ZodReadonly";
})(ZodFirstPartyTypeKind || (ZodFirstPartyTypeKind = {}));
const instanceOfType = (cls, params = {
  message: `Input not instance of ${cls.name}`
}) => custom((data) => data instanceof cls, params);
const stringType = ZodString.create;
const numberType = ZodNumber.create;
const nanType = ZodNaN.create;
const bigIntType = ZodBigInt.create;
const booleanType = ZodBoolean.create;
const dateType = ZodDate.create;
const symbolType = ZodSymbol.create;
const undefinedType = ZodUndefined.create;
const nullType = ZodNull.create;
const anyType = ZodAny.create;
const unknownType = ZodUnknown.create;
const neverType = ZodNever.create;
const voidType = ZodVoid.create;
const arrayType = ZodArray.create;
const objectType = ZodObject.create;
const strictObjectType = ZodObject.strictCreate;
const unionType = ZodUnion.create;
const discriminatedUnionType = ZodDiscriminatedUnion.create;
const intersectionType = ZodIntersection.create;
const tupleType = ZodTuple.create;
const recordType = ZodRecord.create;
const mapType = ZodMap.create;
const setType = ZodSet.create;
const functionType = ZodFunction.create;
const lazyType = ZodLazy.create;
const literalType = ZodLiteral.create;
const enumType = ZodEnum.create;
const nativeEnumType = ZodNativeEnum.create;
const promiseType = ZodPromise.create;
const effectsType = ZodEffects.create;
const optionalType = ZodOptional.create;
const nullableType = ZodNullable.create;
const preprocessType = ZodEffects.createWithPreprocess;
const pipelineType = ZodPipeline.create;
const ostring = () => stringType().optional();
const onumber = () => numberType().optional();
const oboolean = () => booleanType().optional();
const coerce = {
  string: (arg) => ZodString.create({ ...arg, coerce: true }),
  number: (arg) => ZodNumber.create({ ...arg, coerce: true }),
  boolean: (arg) => ZodBoolean.create({
    ...arg,
    coerce: true
  }),
  bigint: (arg) => ZodBigInt.create({ ...arg, coerce: true }),
  date: (arg) => ZodDate.create({ ...arg, coerce: true })
};
const NEVER = INVALID;
var z = /* @__PURE__ */ Object.freeze({
  __proto__: null,
  defaultErrorMap: errorMap,
  setErrorMap,
  getErrorMap,
  makeIssue,
  EMPTY_PATH,
  addIssueToContext,
  ParseStatus,
  INVALID,
  DIRTY,
  OK: OK2,
  isAborted,
  isDirty,
  isValid,
  isAsync,
  get util() {
    return util;
  },
  get objectUtil() {
    return objectUtil;
  },
  ZodParsedType,
  getParsedType,
  ZodType,
  datetimeRegex,
  ZodString,
  ZodNumber,
  ZodBigInt,
  ZodBoolean,
  ZodDate,
  ZodSymbol,
  ZodUndefined,
  ZodNull,
  ZodAny,
  ZodUnknown,
  ZodNever,
  ZodVoid,
  ZodArray,
  ZodObject,
  ZodUnion,
  ZodDiscriminatedUnion,
  ZodIntersection,
  ZodTuple,
  ZodRecord,
  ZodMap,
  ZodSet,
  ZodFunction,
  ZodLazy,
  ZodLiteral,
  ZodEnum,
  ZodNativeEnum,
  ZodPromise,
  ZodEffects,
  ZodTransformer: ZodEffects,
  ZodOptional,
  ZodNullable,
  ZodDefault,
  ZodCatch,
  ZodNaN,
  BRAND,
  ZodBranded,
  ZodPipeline,
  ZodReadonly,
  custom,
  Schema: ZodType,
  ZodSchema: ZodType,
  late,
  get ZodFirstPartyTypeKind() {
    return ZodFirstPartyTypeKind;
  },
  coerce,
  any: anyType,
  array: arrayType,
  bigint: bigIntType,
  boolean: booleanType,
  date: dateType,
  discriminatedUnion: discriminatedUnionType,
  effect: effectsType,
  "enum": enumType,
  "function": functionType,
  "instanceof": instanceOfType,
  intersection: intersectionType,
  lazy: lazyType,
  literal: literalType,
  map: mapType,
  nan: nanType,
  nativeEnum: nativeEnumType,
  never: neverType,
  "null": nullType,
  nullable: nullableType,
  number: numberType,
  object: objectType,
  oboolean,
  onumber,
  optional: optionalType,
  ostring,
  pipeline: pipelineType,
  preprocess: preprocessType,
  promise: promiseType,
  record: recordType,
  set: setType,
  strictObject: strictObjectType,
  string: stringType,
  symbol: symbolType,
  transformer: effectsType,
  tuple: tupleType,
  "undefined": undefinedType,
  union: unionType,
  unknown: unknownType,
  "void": voidType,
  NEVER,
  ZodIssueCode,
  quotelessJson,
  ZodError
});
const TextBookItem = z.object({
  textbook_id: z.number(),
  textbook_name: z.string(),
  version_id: z.number(),
  catalogs: z.array(z.object({
    id: z.number(),
    name: z.string(),
    children: z.array(z.object({
      id: z.number(),
      name: z.string()
    }))
  }))
});
const ListTextBooksRes = z.object({
  limit: z.number(),
  skip: z.number(),
  total: z.number(),
  catalog_data: z.array(TextBookItem)
});
const SubjectEnum = z.enum(["CHI", "MAT", "EN", "PHY", "CHEM", "BIO", "HIS", "GEO", "SXZZ"]);
const APISectionType = z.enum(["single_choice", "multi_choice", "single_blank"]);
const SubjectNameMap = {
  CHI: "语文",
  MAT: "数学",
  EN: "英语",
  PHY: "物理",
  CHEM: "化学",
  BIO: "生物",
  HIS: "历史",
  GEO: "地理",
  SXZZ: "政治"
};
const subjectOptions = SubjectEnum.options.map((key) => ({
  label: SubjectNameMap[key],
  value: key
}));
const QAType = z.object({
  created_at: z.string(),
  id: z.coerce.string(),
  name: z.string(),
  subject: SubjectEnum,
  updated_at: z.string()
});
const QuizState = z.enum(["REVIEW", "OK", "OFFLINE"]);
const SourceFrom = z.enum(["hh_admin", "xkw"]);
const APIQuestionSourceInfo = z.object({
  /**
   * 地区
   */
  district: z.string(),
  /**
   * 考试类型
   */
  source_type: z.string(),
  /**
   * 标题
   */
  title: z.string(),
  /**
   * 年份
   */
  year: z.number(),
  /**
   * 年级
   */
  grade_name: z.string().optional(),
  /**
   * 课本
   */
  catalog_ids: z.array(z.number()).optional(),
  /**
   * 应用场景
   */
  apply_scene: z.string().optional(),
  /**
   * 适用范围
   */
  apply_scope: z.string().optional(),
  /**
   * 标签
   */
  tags: z.array(z.object({ name: z.string(), type: z.string().optional() })).optional()
});
const KPointStage = z.enum(["SENIOR_MIDDLE", "JUNIOR_MIDDLE"]);
const APIKPoint = z.object({
  /**
   * 知识点ID
   */
  id: z.number(),
  /**
   * 课程ID
   */
  course_id: z.number(),
  /**
   * 知识点名称
   */
  name: z.string(),
  /**
   * 排序
   */
  ordinal: z.number().optional(),
  /**
   * 知识点父ID
   */
  parent_id: z.number(),
  is_main: z.literal(0).or(z.literal(1)).optional(),
  /**
   * 知识点所有父ID
   */
  parent_ids: z.array(z.number()),
  /**
   * 高级还是初级
   */
  stage: KPointStage.optional().nullable()
});
z.object({
  content: z.string(),
  images: z.array(z.object({ src: z.string() }))
});
const APIQuestion = z.object({
  id: z.number(),
  subject: SubjectEnum,
  qa_type_id: z.coerce.string(),
  content: z.string(),
  updated_at: z.string(),
  state: QuizState,
  source_from: SourceFrom,
  extra_info: z.object({
    ai_q_analysis: z.string().optional(),
    qa_type_name: z.string()
  }),
  sections_info: z.object({
    sections: z.array(z.object({
      answers: z.array(z.object({
        values: z.array(z.string())
      })),
      choices: z.array(z.object({
        desc: z.string(),
        value: z.string()
      })),
      mark: z.object({
        raw: z.string(),
        value: z.string()
      }),
      content: z.string(),
      sec_type: APISectionType,
      solution: z.string()
    })),
    difficulty_name: z.string().optional(),
    images: z.array(z.object({ src: z.string() })).optional()
  }),
  source_info: APIQuestionSourceInfo
});
const QAConfig = z.object({
  /**
  * 应用场景
  * ["同步教学-开学考试", "同步教学-新授课", "同步教学-单元复习"]
  */
  application_scenarios: z.array(z.string()),
  /**
   * 课本
   */
  catalog_data: z.array(TextBookItem),
  /**
   * 考试类型
   */
  exam_types: z.array(z.string()),
  /**
   * 年级
   */
  grade_names: z.array(z.string()),
  /**
   * 科目
   */
  subjects: z.array(z.string()),
  /**
   * 年份
   */
  years: z.array(z.string()),
  /**
   * 难度
   */
  difficulties: z.array(z.string()),
  /**
   * 试题类型
   */
  qa_types: z.array(QAType)
});
var FileType = /* @__PURE__ */ ((FileType2) => {
  FileType2[FileType2["Unknown"] = 0] = "Unknown";
  FileType2[FileType2["Image"] = 1] = "Image";
  FileType2[FileType2["Video"] = 2] = "Video";
  return FileType2;
})(FileType || {});
var NOTHING = Symbol.for("immer-nothing");
var DRAFTABLE = Symbol.for("immer-draftable");
var DRAFT_STATE = Symbol.for("immer-state");
function die(error, ...args) {
  throw new Error(
    `[Immer] minified error nr: ${error}. Full error at: https://bit.ly/3cXEKWf`
  );
}
var getPrototypeOf$1 = Object.getPrototypeOf;
function isDraft(value) {
  return !!value && !!value[DRAFT_STATE];
}
function isDraftable(value) {
  var _a;
  if (!value)
    return false;
  return isPlainObject$1(value) || Array.isArray(value) || !!value[DRAFTABLE] || !!((_a = value.constructor) == null ? void 0 : _a[DRAFTABLE]) || isMap(value) || isSet(value);
}
var objectCtorString = Object.prototype.constructor.toString();
function isPlainObject$1(value) {
  if (!value || typeof value !== "object")
    return false;
  const proto = getPrototypeOf$1(value);
  if (proto === null) {
    return true;
  }
  const Ctor = Object.hasOwnProperty.call(proto, "constructor") && proto.constructor;
  if (Ctor === Object)
    return true;
  return typeof Ctor == "function" && Function.toString.call(Ctor) === objectCtorString;
}
function each(obj, iter) {
  if (getArchtype(obj) === 0) {
    Reflect.ownKeys(obj).forEach((key) => {
      iter(key, obj[key], obj);
    });
  } else {
    obj.forEach((entry, index) => iter(index, entry, obj));
  }
}
function getArchtype(thing) {
  const state = thing[DRAFT_STATE];
  return state ? state.type_ : Array.isArray(thing) ? 1 : isMap(thing) ? 2 : isSet(thing) ? 3 : 0;
}
function has(thing, prop) {
  return getArchtype(thing) === 2 ? thing.has(prop) : Object.prototype.hasOwnProperty.call(thing, prop);
}
function set(thing, propOrOldValue, value) {
  const t = getArchtype(thing);
  if (t === 2)
    thing.set(propOrOldValue, value);
  else if (t === 3) {
    thing.add(value);
  } else
    thing[propOrOldValue] = value;
}
function is(x, y) {
  if (x === y) {
    return x !== 0 || 1 / x === 1 / y;
  } else {
    return x !== x && y !== y;
  }
}
function isMap(target) {
  return target instanceof Map;
}
function isSet(target) {
  return target instanceof Set;
}
function latest(state) {
  return state.copy_ || state.base_;
}
function shallowCopy(base, strict) {
  if (isMap(base)) {
    return new Map(base);
  }
  if (isSet(base)) {
    return new Set(base);
  }
  if (Array.isArray(base))
    return Array.prototype.slice.call(base);
  const isPlain = isPlainObject$1(base);
  if (strict === true || strict === "class_only" && !isPlain) {
    const descriptors2 = Object.getOwnPropertyDescriptors(base);
    delete descriptors2[DRAFT_STATE];
    let keys = Reflect.ownKeys(descriptors2);
    for (let i = 0; i < keys.length; i++) {
      const key = keys[i];
      const desc = descriptors2[key];
      if (desc.writable === false) {
        desc.writable = true;
        desc.configurable = true;
      }
      if (desc.get || desc.set)
        descriptors2[key] = {
          configurable: true,
          writable: true,
          // could live with !!desc.set as well here...
          enumerable: desc.enumerable,
          value: base[key]
        };
    }
    return Object.create(getPrototypeOf$1(base), descriptors2);
  } else {
    const proto = getPrototypeOf$1(base);
    if (proto !== null && isPlain) {
      return { ...base };
    }
    const obj = Object.create(proto);
    return Object.assign(obj, base);
  }
}
function freeze(obj, deep = false) {
  if (isFrozen(obj) || isDraft(obj) || !isDraftable(obj))
    return obj;
  if (getArchtype(obj) > 1) {
    obj.set = obj.add = obj.clear = obj.delete = dontMutateFrozenCollections;
  }
  Object.freeze(obj);
  if (deep)
    Object.entries(obj).forEach(([key, value]) => freeze(value, true));
  return obj;
}
function dontMutateFrozenCollections() {
  die(2);
}
function isFrozen(obj) {
  return Object.isFrozen(obj);
}
var plugins = {};
function getPlugin(pluginKey) {
  const plugin = plugins[pluginKey];
  if (!plugin) {
    die(0, pluginKey);
  }
  return plugin;
}
var currentScope;
function getCurrentScope() {
  return currentScope;
}
function createScope(parent_, immer_) {
  return {
    drafts_: [],
    parent_,
    immer_,
    // Whenever the modified draft contains a draft from another scope, we
    // need to prevent auto-freezing so the unowned draft can be finalized.
    canAutoFreeze_: true,
    unfinalizedDrafts_: 0
  };
}
function usePatchesInScope(scope, patchListener) {
  if (patchListener) {
    getPlugin("Patches");
    scope.patches_ = [];
    scope.inversePatches_ = [];
    scope.patchListener_ = patchListener;
  }
}
function revokeScope(scope) {
  leaveScope(scope);
  scope.drafts_.forEach(revokeDraft);
  scope.drafts_ = null;
}
function leaveScope(scope) {
  if (scope === currentScope) {
    currentScope = scope.parent_;
  }
}
function enterScope(immer2) {
  return currentScope = createScope(currentScope, immer2);
}
function revokeDraft(draft) {
  const state = draft[DRAFT_STATE];
  if (state.type_ === 0 || state.type_ === 1)
    state.revoke_();
  else
    state.revoked_ = true;
}
function processResult(result, scope) {
  scope.unfinalizedDrafts_ = scope.drafts_.length;
  const baseDraft = scope.drafts_[0];
  const isReplaced = result !== void 0 && result !== baseDraft;
  if (isReplaced) {
    if (baseDraft[DRAFT_STATE].modified_) {
      revokeScope(scope);
      die(4);
    }
    if (isDraftable(result)) {
      result = finalize(scope, result);
      if (!scope.parent_)
        maybeFreeze(scope, result);
    }
    if (scope.patches_) {
      getPlugin("Patches").generateReplacementPatches_(
        baseDraft[DRAFT_STATE].base_,
        result,
        scope.patches_,
        scope.inversePatches_
      );
    }
  } else {
    result = finalize(scope, baseDraft, []);
  }
  revokeScope(scope);
  if (scope.patches_) {
    scope.patchListener_(scope.patches_, scope.inversePatches_);
  }
  return result !== NOTHING ? result : void 0;
}
function finalize(rootScope, value, path) {
  if (isFrozen(value))
    return value;
  const state = value[DRAFT_STATE];
  if (!state) {
    each(
      value,
      (key, childValue) => finalizeProperty(rootScope, state, value, key, childValue, path)
    );
    return value;
  }
  if (state.scope_ !== rootScope)
    return value;
  if (!state.modified_) {
    maybeFreeze(rootScope, state.base_, true);
    return state.base_;
  }
  if (!state.finalized_) {
    state.finalized_ = true;
    state.scope_.unfinalizedDrafts_--;
    const result = state.copy_;
    let resultEach = result;
    let isSet2 = false;
    if (state.type_ === 3) {
      resultEach = new Set(result);
      result.clear();
      isSet2 = true;
    }
    each(
      resultEach,
      (key, childValue) => finalizeProperty(rootScope, state, result, key, childValue, path, isSet2)
    );
    maybeFreeze(rootScope, result, false);
    if (path && rootScope.patches_) {
      getPlugin("Patches").generatePatches_(
        state,
        path,
        rootScope.patches_,
        rootScope.inversePatches_
      );
    }
  }
  return state.copy_;
}
function finalizeProperty(rootScope, parentState, targetObject, prop, childValue, rootPath, targetIsSet) {
  if (isDraft(childValue)) {
    const path = rootPath && parentState && parentState.type_ !== 3 && // Set objects are atomic since they have no keys.
    !has(parentState.assigned_, prop) ? rootPath.concat(prop) : void 0;
    const res = finalize(rootScope, childValue, path);
    set(targetObject, prop, res);
    if (isDraft(res)) {
      rootScope.canAutoFreeze_ = false;
    } else
      return;
  } else if (targetIsSet) {
    targetObject.add(childValue);
  }
  if (isDraftable(childValue) && !isFrozen(childValue)) {
    if (!rootScope.immer_.autoFreeze_ && rootScope.unfinalizedDrafts_ < 1) {
      return;
    }
    finalize(rootScope, childValue);
    if ((!parentState || !parentState.scope_.parent_) && typeof prop !== "symbol" && Object.prototype.propertyIsEnumerable.call(targetObject, prop))
      maybeFreeze(rootScope, childValue);
  }
}
function maybeFreeze(scope, value, deep = false) {
  if (!scope.parent_ && scope.immer_.autoFreeze_ && scope.canAutoFreeze_) {
    freeze(value, deep);
  }
}
function createProxyProxy(base, parent) {
  const isArray2 = Array.isArray(base);
  const state = {
    type_: isArray2 ? 1 : 0,
    // Track which produce call this is associated with.
    scope_: parent ? parent.scope_ : getCurrentScope(),
    // True for both shallow and deep changes.
    modified_: false,
    // Used during finalization.
    finalized_: false,
    // Track which properties have been assigned (true) or deleted (false).
    assigned_: {},
    // The parent draft state.
    parent_: parent,
    // The base state.
    base_: base,
    // The base proxy.
    draft_: null,
    // set below
    // The base copy with any updated values.
    copy_: null,
    // Called by the `produce` function.
    revoke_: null,
    isManual_: false
  };
  let target = state;
  let traps = objectTraps;
  if (isArray2) {
    target = [state];
    traps = arrayTraps;
  }
  const { revoke, proxy } = Proxy.revocable(target, traps);
  state.draft_ = proxy;
  state.revoke_ = revoke;
  return proxy;
}
var objectTraps = {
  get(state, prop) {
    if (prop === DRAFT_STATE)
      return state;
    const source = latest(state);
    if (!has(source, prop)) {
      return readPropFromProto(state, source, prop);
    }
    const value = source[prop];
    if (state.finalized_ || !isDraftable(value)) {
      return value;
    }
    if (value === peek(state.base_, prop)) {
      prepareCopy(state);
      return state.copy_[prop] = createProxy(value, state);
    }
    return value;
  },
  has(state, prop) {
    return prop in latest(state);
  },
  ownKeys(state) {
    return Reflect.ownKeys(latest(state));
  },
  set(state, prop, value) {
    const desc = getDescriptorFromProto(latest(state), prop);
    if (desc == null ? void 0 : desc.set) {
      desc.set.call(state.draft_, value);
      return true;
    }
    if (!state.modified_) {
      const current2 = peek(latest(state), prop);
      const currentState = current2 == null ? void 0 : current2[DRAFT_STATE];
      if (currentState && currentState.base_ === value) {
        state.copy_[prop] = value;
        state.assigned_[prop] = false;
        return true;
      }
      if (is(value, current2) && (value !== void 0 || has(state.base_, prop)))
        return true;
      prepareCopy(state);
      markChanged(state);
    }
    if (state.copy_[prop] === value && // special case: handle new props with value 'undefined'
    (value !== void 0 || prop in state.copy_) || // special case: NaN
    Number.isNaN(value) && Number.isNaN(state.copy_[prop]))
      return true;
    state.copy_[prop] = value;
    state.assigned_[prop] = true;
    return true;
  },
  deleteProperty(state, prop) {
    if (peek(state.base_, prop) !== void 0 || prop in state.base_) {
      state.assigned_[prop] = false;
      prepareCopy(state);
      markChanged(state);
    } else {
      delete state.assigned_[prop];
    }
    if (state.copy_) {
      delete state.copy_[prop];
    }
    return true;
  },
  // Note: We never coerce `desc.value` into an Immer draft, because we can't make
  // the same guarantee in ES5 mode.
  getOwnPropertyDescriptor(state, prop) {
    const owner = latest(state);
    const desc = Reflect.getOwnPropertyDescriptor(owner, prop);
    if (!desc)
      return desc;
    return {
      writable: true,
      configurable: state.type_ !== 1 || prop !== "length",
      enumerable: desc.enumerable,
      value: owner[prop]
    };
  },
  defineProperty() {
    die(11);
  },
  getPrototypeOf(state) {
    return getPrototypeOf$1(state.base_);
  },
  setPrototypeOf() {
    die(12);
  }
};
var arrayTraps = {};
each(objectTraps, (key, fn) => {
  arrayTraps[key] = function() {
    arguments[0] = arguments[0][0];
    return fn.apply(this, arguments);
  };
});
arrayTraps.deleteProperty = function(state, prop) {
  return arrayTraps.set.call(this, state, prop, void 0);
};
arrayTraps.set = function(state, prop, value) {
  return objectTraps.set.call(this, state[0], prop, value, state[0]);
};
function peek(draft, prop) {
  const state = draft[DRAFT_STATE];
  const source = state ? latest(state) : draft;
  return source[prop];
}
function readPropFromProto(state, source, prop) {
  var _a;
  const desc = getDescriptorFromProto(source, prop);
  return desc ? `value` in desc ? desc.value : (
    // This is a very special case, if the prop is a getter defined by the
    // prototype, we should invoke it with the draft as context!
    (_a = desc.get) == null ? void 0 : _a.call(state.draft_)
  ) : void 0;
}
function getDescriptorFromProto(source, prop) {
  if (!(prop in source))
    return void 0;
  let proto = getPrototypeOf$1(source);
  while (proto) {
    const desc = Object.getOwnPropertyDescriptor(proto, prop);
    if (desc)
      return desc;
    proto = getPrototypeOf$1(proto);
  }
  return void 0;
}
function markChanged(state) {
  if (!state.modified_) {
    state.modified_ = true;
    if (state.parent_) {
      markChanged(state.parent_);
    }
  }
}
function prepareCopy(state) {
  if (!state.copy_) {
    state.copy_ = shallowCopy(
      state.base_,
      state.scope_.immer_.useStrictShallowCopy_
    );
  }
}
var Immer2 = class {
  constructor(config) {
    this.autoFreeze_ = true;
    this.useStrictShallowCopy_ = false;
    this.produce = (base, recipe, patchListener) => {
      if (typeof base === "function" && typeof recipe !== "function") {
        const defaultBase = recipe;
        recipe = base;
        const self2 = this;
        return function curriedProduce(base2 = defaultBase, ...args) {
          return self2.produce(base2, (draft) => recipe.call(this, draft, ...args));
        };
      }
      if (typeof recipe !== "function")
        die(6);
      if (patchListener !== void 0 && typeof patchListener !== "function")
        die(7);
      let result;
      if (isDraftable(base)) {
        const scope = enterScope(this);
        const proxy = createProxy(base, void 0);
        let hasError = true;
        try {
          result = recipe(proxy);
          hasError = false;
        } finally {
          if (hasError)
            revokeScope(scope);
          else
            leaveScope(scope);
        }
        usePatchesInScope(scope, patchListener);
        return processResult(result, scope);
      } else if (!base || typeof base !== "object") {
        result = recipe(base);
        if (result === void 0)
          result = base;
        if (result === NOTHING)
          result = void 0;
        if (this.autoFreeze_)
          freeze(result, true);
        if (patchListener) {
          const p = [];
          const ip = [];
          getPlugin("Patches").generateReplacementPatches_(base, result, p, ip);
          patchListener(p, ip);
        }
        return result;
      } else
        die(1, base);
    };
    this.produceWithPatches = (base, recipe) => {
      if (typeof base === "function") {
        return (state, ...args) => this.produceWithPatches(state, (draft) => base(draft, ...args));
      }
      let patches, inversePatches;
      const result = this.produce(base, recipe, (p, ip) => {
        patches = p;
        inversePatches = ip;
      });
      return [result, patches, inversePatches];
    };
    if (typeof (config == null ? void 0 : config.autoFreeze) === "boolean")
      this.setAutoFreeze(config.autoFreeze);
    if (typeof (config == null ? void 0 : config.useStrictShallowCopy) === "boolean")
      this.setUseStrictShallowCopy(config.useStrictShallowCopy);
  }
  createDraft(base) {
    if (!isDraftable(base))
      die(8);
    if (isDraft(base))
      base = current(base);
    const scope = enterScope(this);
    const proxy = createProxy(base, void 0);
    proxy[DRAFT_STATE].isManual_ = true;
    leaveScope(scope);
    return proxy;
  }
  finishDraft(draft, patchListener) {
    const state = draft && draft[DRAFT_STATE];
    if (!state || !state.isManual_)
      die(9);
    const { scope_: scope } = state;
    usePatchesInScope(scope, patchListener);
    return processResult(void 0, scope);
  }
  /**
   * Pass true to automatically freeze all copies created by Immer.
   *
   * By default, auto-freezing is enabled.
   */
  setAutoFreeze(value) {
    this.autoFreeze_ = value;
  }
  /**
   * Pass true to enable strict shallow copy.
   *
   * By default, immer does not copy the object descriptors such as getter, setter and non-enumrable properties.
   */
  setUseStrictShallowCopy(value) {
    this.useStrictShallowCopy_ = value;
  }
  applyPatches(base, patches) {
    let i;
    for (i = patches.length - 1; i >= 0; i--) {
      const patch = patches[i];
      if (patch.path.length === 0 && patch.op === "replace") {
        base = patch.value;
        break;
      }
    }
    if (i > -1) {
      patches = patches.slice(i + 1);
    }
    const applyPatchesImpl = getPlugin("Patches").applyPatches_;
    if (isDraft(base)) {
      return applyPatchesImpl(base, patches);
    }
    return this.produce(
      base,
      (draft) => applyPatchesImpl(draft, patches)
    );
  }
};
function createProxy(value, parent) {
  const draft = isMap(value) ? getPlugin("MapSet").proxyMap_(value, parent) : isSet(value) ? getPlugin("MapSet").proxySet_(value, parent) : createProxyProxy(value, parent);
  const scope = parent ? parent.scope_ : getCurrentScope();
  scope.drafts_.push(draft);
  return draft;
}
function current(value) {
  if (!isDraft(value))
    die(10, value);
  return currentImpl(value);
}
function currentImpl(value) {
  if (!isDraftable(value) || isFrozen(value))
    return value;
  const state = value[DRAFT_STATE];
  let copy;
  if (state) {
    if (!state.modified_)
      return state.base_;
    state.finalized_ = true;
    copy = shallowCopy(value, state.scope_.immer_.useStrictShallowCopy_);
  } else {
    copy = shallowCopy(value, true);
  }
  each(copy, (key, childValue) => {
    set(copy, key, currentImpl(childValue));
  });
  if (state) {
    state.finalized_ = false;
  }
  return copy;
}
var immer = new Immer2();
var produce = immer.produce;
immer.produceWithPatches.bind(
  immer
);
immer.setAutoFreeze.bind(immer);
immer.setUseStrictShallowCopy.bind(immer);
immer.applyPatches.bind(immer);
immer.createDraft.bind(immer);
immer.finishDraft.bind(immer);
const QuizChoice = z.object({
  key: z.string(),
  uid: z.string(),
  value: z.string()
});
const QuizSection = z.object({
  id: z.string(),
  answers: z.array(z.string()),
  solution: z.string(),
  type: APISectionType,
  choices: z.array(QuizChoice),
  content: z.string()
});
const KnowledgePoint = z.object({
  id: z.coerce.string(),
  name: z.string(),
  parent_id: z.coerce.string(),
  stage: KPointStage.optional().nullable(),
  is_main: z.boolean().optional()
});
const AnswerAndKnowledgePoint = z.object({
  question: z.string(),
  answer: z.string(),
  explanation: z.string(),
  knowledgePoints: z.array(KnowledgePoint)
});
z.array(AnswerAndKnowledgePoint);
const Quiz = z.object({
  id: z.string(),
  quizTypeId: z.string(),
  quizTypeName: z.string(),
  subject: SubjectEnum,
  content: z.string(),
  state: QuizState,
  difficulty: z.string(),
  sections: z.array(QuizSection),
  imageUrls: z.array(z.string()).optional(),
  knowledgePoints: z.array(z.object({ id: z.string(), name: z.string(), isMain: z.boolean() })),
  knowledgePointIds: z.array(z.string()).optional(),
  source: APIQuestionSourceInfo.omit({ tags: true }),
  sourceFrom: SourceFrom,
  rate: z.number().default(3),
  /**
   * AI详解
   */
  aiAnalysis: z.string().optional(),
  updatedAt: z.string(),
  tags: z.array(z.object({ name: z.string(), type: z.string(), id: z.string() }))
});
z.object({
  quiz: Quiz,
  knowledges: z.array(KnowledgePoint),
  quizNumList: z.array(z.string()),
  flowResult: z.string().optional()
});
var QuizActions;
((QuizActions2) => {
  function copyWith(q, patch) {
    return produce(q, (draft) => {
      Object.assign(draft, patch);
    });
  }
  QuizActions2.copyWith = copyWith;
  function updateSection(q, predicate, updater) {
    return produce(q, (draft) => {
      const section = draft.sections.find(predicate);
      if (section) {
        Object.assign(section, updater(section));
      }
    });
  }
  QuizActions2.updateSection = updateSection;
  function updateSectionById(q, id2, updater) {
    return updateSection(q, (section) => section.id === id2, updater);
  }
  QuizActions2.updateSectionById = updateSectionById;
  function addSection(q, section) {
    return produce(q, (draft) => {
      draft.sections.push(section);
    });
  }
  QuizActions2.addSection = addSection;
  function removeSection(q, id2) {
    return produce(q, (draft) => {
      draft.sections = draft.sections.filter((section) => section.id !== id2);
    });
  }
  QuizActions2.removeSection = removeSection;
})(QuizActions || (QuizActions = {}));
const UserInfo = z.object({
  nickname: z.string(),
  user_id: z.coerce.string(),
  avatar_url: z.string(),
  is_superuser: z.number()
});
const CozeToken = z.object({
  access_token: z.string(),
  expires_in: z.number()
});
const PaperSourceEnum = z.enum(["Internal", "External"]);
const PaperSourceMap = {
  Internal: "鸿鹄试卷",
  External: "外部试卷"
};
PaperSourceEnum.options.map((key) => ({
  label: PaperSourceMap[key],
  value: key
}));
class BusinessError extends Error {
  constructor(code, message, object) {
    super(message);
    __publicField(this, "code");
    __publicField(this, "object");
    this.code = code;
    this.name = "BusinessError";
    this.object = object;
    Object.setPrototypeOf(this, BusinessError.prototype);
  }
}
class DuplicatedQuizError extends Error {
  constructor(quizId, message) {
    super(message);
    __publicField(this, "quizId");
    this.quizId = quizId;
    this.name = "DuplicatedError";
  }
}
const ProficiencyEnum = z.enum(["excellent", "good", "fair", "poor", "untested"]);
const ExamListItem = z.object({
  // 试卷编号
  ep_no: z.string(),
  // 学号
  student_no: z.string(),
  // 学生姓名
  student_nickname: z.string(),
  // 错题数量
  num_cross: z.number(),
  // 未识别题目数量
  num_blank: z.number(),
  // 正确数量
  num_check: z.number()
});
const ExamDetail = z.object({
  ep_no: z.string(),
  student_no: z.string(),
  img_list: z.array(z.string()),
  exampaper_stats: z.array(z.object({
    eq_id: z.number(),
    qa_id: z.number(),
    list_qa_ids: z.array(z.object({
      qa_no: z.string(),
      mark: z.string()
    }))
  })),
  res_mark_dict: z.record(z.string())
});
const Student = z.object({
  // 学生姓名
  nickname: z.string(),
  // 学号
  no: z.string(),
  // 班级ID
  class_id: z.coerce.string()
});
z.object({
  student: Student,
  examInfo: z.object({
    // 错误数量
    crossNum: z.number(),
    // 总数量
    total: z.number()
  })
});
z.object({
  id: z.string(),
  name: z.string(),
  grade: z.string()
});
const ExamPaperInfo = z.object({
  // 试卷编号
  no: z.string(),
  // 试卷标题
  title: z.string(),
  // 试题总数量
  quizNum: z.number(),
  // 总分
  score: z.number(),
  // 预计时间(分钟)
  estimatedTimeMinutes: z.number(),
  // 难度
  difficulty: z.string(),
  // 学科
  subject: SubjectEnum,
  // 原始图片列表
  sourceImageUrls: z.array(z.string())
});
const ExamInfo = z.object({
  // 得分
  score: z.number(),
  // 答对数量
  correctNum: z.number(),
  // 答错数量
  incorrectNum: z.number(),
  // 考试时间
  examTime: z.string()
});
z.object({
  // 试卷信息
  examPaper: ExamPaperInfo,
  // 学生信息
  student: Student,
  // 考试信息
  exam: ExamInfo,
  knowpoints: z.array(KnowledgePoint)
});
const AddQuizExtra = z.object({
  xkw_kpoints: z.array(z.object({ name: z.string(), id: z.number() })),
  apply_scene: z.string(),
  district: z.string(),
  grade_name: z.string(),
  difficulty_name: z.string(),
  year: z.number(),
  xkw_type_name: z.string()
});
const XKWQuiz = z.object({
  id: z.coerce.string(),
  xkw_qa_id: z.string(),
  type: z.string(),
  kpoints_list: z.array(z.object({ id: z.string(), name: z.string() })),
  difficulty: z.string(),
  htmlSource: z.string(),
  flow_input_image_urls: z.array(z.string()),
  flow_input_text: z.string(),
  qa_content_image_urls: z.array(z.string()),
  qa_id: z.coerce.string().optional(),
  subject: z.string(),
  extra: AddQuizExtra.optional().nullable()
});
const GetXKWQuizByXKWQuizIdResponse = z.object({
  xkw_info: XKWQuiz
});
z.object({
  xkwQuizzes: z.array(XKWQuiz),
  total: z.number()
});
const ListQuizTagsResponse = z.object({
  tagdetail_list: z.array(z.object({
    name: z.string(),
    id: z.number(),
    type: z.string()
  }))
});
const AddQuizRecordType = z.enum(["exampaper_qa", "quiz", "qaxkw_info"]);
const QuizInputRecord = z.object({
  id: z.coerce.string(),
  subject: SubjectEnum,
  from_type: AddQuizRecordType,
  from_id: z.string().optional().nullable(),
  review_comment: z.string().optional(),
  input_data: z.object({
    content: z.string(),
    images: z.array(z.object({
      src: z.string()
    })),
    content_images: z.array(z.object({ src: z.string() })),
    input_type: z.literal("recognize").or(z.literal("split")),
    extra: z.any().optional()
  }),
  result_data: z.object({
    already_exists: z.literal(0).or(z.literal(1)).optional(),
    inputrecord_ids: z.array(z.number()).optional()
  }),
  state: z.string(),
  failed_reason: z.string(),
  flow_data: z.string(),
  qa_id: z.coerce.number().optional(),
  bg_runflow: z.number().optional()
});
const BatchListQuizRecordsResponse = z.object({
  inputrecord_list: z.array(QuizInputRecord)
});
const ListQuestionsRes = z.object({
  items: z.array(z.object({
    qa: APIQuestion,
    kpoints: z.array(APIKPoint)
  })),
  limit: z.number(),
  skip: z.number(),
  total: z.number()
});
const APIAddQuestionRes = z.object({
  qa_id: z.coerce.string()
});
const QuestionDetail = z.object({
  qa: APIQuestion,
  kpoints: z.array(APIKPoint)
});
const QuizItem = z.object({
  quiz: Quiz,
  kpoints: z.array(KnowledgePoint)
});
z.object({
  items: z.array(QuizItem),
  limit: z.number(),
  skip: z.number(),
  total: z.number()
});
const APIPointsResponse = z.object({
  items: z.array(APIKPoint),
  limit: z.number(),
  skip: z.number(),
  total: z.number()
});
const APIPointsRootResponse = z.object({
  items: z.array(APIKPoint)
});
const APIQiniuUploadConfig = z.object({
  token: z.string(),
  bucketName: z.string(),
  accessKey: z.string(),
  domain: z.string()
});
const APIFlowConfig = z.object({
  flow_id: z.string(),
  name: z.string()
});
const GetFlowConfigResponse = z.object({
  url: z.string(),
  configs: z.array(APIFlowConfig)
});
const GetFlowTokenResponse = z.object({
  refresh_token: z.string(),
  api_key: z.string()
});
const GetUserInfoResponse = z.object({
  userinfo: UserInfo
});
const ExamPaper = z.object({
  id: z.number(),
  created_at: z.string(),
  qa_num: z.number(),
  qa_num_completed: z.number(),
  qa_num_init: z.number(),
  user_id: z.number(),
  title: z.string(),
  subject: z.string(),
  year: z.number(),
  grade_name: z.string(),
  catalog_ids: z.array(z.coerce.string()),
  district: z.string(),
  source_type: z.string(),
  apply_scene: z.string(),
  exampaper_source: z.string(),
  updated_at: z.string().datetime({ local: true }),
  apply_scope: z.string().optional()
});
ExamPaper.omit({
  id: true,
  created_at: true,
  qa_num: true,
  qa_num_completed: true,
  user_id: true,
  updated_at: true
});
const CreateExamPaperResponse = z.object({
  exampaper_id: z.number()
});
const ListExamPapersResponse = z.object({
  exampapers: z.array(ExamPaper),
  limit: z.number(),
  skip: z.number(),
  total: z.number()
});
const ExampaperQA = z.object({
  id: z.number(),
  exampaper_id: z.number(),
  data_info: z.object({
    content: z.string(),
    images: z.array(z.object({ src: z.string() })),
    content_images: z.array(z.object({ src: z.string() })).optional()
  }),
  qa_inputrecord_id: z.number().optional(),
  state: z.string(),
  ordinal: z.number(),
  qa_id: z.number(),
  type: z.literal("recognize").or(z.literal("split")),
  already_exists: z.literal(0).or(z.literal(1)),
  list_qa_ids: z.array(z.string())
});
const ExamPaperDetail = z.object({
  exampaper: ExamPaper,
  /**
   * 试卷试题列表
   */
  exampaper_qa_list: z.array(ExampaperQA),
  /**
   * 试题列表
   */
  qa_list: z.array(APIQuestion),
  /**
   * 知识点
   */
  kpoints_detail: z.array(z.object({
    qa_id: z.number(),
    kpoints: APIKPoint
  }))
  // kpoints: z.array(APIKPoint)
});
const GetExamPaperDetailResponse = ExamPaperDetail;
const BatchListExamPaperQuizResponse = z.object({
  exampaper_qa_list: z.array(ExampaperQA),
  inputrecord_list: z.array(QuizInputRecord)
});
const ListExamsResponse = z.object({
  items: z.array(ExamListItem),
  limit: z.number(),
  skip: z.number()
});
const GetExamDetailResponse = ExamDetail;
const Course = z.object({
  id: z.number(),
  name: z.string(),
  stage: z.string()
});
const ListCoursesResponse = z.object({
  items: z.array(Course)
});
const Textbook = z.object({
  id: z.number(),
  name: z.string(),
  course_id: z.number()
});
const ListTextbooksResponse = z.object({
  items: z.array(Textbook)
});
const Catalog = z.object({
  id: z.number(),
  name: z.string(),
  textbook_id: z.number(),
  version_id: z.number(),
  parent_id: z.number()
});
const ListCatalogsResponse = z.object({
  items: z.array(Catalog)
});
const CatalogWithKPoint = z.object({
  id: z.number(),
  name: z.string(),
  // 节点ID，后端需要使用这个ID
  node_id: z.number(),
  // 父ID。通过 id 以及 parent_id 可以把数据组装成一棵树
  parent_id: z.number(),
  // 1: 章节
  // 2: 知识点
  // 3: 单词
  status: z.number(),
  // 书ID
  textbook_id: z.number()
});
const ListCatalogsWithKpointsResponse = z.object({
  items: z.array(CatalogWithKPoint),
  limit: z.number(),
  skip: z.number(),
  total: z.number()
});
const provicesText = `code,name
11,"北京市"
12,"天津市"
13,"河北省"
14,"山西省"
15,"内蒙古自治区"
21,"辽宁省"
22,"吉林省"
23,"黑龙江省"
31,"上海市"
32,"江苏省"
33,"浙江省"
34,"安徽省"
35,"福建省"
36,"江西省"
37,"山东省"
41,"河南省"
42,"湖北省"
43,"湖南省"
44,"广东省"
45,"广西壮族自治区"
46,"海南省"
50,"重庆市"
51,"四川省"
52,"贵州省"
53,"云南省"
54,"西藏自治区"
61,"陕西省"
62,"甘肃省"
63,"青海省"
64,"宁夏回族自治区"
65,"新疆维吾尔自治区"
`;
const citiesText = `code,name,provinceCode
1101,"市辖区",11
1201,"市辖区",12
1301,"石家庄市",13
1302,"唐山市",13
1303,"秦皇岛市",13
1304,"邯郸市",13
1305,"邢台市",13
1306,"保定市",13
1307,"张家口市",13
1308,"承德市",13
1309,"沧州市",13
1310,"廊坊市",13
1311,"衡水市",13
1401,"太原市",14
1402,"大同市",14
1403,"阳泉市",14
1404,"长治市",14
1405,"晋城市",14
1406,"朔州市",14
1407,"晋中市",14
1408,"运城市",14
1409,"忻州市",14
1410,"临汾市",14
1411,"吕梁市",14
1501,"呼和浩特市",15
1502,"包头市",15
1503,"乌海市",15
1504,"赤峰市",15
1505,"通辽市",15
1506,"鄂尔多斯市",15
1507,"呼伦贝尔市",15
1508,"巴彦淖尔市",15
1509,"乌兰察布市",15
1522,"兴安盟",15
1525,"锡林郭勒盟",15
1529,"阿拉善盟",15
2101,"沈阳市",21
2102,"大连市",21
2103,"鞍山市",21
2104,"抚顺市",21
2105,"本溪市",21
2106,"丹东市",21
2107,"锦州市",21
2108,"营口市",21
2109,"阜新市",21
2110,"辽阳市",21
2111,"盘锦市",21
2112,"铁岭市",21
2113,"朝阳市",21
2114,"葫芦岛市",21
2201,"长春市",22
2202,"吉林市",22
2203,"四平市",22
2204,"辽源市",22
2205,"通化市",22
2206,"白山市",22
2207,"松原市",22
2208,"白城市",22
2224,"延边朝鲜族自治州",22
2301,"哈尔滨市",23
2302,"齐齐哈尔市",23
2303,"鸡西市",23
2304,"鹤岗市",23
2305,"双鸭山市",23
2306,"大庆市",23
2307,"伊春市",23
2308,"佳木斯市",23
2309,"七台河市",23
2310,"牡丹江市",23
2311,"黑河市",23
2312,"绥化市",23
2327,"大兴安岭地区",23
3101,"市辖区",31
3201,"南京市",32
3202,"无锡市",32
3203,"徐州市",32
3204,"常州市",32
3205,"苏州市",32
3206,"南通市",32
3207,"连云港市",32
3208,"淮安市",32
3209,"盐城市",32
3210,"扬州市",32
3211,"镇江市",32
3212,"泰州市",32
3213,"宿迁市",32
3301,"杭州市",33
3302,"宁波市",33
3303,"温州市",33
3304,"嘉兴市",33
3305,"湖州市",33
3306,"绍兴市",33
3307,"金华市",33
3308,"衢州市",33
3309,"舟山市",33
3310,"台州市",33
3311,"丽水市",33
3401,"合肥市",34
3402,"芜湖市",34
3403,"蚌埠市",34
3404,"淮南市",34
3405,"马鞍山市",34
3406,"淮北市",34
3407,"铜陵市",34
3408,"安庆市",34
3410,"黄山市",34
3411,"滁州市",34
3412,"阜阳市",34
3413,"宿州市",34
3415,"六安市",34
3416,"亳州市",34
3417,"池州市",34
3418,"宣城市",34
3501,"福州市",35
3502,"厦门市",35
3503,"莆田市",35
3504,"三明市",35
3505,"泉州市",35
3506,"漳州市",35
3507,"南平市",35
3508,"龙岩市",35
3509,"宁德市",35
3601,"南昌市",36
3602,"景德镇市",36
3603,"萍乡市",36
3604,"九江市",36
3605,"新余市",36
3606,"鹰潭市",36
3607,"赣州市",36
3608,"吉安市",36
3609,"宜春市",36
3610,"抚州市",36
3611,"上饶市",36
3701,"济南市",37
3702,"青岛市",37
3703,"淄博市",37
3704,"枣庄市",37
3705,"东营市",37
3706,"烟台市",37
3707,"潍坊市",37
3708,"济宁市",37
3709,"泰安市",37
3710,"威海市",37
3711,"日照市",37
3713,"临沂市",37
3714,"德州市",37
3715,"聊城市",37
3716,"滨州市",37
3717,"菏泽市",37
4101,"郑州市",41
4102,"开封市",41
4103,"洛阳市",41
4104,"平顶山市",41
4105,"安阳市",41
4106,"鹤壁市",41
4107,"新乡市",41
4108,"焦作市",41
4109,"濮阳市",41
4110,"许昌市",41
4111,"漯河市",41
4112,"三门峡市",41
4113,"南阳市",41
4114,"商丘市",41
4115,"信阳市",41
4116,"周口市",41
4117,"驻马店市",41
4190,"省直辖县级行政区划",41
4201,"武汉市",42
4202,"黄石市",42
4203,"十堰市",42
4205,"宜昌市",42
4206,"襄阳市",42
4207,"鄂州市",42
4208,"荆门市",42
4209,"孝感市",42
4210,"荆州市",42
4211,"黄冈市",42
4212,"咸宁市",42
4213,"随州市",42
4228,"恩施土家族苗族自治州",42
4290,"省直辖县级行政区划",42
4301,"长沙市",43
4302,"株洲市",43
4303,"湘潭市",43
4304,"衡阳市",43
4305,"邵阳市",43
4306,"岳阳市",43
4307,"常德市",43
4308,"张家界市",43
4309,"益阳市",43
4310,"郴州市",43
4311,"永州市",43
4312,"怀化市",43
4313,"娄底市",43
4331,"湘西土家族苗族自治州",43
4401,"广州市",44
4402,"韶关市",44
4403,"深圳市",44
4404,"珠海市",44
4405,"汕头市",44
4406,"佛山市",44
4407,"江门市",44
4408,"湛江市",44
4409,"茂名市",44
4412,"肇庆市",44
4413,"惠州市",44
4414,"梅州市",44
4415,"汕尾市",44
4416,"河源市",44
4417,"阳江市",44
4418,"清远市",44
4419,"东莞市",44
4420,"中山市",44
4451,"潮州市",44
4452,"揭阳市",44
4453,"云浮市",44
4501,"南宁市",45
4502,"柳州市",45
4503,"桂林市",45
4504,"梧州市",45
4505,"北海市",45
4506,"防城港市",45
4507,"钦州市",45
4508,"贵港市",45
4509,"玉林市",45
4510,"百色市",45
4511,"贺州市",45
4512,"河池市",45
4513,"来宾市",45
4514,"崇左市",45
4601,"海口市",46
4602,"三亚市",46
4603,"三沙市",46
4604,"儋州市",46
4690,"省直辖县级行政区划",46
5001,"市辖区",50
5002,"县",50
5101,"成都市",51
5103,"自贡市",51
5104,"攀枝花市",51
5105,"泸州市",51
5106,"德阳市",51
5107,"绵阳市",51
5108,"广元市",51
5109,"遂宁市",51
5110,"内江市",51
5111,"乐山市",51
5113,"南充市",51
5114,"眉山市",51
5115,"宜宾市",51
5116,"广安市",51
5117,"达州市",51
5118,"雅安市",51
5119,"巴中市",51
5120,"资阳市",51
5132,"阿坝藏族羌族自治州",51
5133,"甘孜藏族自治州",51
5134,"凉山彝族自治州",51
5201,"贵阳市",52
5202,"六盘水市",52
5203,"遵义市",52
5204,"安顺市",52
5205,"毕节市",52
5206,"铜仁市",52
5223,"黔西南布依族苗族自治州",52
5226,"黔东南苗族侗族自治州",52
5227,"黔南布依族苗族自治州",52
5301,"昆明市",53
5303,"曲靖市",53
5304,"玉溪市",53
5305,"保山市",53
5306,"昭通市",53
5307,"丽江市",53
5308,"普洱市",53
5309,"临沧市",53
5323,"楚雄彝族自治州",53
5325,"红河哈尼族彝族自治州",53
5326,"文山壮族苗族自治州",53
5328,"西双版纳傣族自治州",53
5329,"大理白族自治州",53
5331,"德宏傣族景颇族自治州",53
5333,"怒江傈僳族自治州",53
5334,"迪庆藏族自治州",53
5401,"拉萨市",54
5402,"日喀则市",54
5403,"昌都市",54
5404,"林芝市",54
5405,"山南市",54
5406,"那曲市",54
5425,"阿里地区",54
6101,"西安市",61
6102,"铜川市",61
6103,"宝鸡市",61
6104,"咸阳市",61
6105,"渭南市",61
6106,"延安市",61
6107,"汉中市",61
6108,"榆林市",61
6109,"安康市",61
6110,"商洛市",61
6201,"兰州市",62
6202,"嘉峪关市",62
6203,"金昌市",62
6204,"白银市",62
6205,"天水市",62
6206,"武威市",62
6207,"张掖市",62
6208,"平凉市",62
6209,"酒泉市",62
6210,"庆阳市",62
6211,"定西市",62
6212,"陇南市",62
6229,"临夏回族自治州",62
6230,"甘南藏族自治州",62
6301,"西宁市",63
6302,"海东市",63
6322,"海北藏族自治州",63
6323,"黄南藏族自治州",63
6325,"海南藏族自治州",63
6326,"果洛藏族自治州",63
6327,"玉树藏族自治州",63
6328,"海西蒙古族藏族自治州",63
6401,"银川市",64
6402,"石嘴山市",64
6403,"吴忠市",64
6404,"固原市",64
6405,"中卫市",64
6501,"乌鲁木齐市",65
6502,"克拉玛依市",65
6504,"吐鲁番市",65
6505,"哈密市",65
6523,"昌吉回族自治州",65
6527,"博尔塔拉蒙古自治州",65
6528,"巴音郭楞蒙古自治州",65
6529,"阿克苏地区",65
6530,"克孜勒苏柯尔克孜自治州",65
6531,"喀什地区",65
6532,"和田地区",65
6540,"伊犁哈萨克自治州",65
6542,"塔城地区",65
6543,"阿勒泰地区",65
6590,"自治区直辖县级行政区划",65
`;
const areasText = `code,name,cityCode,provinceCode
110101,"东城区",1101,11
110102,"西城区",1101,11
110105,"朝阳区",1101,11
110106,"丰台区",1101,11
110107,"石景山区",1101,11
110108,"海淀区",1101,11
110109,"门头沟区",1101,11
110111,"房山区",1101,11
110112,"通州区",1101,11
110113,"顺义区",1101,11
110114,"昌平区",1101,11
110115,"大兴区",1101,11
110116,"怀柔区",1101,11
110117,"平谷区",1101,11
110118,"密云区",1101,11
110119,"延庆区",1101,11
120101,"和平区",1201,12
120102,"河东区",1201,12
120103,"河西区",1201,12
120104,"南开区",1201,12
120105,"河北区",1201,12
120106,"红桥区",1201,12
120110,"东丽区",1201,12
120111,"西青区",1201,12
120112,"津南区",1201,12
120113,"北辰区",1201,12
120114,"武清区",1201,12
120115,"宝坻区",1201,12
120116,"滨海新区",1201,12
120117,"宁河区",1201,12
120118,"静海区",1201,12
120119,"蓟州区",1201,12
130102,"长安区",1301,13
130104,"桥西区",1301,13
130105,"新华区",1301,13
130107,"井陉矿区",1301,13
130108,"裕华区",1301,13
130109,"藁城区",1301,13
130110,"鹿泉区",1301,13
130111,"栾城区",1301,13
130121,"井陉县",1301,13
130123,"正定县",1301,13
130125,"行唐县",1301,13
130126,"灵寿县",1301,13
130127,"高邑县",1301,13
130128,"深泽县",1301,13
130129,"赞皇县",1301,13
130130,"无极县",1301,13
130131,"平山县",1301,13
130132,"元氏县",1301,13
130133,"赵县",1301,13
130171,"石家庄高新技术产业开发区",1301,13
130172,"石家庄循环化工园区",1301,13
130181,"辛集市",1301,13
130183,"晋州市",1301,13
130184,"新乐市",1301,13
130202,"路南区",1302,13
130203,"路北区",1302,13
130204,"古冶区",1302,13
130205,"开平区",1302,13
130207,"丰南区",1302,13
130208,"丰润区",1302,13
130209,"曹妃甸区",1302,13
130224,"滦南县",1302,13
130225,"乐亭县",1302,13
130227,"迁西县",1302,13
130229,"玉田县",1302,13
130271,"河北唐山芦台经济开发区",1302,13
130272,"唐山市汉沽管理区",1302,13
130273,"唐山高新技术产业开发区",1302,13
130274,"河北唐山海港经济开发区",1302,13
130281,"遵化市",1302,13
130283,"迁安市",1302,13
130284,"滦州市",1302,13
130302,"海港区",1303,13
130303,"山海关区",1303,13
130304,"北戴河区",1303,13
130306,"抚宁区",1303,13
130321,"青龙满族自治县",1303,13
130322,"昌黎县",1303,13
130324,"卢龙县",1303,13
130371,"秦皇岛市经济技术开发区",1303,13
130372,"北戴河新区",1303,13
130402,"邯山区",1304,13
130403,"丛台区",1304,13
130404,"复兴区",1304,13
130406,"峰峰矿区",1304,13
130407,"肥乡区",1304,13
130408,"永年区",1304,13
130423,"临漳县",1304,13
130424,"成安县",1304,13
130425,"大名县",1304,13
130426,"涉县",1304,13
130427,"磁县",1304,13
130430,"邱县",1304,13
130431,"鸡泽县",1304,13
130432,"广平县",1304,13
130433,"馆陶县",1304,13
130434,"魏县",1304,13
130435,"曲周县",1304,13
130471,"邯郸经济技术开发区",1304,13
130473,"邯郸冀南新区",1304,13
130481,"武安市",1304,13
130502,"襄都区",1305,13
130503,"信都区",1305,13
130505,"任泽区",1305,13
130506,"南和区",1305,13
130522,"临城县",1305,13
130523,"内丘县",1305,13
130524,"柏乡县",1305,13
130525,"隆尧县",1305,13
130528,"宁晋县",1305,13
130529,"巨鹿县",1305,13
130530,"新河县",1305,13
130531,"广宗县",1305,13
130532,"平乡县",1305,13
130533,"威县",1305,13
130534,"清河县",1305,13
130535,"临西县",1305,13
130571,"河北邢台经济开发区",1305,13
130581,"南宫市",1305,13
130582,"沙河市",1305,13
130602,"竞秀区",1306,13
130606,"莲池区",1306,13
130607,"满城区",1306,13
130608,"清苑区",1306,13
130609,"徐水区",1306,13
130623,"涞水县",1306,13
130624,"阜平县",1306,13
130626,"定兴县",1306,13
130627,"唐县",1306,13
130628,"高阳县",1306,13
130629,"容城县",1306,13
130630,"涞源县",1306,13
130631,"望都县",1306,13
130632,"安新县",1306,13
130633,"易县",1306,13
130634,"曲阳县",1306,13
130635,"蠡县",1306,13
130636,"顺平县",1306,13
130637,"博野县",1306,13
130638,"雄县",1306,13
130671,"保定高新技术产业开发区",1306,13
130672,"保定白沟新城",1306,13
130681,"涿州市",1306,13
130682,"定州市",1306,13
130683,"安国市",1306,13
130684,"高碑店市",1306,13
130702,"桥东区",1307,13
130703,"桥西区",1307,13
130705,"宣化区",1307,13
130706,"下花园区",1307,13
130708,"万全区",1307,13
130709,"崇礼区",1307,13
130722,"张北县",1307,13
130723,"康保县",1307,13
130724,"沽源县",1307,13
130725,"尚义县",1307,13
130726,"蔚县",1307,13
130727,"阳原县",1307,13
130728,"怀安县",1307,13
130730,"怀来县",1307,13
130731,"涿鹿县",1307,13
130732,"赤城县",1307,13
130771,"张家口经济开发区",1307,13
130772,"张家口市察北管理区",1307,13
130773,"张家口市塞北管理区",1307,13
130802,"双桥区",1308,13
130803,"双滦区",1308,13
130804,"鹰手营子矿区",1308,13
130821,"承德县",1308,13
130822,"兴隆县",1308,13
130824,"滦平县",1308,13
130825,"隆化县",1308,13
130826,"丰宁满族自治县",1308,13
130827,"宽城满族自治县",1308,13
130828,"围场满族蒙古族自治县",1308,13
130871,"承德高新技术产业开发区",1308,13
130881,"平泉市",1308,13
130902,"新华区",1309,13
130903,"运河区",1309,13
130921,"沧县",1309,13
130922,"青县",1309,13
130923,"东光县",1309,13
130924,"海兴县",1309,13
130925,"盐山县",1309,13
130926,"肃宁县",1309,13
130927,"南皮县",1309,13
130928,"吴桥县",1309,13
130929,"献县",1309,13
130930,"孟村回族自治县",1309,13
130971,"河北沧州经济开发区",1309,13
130972,"沧州高新技术产业开发区",1309,13
130973,"沧州渤海新区",1309,13
130981,"泊头市",1309,13
130982,"任丘市",1309,13
130983,"黄骅市",1309,13
130984,"河间市",1309,13
131002,"安次区",1310,13
131003,"广阳区",1310,13
131022,"固安县",1310,13
131023,"永清县",1310,13
131024,"香河县",1310,13
131025,"大城县",1310,13
131026,"文安县",1310,13
131028,"大厂回族自治县",1310,13
131071,"廊坊经济技术开发区",1310,13
131081,"霸州市",1310,13
131082,"三河市",1310,13
131102,"桃城区",1311,13
131103,"冀州区",1311,13
131121,"枣强县",1311,13
131122,"武邑县",1311,13
131123,"武强县",1311,13
131124,"饶阳县",1311,13
131125,"安平县",1311,13
131126,"故城县",1311,13
131127,"景县",1311,13
131128,"阜城县",1311,13
131171,"河北衡水高新技术产业开发区",1311,13
131172,"衡水滨湖新区",1311,13
131182,"深州市",1311,13
140105,"小店区",1401,14
140106,"迎泽区",1401,14
140107,"杏花岭区",1401,14
140108,"尖草坪区",1401,14
140109,"万柏林区",1401,14
140110,"晋源区",1401,14
140121,"清徐县",1401,14
140122,"阳曲县",1401,14
140123,"娄烦县",1401,14
140171,"山西转型综合改革示范区",1401,14
140181,"古交市",1401,14
140212,"新荣区",1402,14
140213,"平城区",1402,14
140214,"云冈区",1402,14
140215,"云州区",1402,14
140221,"阳高县",1402,14
140222,"天镇县",1402,14
140223,"广灵县",1402,14
140224,"灵丘县",1402,14
140225,"浑源县",1402,14
140226,"左云县",1402,14
140271,"山西大同经济开发区",1402,14
140302,"城区",1403,14
140303,"矿区",1403,14
140311,"郊区",1403,14
140321,"平定县",1403,14
140322,"盂县",1403,14
140403,"潞州区",1404,14
140404,"上党区",1404,14
140405,"屯留区",1404,14
140406,"潞城区",1404,14
140423,"襄垣县",1404,14
140425,"平顺县",1404,14
140426,"黎城县",1404,14
140427,"壶关县",1404,14
140428,"长子县",1404,14
140429,"武乡县",1404,14
140430,"沁县",1404,14
140431,"沁源县",1404,14
140502,"城区",1405,14
140521,"沁水县",1405,14
140522,"阳城县",1405,14
140524,"陵川县",1405,14
140525,"泽州县",1405,14
140581,"高平市",1405,14
140602,"朔城区",1406,14
140603,"平鲁区",1406,14
140621,"山阴县",1406,14
140622,"应县",1406,14
140623,"右玉县",1406,14
140671,"山西朔州经济开发区",1406,14
140681,"怀仁市",1406,14
140702,"榆次区",1407,14
140703,"太谷区",1407,14
140721,"榆社县",1407,14
140722,"左权县",1407,14
140723,"和顺县",1407,14
140724,"昔阳县",1407,14
140725,"寿阳县",1407,14
140727,"祁县",1407,14
140728,"平遥县",1407,14
140729,"灵石县",1407,14
140781,"介休市",1407,14
140802,"盐湖区",1408,14
140821,"临猗县",1408,14
140822,"万荣县",1408,14
140823,"闻喜县",1408,14
140824,"稷山县",1408,14
140825,"新绛县",1408,14
140826,"绛县",1408,14
140827,"垣曲县",1408,14
140828,"夏县",1408,14
140829,"平陆县",1408,14
140830,"芮城县",1408,14
140881,"永济市",1408,14
140882,"河津市",1408,14
140902,"忻府区",1409,14
140921,"定襄县",1409,14
140922,"五台县",1409,14
140923,"代县",1409,14
140924,"繁峙县",1409,14
140925,"宁武县",1409,14
140926,"静乐县",1409,14
140927,"神池县",1409,14
140928,"五寨县",1409,14
140929,"岢岚县",1409,14
140930,"河曲县",1409,14
140931,"保德县",1409,14
140932,"偏关县",1409,14
140971,"五台山风景名胜区",1409,14
140981,"原平市",1409,14
141002,"尧都区",1410,14
141021,"曲沃县",1410,14
141022,"翼城县",1410,14
141023,"襄汾县",1410,14
141024,"洪洞县",1410,14
141025,"古县",1410,14
141026,"安泽县",1410,14
141027,"浮山县",1410,14
141028,"吉县",1410,14
141029,"乡宁县",1410,14
141030,"大宁县",1410,14
141031,"隰县",1410,14
141032,"永和县",1410,14
141033,"蒲县",1410,14
141034,"汾西县",1410,14
141081,"侯马市",1410,14
141082,"霍州市",1410,14
141102,"离石区",1411,14
141121,"文水县",1411,14
141122,"交城县",1411,14
141123,"兴县",1411,14
141124,"临县",1411,14
141125,"柳林县",1411,14
141126,"石楼县",1411,14
141127,"岚县",1411,14
141128,"方山县",1411,14
141129,"中阳县",1411,14
141130,"交口县",1411,14
141181,"孝义市",1411,14
141182,"汾阳市",1411,14
150102,"新城区",1501,15
150103,"回民区",1501,15
150104,"玉泉区",1501,15
150105,"赛罕区",1501,15
150121,"土默特左旗",1501,15
150122,"托克托县",1501,15
150123,"和林格尔县",1501,15
150124,"清水河县",1501,15
150125,"武川县",1501,15
150172,"呼和浩特经济技术开发区",1501,15
150202,"东河区",1502,15
150203,"昆都仑区",1502,15
150204,"青山区",1502,15
150205,"石拐区",1502,15
150206,"白云鄂博矿区",1502,15
150207,"九原区",1502,15
150221,"土默特右旗",1502,15
150222,"固阳县",1502,15
150223,"达尔罕茂明安联合旗",1502,15
150271,"包头稀土高新技术产业开发区",1502,15
150302,"海勃湾区",1503,15
150303,"海南区",1503,15
150304,"乌达区",1503,15
150402,"红山区",1504,15
150403,"元宝山区",1504,15
150404,"松山区",1504,15
150421,"阿鲁科尔沁旗",1504,15
150422,"巴林左旗",1504,15
150423,"巴林右旗",1504,15
150424,"林西县",1504,15
150425,"克什克腾旗",1504,15
150426,"翁牛特旗",1504,15
150428,"喀喇沁旗",1504,15
150429,"宁城县",1504,15
150430,"敖汉旗",1504,15
150502,"科尔沁区",1505,15
150521,"科尔沁左翼中旗",1505,15
150522,"科尔沁左翼后旗",1505,15
150523,"开鲁县",1505,15
150524,"库伦旗",1505,15
150525,"奈曼旗",1505,15
150526,"扎鲁特旗",1505,15
150571,"通辽经济技术开发区",1505,15
150581,"霍林郭勒市",1505,15
150602,"东胜区",1506,15
150603,"康巴什区",1506,15
150621,"达拉特旗",1506,15
150622,"准格尔旗",1506,15
150623,"鄂托克前旗",1506,15
150624,"鄂托克旗",1506,15
150625,"杭锦旗",1506,15
150626,"乌审旗",1506,15
150627,"伊金霍洛旗",1506,15
150702,"海拉尔区",1507,15
150703,"扎赉诺尔区",1507,15
150721,"阿荣旗",1507,15
150722,"莫力达瓦达斡尔族自治旗",1507,15
150723,"鄂伦春自治旗",1507,15
150724,"鄂温克族自治旗",1507,15
150725,"陈巴尔虎旗",1507,15
150726,"新巴尔虎左旗",1507,15
150727,"新巴尔虎右旗",1507,15
150781,"满洲里市",1507,15
150782,"牙克石市",1507,15
150783,"扎兰屯市",1507,15
150784,"额尔古纳市",1507,15
150785,"根河市",1507,15
150802,"临河区",1508,15
150821,"五原县",1508,15
150822,"磴口县",1508,15
150823,"乌拉特前旗",1508,15
150824,"乌拉特中旗",1508,15
150825,"乌拉特后旗",1508,15
150826,"杭锦后旗",1508,15
150902,"集宁区",1509,15
150921,"卓资县",1509,15
150922,"化德县",1509,15
150923,"商都县",1509,15
150924,"兴和县",1509,15
150925,"凉城县",1509,15
150926,"察哈尔右翼前旗",1509,15
150927,"察哈尔右翼中旗",1509,15
150928,"察哈尔右翼后旗",1509,15
150929,"四子王旗",1509,15
150981,"丰镇市",1509,15
152201,"乌兰浩特市",1522,15
152202,"阿尔山市",1522,15
152221,"科尔沁右翼前旗",1522,15
152222,"科尔沁右翼中旗",1522,15
152223,"扎赉特旗",1522,15
152224,"突泉县",1522,15
152501,"二连浩特市",1525,15
152502,"锡林浩特市",1525,15
152522,"阿巴嘎旗",1525,15
152523,"苏尼特左旗",1525,15
152524,"苏尼特右旗",1525,15
152525,"东乌珠穆沁旗",1525,15
152526,"西乌珠穆沁旗",1525,15
152527,"太仆寺旗",1525,15
152528,"镶黄旗",1525,15
152529,"正镶白旗",1525,15
152530,"正蓝旗",1525,15
152531,"多伦县",1525,15
152571,"乌拉盖管理区管委会",1525,15
152921,"阿拉善左旗",1529,15
152922,"阿拉善右旗",1529,15
152923,"额济纳旗",1529,15
152971,"内蒙古阿拉善高新技术产业开发区",1529,15
210102,"和平区",2101,21
210103,"沈河区",2101,21
210104,"大东区",2101,21
210105,"皇姑区",2101,21
210106,"铁西区",2101,21
210111,"苏家屯区",2101,21
210112,"浑南区",2101,21
210113,"沈北新区",2101,21
210114,"于洪区",2101,21
210115,"辽中区",2101,21
210123,"康平县",2101,21
210124,"法库县",2101,21
210181,"新民市",2101,21
210202,"中山区",2102,21
210203,"西岗区",2102,21
210204,"沙河口区",2102,21
210211,"甘井子区",2102,21
210212,"旅顺口区",2102,21
210213,"金州区",2102,21
210214,"普兰店区",2102,21
210224,"长海县",2102,21
210281,"瓦房店市",2102,21
210283,"庄河市",2102,21
210302,"铁东区",2103,21
210303,"铁西区",2103,21
210304,"立山区",2103,21
210311,"千山区",2103,21
210321,"台安县",2103,21
210323,"岫岩满族自治县",2103,21
210381,"海城市",2103,21
210402,"新抚区",2104,21
210403,"东洲区",2104,21
210404,"望花区",2104,21
210411,"顺城区",2104,21
210421,"抚顺县",2104,21
210422,"新宾满族自治县",2104,21
210423,"清原满族自治县",2104,21
210502,"平山区",2105,21
210503,"溪湖区",2105,21
210504,"明山区",2105,21
210505,"南芬区",2105,21
210521,"本溪满族自治县",2105,21
210522,"桓仁满族自治县",2105,21
210602,"元宝区",2106,21
210603,"振兴区",2106,21
210604,"振安区",2106,21
210624,"宽甸满族自治县",2106,21
210681,"东港市",2106,21
210682,"凤城市",2106,21
210702,"古塔区",2107,21
210703,"凌河区",2107,21
210711,"太和区",2107,21
210726,"黑山县",2107,21
210727,"义县",2107,21
210781,"凌海市",2107,21
210782,"北镇市",2107,21
210802,"站前区",2108,21
210803,"西市区",2108,21
210804,"鲅鱼圈区",2108,21
210811,"老边区",2108,21
210881,"盖州市",2108,21
210882,"大石桥市",2108,21
210902,"海州区",2109,21
210903,"新邱区",2109,21
210904,"太平区",2109,21
210905,"清河门区",2109,21
210911,"细河区",2109,21
210921,"阜新蒙古族自治县",2109,21
210922,"彰武县",2109,21
211002,"白塔区",2110,21
211003,"文圣区",2110,21
211004,"宏伟区",2110,21
211005,"弓长岭区",2110,21
211011,"太子河区",2110,21
211021,"辽阳县",2110,21
211081,"灯塔市",2110,21
211102,"双台子区",2111,21
211103,"兴隆台区",2111,21
211104,"大洼区",2111,21
211122,"盘山县",2111,21
211202,"银州区",2112,21
211204,"清河区",2112,21
211221,"铁岭县",2112,21
211223,"西丰县",2112,21
211224,"昌图县",2112,21
211281,"调兵山市",2112,21
211282,"开原市",2112,21
211302,"双塔区",2113,21
211303,"龙城区",2113,21
211321,"朝阳县",2113,21
211322,"建平县",2113,21
211324,"喀喇沁左翼蒙古族自治县",2113,21
211381,"北票市",2113,21
211382,"凌源市",2113,21
211402,"连山区",2114,21
211403,"龙港区",2114,21
211404,"南票区",2114,21
211421,"绥中县",2114,21
211422,"建昌县",2114,21
211481,"兴城市",2114,21
220102,"南关区",2201,22
220103,"宽城区",2201,22
220104,"朝阳区",2201,22
220105,"二道区",2201,22
220106,"绿园区",2201,22
220112,"双阳区",2201,22
220113,"九台区",2201,22
220122,"农安县",2201,22
220171,"长春经济技术开发区",2201,22
220172,"长春净月高新技术产业开发区",2201,22
220173,"长春高新技术产业开发区",2201,22
220174,"长春汽车经济技术开发区",2201,22
220182,"榆树市",2201,22
220183,"德惠市",2201,22
220184,"公主岭市",2201,22
220202,"昌邑区",2202,22
220203,"龙潭区",2202,22
220204,"船营区",2202,22
220211,"丰满区",2202,22
220221,"永吉县",2202,22
220271,"吉林经济开发区",2202,22
220272,"吉林高新技术产业开发区",2202,22
220273,"吉林中国新加坡食品区",2202,22
220281,"蛟河市",2202,22
220282,"桦甸市",2202,22
220283,"舒兰市",2202,22
220284,"磐石市",2202,22
220302,"铁西区",2203,22
220303,"铁东区",2203,22
220322,"梨树县",2203,22
220323,"伊通满族自治县",2203,22
220382,"双辽市",2203,22
220402,"龙山区",2204,22
220403,"西安区",2204,22
220421,"东丰县",2204,22
220422,"东辽县",2204,22
220502,"东昌区",2205,22
220503,"二道江区",2205,22
220521,"通化县",2205,22
220523,"辉南县",2205,22
220524,"柳河县",2205,22
220581,"梅河口市",2205,22
220582,"集安市",2205,22
220602,"浑江区",2206,22
220605,"江源区",2206,22
220621,"抚松县",2206,22
220622,"靖宇县",2206,22
220623,"长白朝鲜族自治县",2206,22
220681,"临江市",2206,22
220702,"宁江区",2207,22
220721,"前郭尔罗斯蒙古族自治县",2207,22
220722,"长岭县",2207,22
220723,"乾安县",2207,22
220771,"吉林松原经济开发区",2207,22
220781,"扶余市",2207,22
220802,"洮北区",2208,22
220821,"镇赉县",2208,22
220822,"通榆县",2208,22
220871,"吉林白城经济开发区",2208,22
220881,"洮南市",2208,22
220882,"大安市",2208,22
222401,"延吉市",2224,22
222402,"图们市",2224,22
222403,"敦化市",2224,22
222404,"珲春市",2224,22
222405,"龙井市",2224,22
222406,"和龙市",2224,22
222424,"汪清县",2224,22
222426,"安图县",2224,22
230102,"道里区",2301,23
230103,"南岗区",2301,23
230104,"道外区",2301,23
230108,"平房区",2301,23
230109,"松北区",2301,23
230110,"香坊区",2301,23
230111,"呼兰区",2301,23
230112,"阿城区",2301,23
230113,"双城区",2301,23
230123,"依兰县",2301,23
230124,"方正县",2301,23
230125,"宾县",2301,23
230126,"巴彦县",2301,23
230127,"木兰县",2301,23
230128,"通河县",2301,23
230129,"延寿县",2301,23
230183,"尚志市",2301,23
230184,"五常市",2301,23
230202,"龙沙区",2302,23
230203,"建华区",2302,23
230204,"铁锋区",2302,23
230205,"昂昂溪区",2302,23
230206,"富拉尔基区",2302,23
230207,"碾子山区",2302,23
230208,"梅里斯达斡尔族区",2302,23
230221,"龙江县",2302,23
230223,"依安县",2302,23
230224,"泰来县",2302,23
230225,"甘南县",2302,23
230227,"富裕县",2302,23
230229,"克山县",2302,23
230230,"克东县",2302,23
230231,"拜泉县",2302,23
230281,"讷河市",2302,23
230302,"鸡冠区",2303,23
230303,"恒山区",2303,23
230304,"滴道区",2303,23
230305,"梨树区",2303,23
230306,"城子河区",2303,23
230307,"麻山区",2303,23
230321,"鸡东县",2303,23
230381,"虎林市",2303,23
230382,"密山市",2303,23
230402,"向阳区",2304,23
230403,"工农区",2304,23
230404,"南山区",2304,23
230405,"兴安区",2304,23
230406,"东山区",2304,23
230407,"兴山区",2304,23
230421,"萝北县",2304,23
230422,"绥滨县",2304,23
230502,"尖山区",2305,23
230503,"岭东区",2305,23
230505,"四方台区",2305,23
230506,"宝山区",2305,23
230521,"集贤县",2305,23
230522,"友谊县",2305,23
230523,"宝清县",2305,23
230524,"饶河县",2305,23
230602,"萨尔图区",2306,23
230603,"龙凤区",2306,23
230604,"让胡路区",2306,23
230605,"红岗区",2306,23
230606,"大同区",2306,23
230621,"肇州县",2306,23
230622,"肇源县",2306,23
230623,"林甸县",2306,23
230624,"杜尔伯特蒙古族自治县",2306,23
230671,"大庆高新技术产业开发区",2306,23
230717,"伊美区",2307,23
230718,"乌翠区",2307,23
230719,"友好区",2307,23
230722,"嘉荫县",2307,23
230723,"汤旺县",2307,23
230724,"丰林县",2307,23
230725,"大箐山县",2307,23
230726,"南岔县",2307,23
230751,"金林区",2307,23
230781,"铁力市",2307,23
230803,"向阳区",2308,23
230804,"前进区",2308,23
230805,"东风区",2308,23
230811,"郊区",2308,23
230822,"桦南县",2308,23
230826,"桦川县",2308,23
230828,"汤原县",2308,23
230881,"同江市",2308,23
230882,"富锦市",2308,23
230883,"抚远市",2308,23
230902,"新兴区",2309,23
230903,"桃山区",2309,23
230904,"茄子河区",2309,23
230921,"勃利县",2309,23
231002,"东安区",2310,23
231003,"阳明区",2310,23
231004,"爱民区",2310,23
231005,"西安区",2310,23
231025,"林口县",2310,23
231081,"绥芬河市",2310,23
231083,"海林市",2310,23
231084,"宁安市",2310,23
231085,"穆棱市",2310,23
231086,"东宁市",2310,23
231102,"爱辉区",2311,23
231123,"逊克县",2311,23
231124,"孙吴县",2311,23
231181,"北安市",2311,23
231182,"五大连池市",2311,23
231183,"嫩江市",2311,23
231202,"北林区",2312,23
231221,"望奎县",2312,23
231222,"兰西县",2312,23
231223,"青冈县",2312,23
231224,"庆安县",2312,23
231225,"明水县",2312,23
231226,"绥棱县",2312,23
231281,"安达市",2312,23
231282,"肇东市",2312,23
231283,"海伦市",2312,23
232701,"漠河市",2327,23
232721,"呼玛县",2327,23
232722,"塔河县",2327,23
232761,"加格达奇区",2327,23
232762,"松岭区",2327,23
232763,"新林区",2327,23
232764,"呼中区",2327,23
310101,"黄浦区",3101,31
310104,"徐汇区",3101,31
310105,"长宁区",3101,31
310106,"静安区",3101,31
310107,"普陀区",3101,31
310109,"虹口区",3101,31
310110,"杨浦区",3101,31
310112,"闵行区",3101,31
310113,"宝山区",3101,31
310114,"嘉定区",3101,31
310115,"浦东新区",3101,31
310116,"金山区",3101,31
310117,"松江区",3101,31
310118,"青浦区",3101,31
310120,"奉贤区",3101,31
310151,"崇明区",3101,31
320102,"玄武区",3201,32
320104,"秦淮区",3201,32
320105,"建邺区",3201,32
320106,"鼓楼区",3201,32
320111,"浦口区",3201,32
320113,"栖霞区",3201,32
320114,"雨花台区",3201,32
320115,"江宁区",3201,32
320116,"六合区",3201,32
320117,"溧水区",3201,32
320118,"高淳区",3201,32
320205,"锡山区",3202,32
320206,"惠山区",3202,32
320211,"滨湖区",3202,32
320213,"梁溪区",3202,32
320214,"新吴区",3202,32
320281,"江阴市",3202,32
320282,"宜兴市",3202,32
320302,"鼓楼区",3203,32
320303,"云龙区",3203,32
320305,"贾汪区",3203,32
320311,"泉山区",3203,32
320312,"铜山区",3203,32
320321,"丰县",3203,32
320322,"沛县",3203,32
320324,"睢宁县",3203,32
320371,"徐州经济技术开发区",3203,32
320381,"新沂市",3203,32
320382,"邳州市",3203,32
320402,"天宁区",3204,32
320404,"钟楼区",3204,32
320411,"新北区",3204,32
320412,"武进区",3204,32
320413,"金坛区",3204,32
320481,"溧阳市",3204,32
320505,"虎丘区",3205,32
320506,"吴中区",3205,32
320507,"相城区",3205,32
320508,"姑苏区",3205,32
320509,"吴江区",3205,32
320576,"苏州工业园区",3205,32
320581,"常熟市",3205,32
320582,"张家港市",3205,32
320583,"昆山市",3205,32
320585,"太仓市",3205,32
320612,"通州区",3206,32
320613,"崇川区",3206,32
320614,"海门区",3206,32
320623,"如东县",3206,32
320671,"南通经济技术开发区",3206,32
320681,"启东市",3206,32
320682,"如皋市",3206,32
320685,"海安市",3206,32
320703,"连云区",3207,32
320706,"海州区",3207,32
320707,"赣榆区",3207,32
320722,"东海县",3207,32
320723,"灌云县",3207,32
320724,"灌南县",3207,32
320771,"连云港经济技术开发区",3207,32
320803,"淮安区",3208,32
320804,"淮阴区",3208,32
320812,"清江浦区",3208,32
320813,"洪泽区",3208,32
320826,"涟水县",3208,32
320830,"盱眙县",3208,32
320831,"金湖县",3208,32
320871,"淮安经济技术开发区",3208,32
320902,"亭湖区",3209,32
320903,"盐都区",3209,32
320904,"大丰区",3209,32
320921,"响水县",3209,32
320922,"滨海县",3209,32
320923,"阜宁县",3209,32
320924,"射阳县",3209,32
320925,"建湖县",3209,32
320971,"盐城经济技术开发区",3209,32
320981,"东台市",3209,32
321002,"广陵区",3210,32
321003,"邗江区",3210,32
321012,"江都区",3210,32
321023,"宝应县",3210,32
321071,"扬州经济技术开发区",3210,32
321081,"仪征市",3210,32
321084,"高邮市",3210,32
321102,"京口区",3211,32
321111,"润州区",3211,32
321112,"丹徒区",3211,32
321171,"镇江新区",3211,32
321181,"丹阳市",3211,32
321182,"扬中市",3211,32
321183,"句容市",3211,32
321202,"海陵区",3212,32
321203,"高港区",3212,32
321204,"姜堰区",3212,32
321281,"兴化市",3212,32
321282,"靖江市",3212,32
321283,"泰兴市",3212,32
321302,"宿城区",3213,32
321311,"宿豫区",3213,32
321322,"沭阳县",3213,32
321323,"泗阳县",3213,32
321324,"泗洪县",3213,32
321371,"宿迁经济技术开发区",3213,32
330102,"上城区",3301,33
330105,"拱墅区",3301,33
330106,"西湖区",3301,33
330108,"滨江区",3301,33
330109,"萧山区",3301,33
330110,"余杭区",3301,33
330111,"富阳区",3301,33
330112,"临安区",3301,33
330113,"临平区",3301,33
330114,"钱塘区",3301,33
330122,"桐庐县",3301,33
330127,"淳安县",3301,33
330182,"建德市",3301,33
330203,"海曙区",3302,33
330205,"江北区",3302,33
330206,"北仑区",3302,33
330211,"镇海区",3302,33
330212,"鄞州区",3302,33
330213,"奉化区",3302,33
330225,"象山县",3302,33
330226,"宁海县",3302,33
330281,"余姚市",3302,33
330282,"慈溪市",3302,33
330302,"鹿城区",3303,33
330303,"龙湾区",3303,33
330304,"瓯海区",3303,33
330305,"洞头区",3303,33
330324,"永嘉县",3303,33
330326,"平阳县",3303,33
330327,"苍南县",3303,33
330328,"文成县",3303,33
330329,"泰顺县",3303,33
330381,"瑞安市",3303,33
330382,"乐清市",3303,33
330383,"龙港市",3303,33
330402,"南湖区",3304,33
330411,"秀洲区",3304,33
330421,"嘉善县",3304,33
330424,"海盐县",3304,33
330481,"海宁市",3304,33
330482,"平湖市",3304,33
330483,"桐乡市",3304,33
330502,"吴兴区",3305,33
330503,"南浔区",3305,33
330521,"德清县",3305,33
330522,"长兴县",3305,33
330523,"安吉县",3305,33
330602,"越城区",3306,33
330603,"柯桥区",3306,33
330604,"上虞区",3306,33
330624,"新昌县",3306,33
330681,"诸暨市",3306,33
330683,"嵊州市",3306,33
330702,"婺城区",3307,33
330703,"金东区",3307,33
330723,"武义县",3307,33
330726,"浦江县",3307,33
330727,"磐安县",3307,33
330781,"兰溪市",3307,33
330782,"义乌市",3307,33
330783,"东阳市",3307,33
330784,"永康市",3307,33
330802,"柯城区",3308,33
330803,"衢江区",3308,33
330822,"常山县",3308,33
330824,"开化县",3308,33
330825,"龙游县",3308,33
330881,"江山市",3308,33
330902,"定海区",3309,33
330903,"普陀区",3309,33
330921,"岱山县",3309,33
330922,"嵊泗县",3309,33
331002,"椒江区",3310,33
331003,"黄岩区",3310,33
331004,"路桥区",3310,33
331022,"三门县",3310,33
331023,"天台县",3310,33
331024,"仙居县",3310,33
331081,"温岭市",3310,33
331082,"临海市",3310,33
331083,"玉环市",3310,33
331102,"莲都区",3311,33
331121,"青田县",3311,33
331122,"缙云县",3311,33
331123,"遂昌县",3311,33
331124,"松阳县",3311,33
331125,"云和县",3311,33
331126,"庆元县",3311,33
331127,"景宁畲族自治县",3311,33
331181,"龙泉市",3311,33
340102,"瑶海区",3401,34
340103,"庐阳区",3401,34
340104,"蜀山区",3401,34
340111,"包河区",3401,34
340121,"长丰县",3401,34
340122,"肥东县",3401,34
340123,"肥西县",3401,34
340124,"庐江县",3401,34
340176,"合肥高新技术产业开发区",3401,34
340177,"合肥经济技术开发区",3401,34
340178,"合肥新站高新技术产业开发区",3401,34
340181,"巢湖市",3401,34
340202,"镜湖区",3402,34
340207,"鸠江区",3402,34
340209,"弋江区",3402,34
340210,"湾沚区",3402,34
340212,"繁昌区",3402,34
340223,"南陵县",3402,34
340271,"芜湖经济技术开发区",3402,34
340272,"安徽芜湖三山经济开发区",3402,34
340281,"无为市",3402,34
340302,"龙子湖区",3403,34
340303,"蚌山区",3403,34
340304,"禹会区",3403,34
340311,"淮上区",3403,34
340321,"怀远县",3403,34
340322,"五河县",3403,34
340323,"固镇县",3403,34
340371,"蚌埠市高新技术开发区",3403,34
340372,"蚌埠市经济开发区",3403,34
340402,"大通区",3404,34
340403,"田家庵区",3404,34
340404,"谢家集区",3404,34
340405,"八公山区",3404,34
340406,"潘集区",3404,34
340421,"凤台县",3404,34
340422,"寿县",3404,34
340503,"花山区",3405,34
340504,"雨山区",3405,34
340506,"博望区",3405,34
340521,"当涂县",3405,34
340522,"含山县",3405,34
340523,"和县",3405,34
340602,"杜集区",3406,34
340603,"相山区",3406,34
340604,"烈山区",3406,34
340621,"濉溪县",3406,34
340705,"铜官区",3407,34
340706,"义安区",3407,34
340711,"郊区",3407,34
340722,"枞阳县",3407,34
340802,"迎江区",3408,34
340803,"大观区",3408,34
340811,"宜秀区",3408,34
340822,"怀宁县",3408,34
340825,"太湖县",3408,34
340826,"宿松县",3408,34
340827,"望江县",3408,34
340828,"岳西县",3408,34
340871,"安徽安庆经济开发区",3408,34
340881,"桐城市",3408,34
340882,"潜山市",3408,34
341002,"屯溪区",3410,34
341003,"黄山区",3410,34
341004,"徽州区",3410,34
341021,"歙县",3410,34
341022,"休宁县",3410,34
341023,"黟县",3410,34
341024,"祁门县",3410,34
341102,"琅琊区",3411,34
341103,"南谯区",3411,34
341122,"来安县",3411,34
341124,"全椒县",3411,34
341125,"定远县",3411,34
341126,"凤阳县",3411,34
341171,"中新苏滁高新技术产业开发区",3411,34
341172,"滁州经济技术开发区",3411,34
341181,"天长市",3411,34
341182,"明光市",3411,34
341202,"颍州区",3412,34
341203,"颍东区",3412,34
341204,"颍泉区",3412,34
341221,"临泉县",3412,34
341222,"太和县",3412,34
341225,"阜南县",3412,34
341226,"颍上县",3412,34
341271,"阜阳合肥现代产业园区",3412,34
341272,"阜阳经济技术开发区",3412,34
341282,"界首市",3412,34
341302,"埇桥区",3413,34
341321,"砀山县",3413,34
341322,"萧县",3413,34
341323,"灵璧县",3413,34
341324,"泗县",3413,34
341371,"宿州马鞍山现代产业园区",3413,34
341372,"宿州经济技术开发区",3413,34
341502,"金安区",3415,34
341503,"裕安区",3415,34
341504,"叶集区",3415,34
341522,"霍邱县",3415,34
341523,"舒城县",3415,34
341524,"金寨县",3415,34
341525,"霍山县",3415,34
341602,"谯城区",3416,34
341621,"涡阳县",3416,34
341622,"蒙城县",3416,34
341623,"利辛县",3416,34
341702,"贵池区",3417,34
341721,"东至县",3417,34
341722,"石台县",3417,34
341723,"青阳县",3417,34
341802,"宣州区",3418,34
341821,"郎溪县",3418,34
341823,"泾县",3418,34
341824,"绩溪县",3418,34
341825,"旌德县",3418,34
341871,"宣城市经济开发区",3418,34
341881,"宁国市",3418,34
341882,"广德市",3418,34
350102,"鼓楼区",3501,35
350103,"台江区",3501,35
350104,"仓山区",3501,35
350105,"马尾区",3501,35
350111,"晋安区",3501,35
350112,"长乐区",3501,35
350121,"闽侯县",3501,35
350122,"连江县",3501,35
350123,"罗源县",3501,35
350124,"闽清县",3501,35
350125,"永泰县",3501,35
350128,"平潭县",3501,35
350181,"福清市",3501,35
350203,"思明区",3502,35
350205,"海沧区",3502,35
350206,"湖里区",3502,35
350211,"集美区",3502,35
350212,"同安区",3502,35
350213,"翔安区",3502,35
350302,"城厢区",3503,35
350303,"涵江区",3503,35
350304,"荔城区",3503,35
350305,"秀屿区",3503,35
350322,"仙游县",3503,35
350404,"三元区",3504,35
350405,"沙县区",3504,35
350421,"明溪县",3504,35
350423,"清流县",3504,35
350424,"宁化县",3504,35
350425,"大田县",3504,35
350426,"尤溪县",3504,35
350428,"将乐县",3504,35
350429,"泰宁县",3504,35
350430,"建宁县",3504,35
350481,"永安市",3504,35
350502,"鲤城区",3505,35
350503,"丰泽区",3505,35
350504,"洛江区",3505,35
350505,"泉港区",3505,35
350521,"惠安县",3505,35
350524,"安溪县",3505,35
350525,"永春县",3505,35
350526,"德化县",3505,35
350527,"金门县",3505,35
350581,"石狮市",3505,35
350582,"晋江市",3505,35
350583,"南安市",3505,35
350602,"芗城区",3506,35
350603,"龙文区",3506,35
350604,"龙海区",3506,35
350605,"长泰区",3506,35
350622,"云霄县",3506,35
350623,"漳浦县",3506,35
350624,"诏安县",3506,35
350626,"东山县",3506,35
350627,"南靖县",3506,35
350628,"平和县",3506,35
350629,"华安县",3506,35
350702,"延平区",3507,35
350703,"建阳区",3507,35
350721,"顺昌县",3507,35
350722,"浦城县",3507,35
350723,"光泽县",3507,35
350724,"松溪县",3507,35
350725,"政和县",3507,35
350781,"邵武市",3507,35
350782,"武夷山市",3507,35
350783,"建瓯市",3507,35
350802,"新罗区",3508,35
350803,"永定区",3508,35
350821,"长汀县",3508,35
350823,"上杭县",3508,35
350824,"武平县",3508,35
350825,"连城县",3508,35
350881,"漳平市",3508,35
350902,"蕉城区",3509,35
350921,"霞浦县",3509,35
350922,"古田县",3509,35
350923,"屏南县",3509,35
350924,"寿宁县",3509,35
350925,"周宁县",3509,35
350926,"柘荣县",3509,35
350981,"福安市",3509,35
350982,"福鼎市",3509,35
360102,"东湖区",3601,36
360103,"西湖区",3601,36
360104,"青云谱区",3601,36
360111,"青山湖区",3601,36
360112,"新建区",3601,36
360113,"红谷滩区",3601,36
360121,"南昌县",3601,36
360123,"安义县",3601,36
360124,"进贤县",3601,36
360202,"昌江区",3602,36
360203,"珠山区",3602,36
360222,"浮梁县",3602,36
360281,"乐平市",3602,36
360302,"安源区",3603,36
360313,"湘东区",3603,36
360321,"莲花县",3603,36
360322,"上栗县",3603,36
360323,"芦溪县",3603,36
360402,"濂溪区",3604,36
360403,"浔阳区",3604,36
360404,"柴桑区",3604,36
360423,"武宁县",3604,36
360424,"修水县",3604,36
360425,"永修县",3604,36
360426,"德安县",3604,36
360428,"都昌县",3604,36
360429,"湖口县",3604,36
360430,"彭泽县",3604,36
360481,"瑞昌市",3604,36
360482,"共青城市",3604,36
360483,"庐山市",3604,36
360502,"渝水区",3605,36
360521,"分宜县",3605,36
360602,"月湖区",3606,36
360603,"余江区",3606,36
360681,"贵溪市",3606,36
360702,"章贡区",3607,36
360703,"南康区",3607,36
360704,"赣县区",3607,36
360722,"信丰县",3607,36
360723,"大余县",3607,36
360724,"上犹县",3607,36
360725,"崇义县",3607,36
360726,"安远县",3607,36
360728,"定南县",3607,36
360729,"全南县",3607,36
360730,"宁都县",3607,36
360731,"于都县",3607,36
360732,"兴国县",3607,36
360733,"会昌县",3607,36
360734,"寻乌县",3607,36
360735,"石城县",3607,36
360781,"瑞金市",3607,36
360783,"龙南市",3607,36
360802,"吉州区",3608,36
360803,"青原区",3608,36
360821,"吉安县",3608,36
360822,"吉水县",3608,36
360823,"峡江县",3608,36
360824,"新干县",3608,36
360825,"永丰县",3608,36
360826,"泰和县",3608,36
360827,"遂川县",3608,36
360828,"万安县",3608,36
360829,"安福县",3608,36
360830,"永新县",3608,36
360881,"井冈山市",3608,36
360902,"袁州区",3609,36
360921,"奉新县",3609,36
360922,"万载县",3609,36
360923,"上高县",3609,36
360924,"宜丰县",3609,36
360925,"靖安县",3609,36
360926,"铜鼓县",3609,36
360981,"丰城市",3609,36
360982,"樟树市",3609,36
360983,"高安市",3609,36
361002,"临川区",3610,36
361003,"东乡区",3610,36
361021,"南城县",3610,36
361022,"黎川县",3610,36
361023,"南丰县",3610,36
361024,"崇仁县",3610,36
361025,"乐安县",3610,36
361026,"宜黄县",3610,36
361027,"金溪县",3610,36
361028,"资溪县",3610,36
361030,"广昌县",3610,36
361102,"信州区",3611,36
361103,"广丰区",3611,36
361104,"广信区",3611,36
361123,"玉山县",3611,36
361124,"铅山县",3611,36
361125,"横峰县",3611,36
361126,"弋阳县",3611,36
361127,"余干县",3611,36
361128,"鄱阳县",3611,36
361129,"万年县",3611,36
361130,"婺源县",3611,36
361181,"德兴市",3611,36
370102,"历下区",3701,37
370103,"市中区",3701,37
370104,"槐荫区",3701,37
370105,"天桥区",3701,37
370112,"历城区",3701,37
370113,"长清区",3701,37
370114,"章丘区",3701,37
370115,"济阳区",3701,37
370116,"莱芜区",3701,37
370117,"钢城区",3701,37
370124,"平阴县",3701,37
370126,"商河县",3701,37
370176,"济南高新技术产业开发区",3701,37
370202,"市南区",3702,37
370203,"市北区",3702,37
370211,"黄岛区",3702,37
370212,"崂山区",3702,37
370213,"李沧区",3702,37
370214,"城阳区",3702,37
370215,"即墨区",3702,37
370281,"胶州市",3702,37
370283,"平度市",3702,37
370285,"莱西市",3702,37
370302,"淄川区",3703,37
370303,"张店区",3703,37
370304,"博山区",3703,37
370305,"临淄区",3703,37
370306,"周村区",3703,37
370321,"桓台县",3703,37
370322,"高青县",3703,37
370323,"沂源县",3703,37
370402,"市中区",3704,37
370403,"薛城区",3704,37
370404,"峄城区",3704,37
370405,"台儿庄区",3704,37
370406,"山亭区",3704,37
370481,"滕州市",3704,37
370502,"东营区",3705,37
370503,"河口区",3705,37
370505,"垦利区",3705,37
370522,"利津县",3705,37
370523,"广饶县",3705,37
370571,"东营经济技术开发区",3705,37
370572,"东营港经济开发区",3705,37
370602,"芝罘区",3706,37
370611,"福山区",3706,37
370612,"牟平区",3706,37
370613,"莱山区",3706,37
370614,"蓬莱区",3706,37
370671,"烟台高新技术产业开发区",3706,37
370676,"烟台经济技术开发区",3706,37
370681,"龙口市",3706,37
370682,"莱阳市",3706,37
370683,"莱州市",3706,37
370685,"招远市",3706,37
370686,"栖霞市",3706,37
370687,"海阳市",3706,37
370702,"潍城区",3707,37
370703,"寒亭区",3707,37
370704,"坊子区",3707,37
370705,"奎文区",3707,37
370724,"临朐县",3707,37
370725,"昌乐县",3707,37
370772,"潍坊滨海经济技术开发区",3707,37
370781,"青州市",3707,37
370782,"诸城市",3707,37
370783,"寿光市",3707,37
370784,"安丘市",3707,37
370785,"高密市",3707,37
370786,"昌邑市",3707,37
370811,"任城区",3708,37
370812,"兖州区",3708,37
370826,"微山县",3708,37
370827,"鱼台县",3708,37
370828,"金乡县",3708,37
370829,"嘉祥县",3708,37
370830,"汶上县",3708,37
370831,"泗水县",3708,37
370832,"梁山县",3708,37
370871,"济宁高新技术产业开发区",3708,37
370881,"曲阜市",3708,37
370883,"邹城市",3708,37
370902,"泰山区",3709,37
370911,"岱岳区",3709,37
370921,"宁阳县",3709,37
370923,"东平县",3709,37
370982,"新泰市",3709,37
370983,"肥城市",3709,37
371002,"环翠区",3710,37
371003,"文登区",3710,37
371071,"威海火炬高技术产业开发区",3710,37
371072,"威海经济技术开发区",3710,37
371073,"威海临港经济技术开发区",3710,37
371082,"荣成市",3710,37
371083,"乳山市",3710,37
371102,"东港区",3711,37
371103,"岚山区",3711,37
371121,"五莲县",3711,37
371122,"莒县",3711,37
371171,"日照经济技术开发区",3711,37
371302,"兰山区",3713,37
371311,"罗庄区",3713,37
371312,"河东区",3713,37
371321,"沂南县",3713,37
371322,"郯城县",3713,37
371323,"沂水县",3713,37
371324,"兰陵县",3713,37
371325,"费县",3713,37
371326,"平邑县",3713,37
371327,"莒南县",3713,37
371328,"蒙阴县",3713,37
371329,"临沭县",3713,37
371371,"临沂高新技术产业开发区",3713,37
371402,"德城区",3714,37
371403,"陵城区",3714,37
371422,"宁津县",3714,37
371423,"庆云县",3714,37
371424,"临邑县",3714,37
371425,"齐河县",3714,37
371426,"平原县",3714,37
371427,"夏津县",3714,37
371428,"武城县",3714,37
371471,"德州天衢新区",3714,37
371481,"乐陵市",3714,37
371482,"禹城市",3714,37
371502,"东昌府区",3715,37
371503,"茌平区",3715,37
371521,"阳谷县",3715,37
371522,"莘县",3715,37
371524,"东阿县",3715,37
371525,"冠县",3715,37
371526,"高唐县",3715,37
371581,"临清市",3715,37
371602,"滨城区",3716,37
371603,"沾化区",3716,37
371621,"惠民县",3716,37
371622,"阳信县",3716,37
371623,"无棣县",3716,37
371625,"博兴县",3716,37
371681,"邹平市",3716,37
371702,"牡丹区",3717,37
371703,"定陶区",3717,37
371721,"曹县",3717,37
371722,"单县",3717,37
371723,"成武县",3717,37
371724,"巨野县",3717,37
371725,"郓城县",3717,37
371726,"鄄城县",3717,37
371728,"东明县",3717,37
371771,"菏泽经济技术开发区",3717,37
371772,"菏泽高新技术开发区",3717,37
410102,"中原区",4101,41
410103,"二七区",4101,41
410104,"管城回族区",4101,41
410105,"金水区",4101,41
410106,"上街区",4101,41
410108,"惠济区",4101,41
410122,"中牟县",4101,41
410171,"郑州经济技术开发区",4101,41
410172,"郑州高新技术产业开发区",4101,41
410173,"郑州航空港经济综合实验区",4101,41
410181,"巩义市",4101,41
410182,"荥阳市",4101,41
410183,"新密市",4101,41
410184,"新郑市",4101,41
410185,"登封市",4101,41
410202,"龙亭区",4102,41
410203,"顺河回族区",4102,41
410204,"鼓楼区",4102,41
410205,"禹王台区",4102,41
410212,"祥符区",4102,41
410221,"杞县",4102,41
410222,"通许县",4102,41
410223,"尉氏县",4102,41
410225,"兰考县",4102,41
410302,"老城区",4103,41
410303,"西工区",4103,41
410304,"瀍河回族区",4103,41
410305,"涧西区",4103,41
410307,"偃师区",4103,41
410308,"孟津区",4103,41
410311,"洛龙区",4103,41
410323,"新安县",4103,41
410324,"栾川县",4103,41
410325,"嵩县",4103,41
410326,"汝阳县",4103,41
410327,"宜阳县",4103,41
410328,"洛宁县",4103,41
410329,"伊川县",4103,41
410371,"洛阳高新技术产业开发区",4103,41
410402,"新华区",4104,41
410403,"卫东区",4104,41
410404,"石龙区",4104,41
410411,"湛河区",4104,41
410421,"宝丰县",4104,41
410422,"叶县",4104,41
410423,"鲁山县",4104,41
410425,"郏县",4104,41
410471,"平顶山高新技术产业开发区",4104,41
410472,"平顶山市城乡一体化示范区",4104,41
410481,"舞钢市",4104,41
410482,"汝州市",4104,41
410502,"文峰区",4105,41
410503,"北关区",4105,41
410505,"殷都区",4105,41
410506,"龙安区",4105,41
410522,"安阳县",4105,41
410523,"汤阴县",4105,41
410526,"滑县",4105,41
410527,"内黄县",4105,41
410571,"安阳高新技术产业开发区",4105,41
410581,"林州市",4105,41
410602,"鹤山区",4106,41
410603,"山城区",4106,41
410611,"淇滨区",4106,41
410621,"浚县",4106,41
410622,"淇县",4106,41
410671,"鹤壁经济技术开发区",4106,41
410702,"红旗区",4107,41
410703,"卫滨区",4107,41
410704,"凤泉区",4107,41
410711,"牧野区",4107,41
410721,"新乡县",4107,41
410724,"获嘉县",4107,41
410725,"原阳县",4107,41
410726,"延津县",4107,41
410727,"封丘县",4107,41
410771,"新乡高新技术产业开发区",4107,41
410772,"新乡经济技术开发区",4107,41
410773,"新乡市平原城乡一体化示范区",4107,41
410781,"卫辉市",4107,41
410782,"辉县市",4107,41
410783,"长垣市",4107,41
410802,"解放区",4108,41
410803,"中站区",4108,41
410804,"马村区",4108,41
410811,"山阳区",4108,41
410821,"修武县",4108,41
410822,"博爱县",4108,41
410823,"武陟县",4108,41
410825,"温县",4108,41
410871,"焦作城乡一体化示范区",4108,41
410882,"沁阳市",4108,41
410883,"孟州市",4108,41
410902,"华龙区",4109,41
410922,"清丰县",4109,41
410923,"南乐县",4109,41
410926,"范县",4109,41
410927,"台前县",4109,41
410928,"濮阳县",4109,41
410971,"河南濮阳工业园区",4109,41
410972,"濮阳经济技术开发区",4109,41
411002,"魏都区",4110,41
411003,"建安区",4110,41
411024,"鄢陵县",4110,41
411025,"襄城县",4110,41
411071,"许昌经济技术开发区",4110,41
411081,"禹州市",4110,41
411082,"长葛市",4110,41
411102,"源汇区",4111,41
411103,"郾城区",4111,41
411104,"召陵区",4111,41
411121,"舞阳县",4111,41
411122,"临颍县",4111,41
411171,"漯河经济技术开发区",4111,41
411202,"湖滨区",4112,41
411203,"陕州区",4112,41
411221,"渑池县",4112,41
411224,"卢氏县",4112,41
411271,"河南三门峡经济开发区",4112,41
411281,"义马市",4112,41
411282,"灵宝市",4112,41
411302,"宛城区",4113,41
411303,"卧龙区",4113,41
411321,"南召县",4113,41
411322,"方城县",4113,41
411323,"西峡县",4113,41
411324,"镇平县",4113,41
411325,"内乡县",4113,41
411326,"淅川县",4113,41
411327,"社旗县",4113,41
411328,"唐河县",4113,41
411329,"新野县",4113,41
411330,"桐柏县",4113,41
411371,"南阳高新技术产业开发区",4113,41
411372,"南阳市城乡一体化示范区",4113,41
411381,"邓州市",4113,41
411402,"梁园区",4114,41
411403,"睢阳区",4114,41
411421,"民权县",4114,41
411422,"睢县",4114,41
411423,"宁陵县",4114,41
411424,"柘城县",4114,41
411425,"虞城县",4114,41
411426,"夏邑县",4114,41
411471,"豫东综合物流产业聚集区",4114,41
411472,"河南商丘经济开发区",4114,41
411481,"永城市",4114,41
411502,"浉河区",4115,41
411503,"平桥区",4115,41
411521,"罗山县",4115,41
411522,"光山县",4115,41
411523,"新县",4115,41
411524,"商城县",4115,41
411525,"固始县",4115,41
411526,"潢川县",4115,41
411527,"淮滨县",4115,41
411528,"息县",4115,41
411571,"信阳高新技术产业开发区",4115,41
411602,"川汇区",4116,41
411603,"淮阳区",4116,41
411621,"扶沟县",4116,41
411622,"西华县",4116,41
411623,"商水县",4116,41
411624,"沈丘县",4116,41
411625,"郸城县",4116,41
411627,"太康县",4116,41
411628,"鹿邑县",4116,41
411671,"周口临港开发区",4116,41
411681,"项城市",4116,41
411702,"驿城区",4117,41
411721,"西平县",4117,41
411722,"上蔡县",4117,41
411723,"平舆县",4117,41
411724,"正阳县",4117,41
411725,"确山县",4117,41
411726,"泌阳县",4117,41
411727,"汝南县",4117,41
411728,"遂平县",4117,41
411729,"新蔡县",4117,41
411771,"河南驻马店经济开发区",4117,41
419001,"济源市",4190,41
420102,"江岸区",4201,42
420103,"江汉区",4201,42
420104,"硚口区",4201,42
420105,"汉阳区",4201,42
420106,"武昌区",4201,42
420107,"青山区",4201,42
420111,"洪山区",4201,42
420112,"东西湖区",4201,42
420113,"汉南区",4201,42
420114,"蔡甸区",4201,42
420115,"江夏区",4201,42
420116,"黄陂区",4201,42
420117,"新洲区",4201,42
420202,"黄石港区",4202,42
420203,"西塞山区",4202,42
420204,"下陆区",4202,42
420205,"铁山区",4202,42
420222,"阳新县",4202,42
420281,"大冶市",4202,42
420302,"茅箭区",4203,42
420303,"张湾区",4203,42
420304,"郧阳区",4203,42
420322,"郧西县",4203,42
420323,"竹山县",4203,42
420324,"竹溪县",4203,42
420325,"房县",4203,42
420381,"丹江口市",4203,42
420502,"西陵区",4205,42
420503,"伍家岗区",4205,42
420504,"点军区",4205,42
420505,"猇亭区",4205,42
420506,"夷陵区",4205,42
420525,"远安县",4205,42
420526,"兴山县",4205,42
420527,"秭归县",4205,42
420528,"长阳土家族自治县",4205,42
420529,"五峰土家族自治县",4205,42
420581,"宜都市",4205,42
420582,"当阳市",4205,42
420583,"枝江市",4205,42
420602,"襄城区",4206,42
420606,"樊城区",4206,42
420607,"襄州区",4206,42
420624,"南漳县",4206,42
420625,"谷城县",4206,42
420626,"保康县",4206,42
420682,"老河口市",4206,42
420683,"枣阳市",4206,42
420684,"宜城市",4206,42
420702,"梁子湖区",4207,42
420703,"华容区",4207,42
420704,"鄂城区",4207,42
420802,"东宝区",4208,42
420804,"掇刀区",4208,42
420822,"沙洋县",4208,42
420881,"钟祥市",4208,42
420882,"京山市",4208,42
420902,"孝南区",4209,42
420921,"孝昌县",4209,42
420922,"大悟县",4209,42
420923,"云梦县",4209,42
420981,"应城市",4209,42
420982,"安陆市",4209,42
420984,"汉川市",4209,42
421002,"沙市区",4210,42
421003,"荆州区",4210,42
421022,"公安县",4210,42
421024,"江陵县",4210,42
421071,"荆州经济技术开发区",4210,42
421081,"石首市",4210,42
421083,"洪湖市",4210,42
421087,"松滋市",4210,42
421088,"监利市",4210,42
421102,"黄州区",4211,42
421121,"团风县",4211,42
421122,"红安县",4211,42
421123,"罗田县",4211,42
421124,"英山县",4211,42
421125,"浠水县",4211,42
421126,"蕲春县",4211,42
421127,"黄梅县",4211,42
421171,"龙感湖管理区",4211,42
421181,"麻城市",4211,42
421182,"武穴市",4211,42
421202,"咸安区",4212,42
421221,"嘉鱼县",4212,42
421222,"通城县",4212,42
421223,"崇阳县",4212,42
421224,"通山县",4212,42
421281,"赤壁市",4212,42
421303,"曾都区",4213,42
421321,"随县",4213,42
421381,"广水市",4213,42
422801,"恩施市",4228,42
422802,"利川市",4228,42
422822,"建始县",4228,42
422823,"巴东县",4228,42
422825,"宣恩县",4228,42
422826,"咸丰县",4228,42
422827,"来凤县",4228,42
422828,"鹤峰县",4228,42
429004,"仙桃市",4290,42
429005,"潜江市",4290,42
429006,"天门市",4290,42
429021,"神农架林区",4290,42
430102,"芙蓉区",4301,43
430103,"天心区",4301,43
430104,"岳麓区",4301,43
430105,"开福区",4301,43
430111,"雨花区",4301,43
430112,"望城区",4301,43
430121,"长沙县",4301,43
430181,"浏阳市",4301,43
430182,"宁乡市",4301,43
430202,"荷塘区",4302,43
430203,"芦淞区",4302,43
430204,"石峰区",4302,43
430211,"天元区",4302,43
430212,"渌口区",4302,43
430223,"攸县",4302,43
430224,"茶陵县",4302,43
430225,"炎陵县",4302,43
430281,"醴陵市",4302,43
430302,"雨湖区",4303,43
430304,"岳塘区",4303,43
430321,"湘潭县",4303,43
430371,"湖南湘潭高新技术产业园区",4303,43
430372,"湘潭昭山示范区",4303,43
430373,"湘潭九华示范区",4303,43
430381,"湘乡市",4303,43
430382,"韶山市",4303,43
430405,"珠晖区",4304,43
430406,"雁峰区",4304,43
430407,"石鼓区",4304,43
430408,"蒸湘区",4304,43
430412,"南岳区",4304,43
430421,"衡阳县",4304,43
430422,"衡南县",4304,43
430423,"衡山县",4304,43
430424,"衡东县",4304,43
430426,"祁东县",4304,43
430473,"湖南衡阳松木经济开发区",4304,43
430476,"湖南衡阳高新技术产业园区",4304,43
430481,"耒阳市",4304,43
430482,"常宁市",4304,43
430502,"双清区",4305,43
430503,"大祥区",4305,43
430511,"北塔区",4305,43
430522,"新邵县",4305,43
430523,"邵阳县",4305,43
430524,"隆回县",4305,43
430525,"洞口县",4305,43
430527,"绥宁县",4305,43
430528,"新宁县",4305,43
430529,"城步苗族自治县",4305,43
430581,"武冈市",4305,43
430582,"邵东市",4305,43
430602,"岳阳楼区",4306,43
430603,"云溪区",4306,43
430611,"君山区",4306,43
430621,"岳阳县",4306,43
430623,"华容县",4306,43
430624,"湘阴县",4306,43
430626,"平江县",4306,43
430671,"岳阳市屈原管理区",4306,43
430681,"汨罗市",4306,43
430682,"临湘市",4306,43
430702,"武陵区",4307,43
430703,"鼎城区",4307,43
430721,"安乡县",4307,43
430722,"汉寿县",4307,43
430723,"澧县",4307,43
430724,"临澧县",4307,43
430725,"桃源县",4307,43
430726,"石门县",4307,43
430771,"常德市西洞庭管理区",4307,43
430781,"津市市",4307,43
430802,"永定区",4308,43
430811,"武陵源区",4308,43
430821,"慈利县",4308,43
430822,"桑植县",4308,43
430902,"资阳区",4309,43
430903,"赫山区",4309,43
430921,"南县",4309,43
430922,"桃江县",4309,43
430923,"安化县",4309,43
430971,"益阳市大通湖管理区",4309,43
430972,"湖南益阳高新技术产业园区",4309,43
430981,"沅江市",4309,43
431002,"北湖区",4310,43
431003,"苏仙区",4310,43
431021,"桂阳县",4310,43
431022,"宜章县",4310,43
431023,"永兴县",4310,43
431024,"嘉禾县",4310,43
431025,"临武县",4310,43
431026,"汝城县",4310,43
431027,"桂东县",4310,43
431028,"安仁县",4310,43
431081,"资兴市",4310,43
431102,"零陵区",4311,43
431103,"冷水滩区",4311,43
431122,"东安县",4311,43
431123,"双牌县",4311,43
431124,"道县",4311,43
431125,"江永县",4311,43
431126,"宁远县",4311,43
431127,"蓝山县",4311,43
431128,"新田县",4311,43
431129,"江华瑶族自治县",4311,43
431171,"永州经济技术开发区",4311,43
431173,"永州市回龙圩管理区",4311,43
431181,"祁阳市",4311,43
431202,"鹤城区",4312,43
431221,"中方县",4312,43
431222,"沅陵县",4312,43
431223,"辰溪县",4312,43
431224,"溆浦县",4312,43
431225,"会同县",4312,43
431226,"麻阳苗族自治县",4312,43
431227,"新晃侗族自治县",4312,43
431228,"芷江侗族自治县",4312,43
431229,"靖州苗族侗族自治县",4312,43
431230,"通道侗族自治县",4312,43
431271,"怀化市洪江管理区",4312,43
431281,"洪江市",4312,43
431302,"娄星区",4313,43
431321,"双峰县",4313,43
431322,"新化县",4313,43
431381,"冷水江市",4313,43
431382,"涟源市",4313,43
433101,"吉首市",4331,43
433122,"泸溪县",4331,43
433123,"凤凰县",4331,43
433124,"花垣县",4331,43
433125,"保靖县",4331,43
433126,"古丈县",4331,43
433127,"永顺县",4331,43
433130,"龙山县",4331,43
440103,"荔湾区",4401,44
440104,"越秀区",4401,44
440105,"海珠区",4401,44
440106,"天河区",4401,44
440111,"白云区",4401,44
440112,"黄埔区",4401,44
440113,"番禺区",4401,44
440114,"花都区",4401,44
440115,"南沙区",4401,44
440117,"从化区",4401,44
440118,"增城区",4401,44
440203,"武江区",4402,44
440204,"浈江区",4402,44
440205,"曲江区",4402,44
440222,"始兴县",4402,44
440224,"仁化县",4402,44
440229,"翁源县",4402,44
440232,"乳源瑶族自治县",4402,44
440233,"新丰县",4402,44
440281,"乐昌市",4402,44
440282,"南雄市",4402,44
440303,"罗湖区",4403,44
440304,"福田区",4403,44
440305,"南山区",4403,44
440306,"宝安区",4403,44
440307,"龙岗区",4403,44
440308,"盐田区",4403,44
440309,"龙华区",4403,44
440310,"坪山区",4403,44
440311,"光明区",4403,44
440402,"香洲区",4404,44
440403,"斗门区",4404,44
440404,"金湾区",4404,44
440507,"龙湖区",4405,44
440511,"金平区",4405,44
440512,"濠江区",4405,44
440513,"潮阳区",4405,44
440514,"潮南区",4405,44
440515,"澄海区",4405,44
440523,"南澳县",4405,44
440604,"禅城区",4406,44
440605,"南海区",4406,44
440606,"顺德区",4406,44
440607,"三水区",4406,44
440608,"高明区",4406,44
440703,"蓬江区",4407,44
440704,"江海区",4407,44
440705,"新会区",4407,44
440781,"台山市",4407,44
440783,"开平市",4407,44
440784,"鹤山市",4407,44
440785,"恩平市",4407,44
440802,"赤坎区",4408,44
440803,"霞山区",4408,44
440804,"坡头区",4408,44
440811,"麻章区",4408,44
440823,"遂溪县",4408,44
440825,"徐闻县",4408,44
440881,"廉江市",4408,44
440882,"雷州市",4408,44
440883,"吴川市",4408,44
440902,"茂南区",4409,44
440904,"电白区",4409,44
440981,"高州市",4409,44
440982,"化州市",4409,44
440983,"信宜市",4409,44
441202,"端州区",4412,44
441203,"鼎湖区",4412,44
441204,"高要区",4412,44
441223,"广宁县",4412,44
441224,"怀集县",4412,44
441225,"封开县",4412,44
441226,"德庆县",4412,44
441284,"四会市",4412,44
441302,"惠城区",4413,44
441303,"惠阳区",4413,44
441322,"博罗县",4413,44
441323,"惠东县",4413,44
441324,"龙门县",4413,44
441402,"梅江区",4414,44
441403,"梅县区",4414,44
441422,"大埔县",4414,44
441423,"丰顺县",4414,44
441424,"五华县",4414,44
441426,"平远县",4414,44
441427,"蕉岭县",4414,44
441481,"兴宁市",4414,44
441502,"城区",4415,44
441521,"海丰县",4415,44
441523,"陆河县",4415,44
441581,"陆丰市",4415,44
441602,"源城区",4416,44
441621,"紫金县",4416,44
441622,"龙川县",4416,44
441623,"连平县",4416,44
441624,"和平县",4416,44
441625,"东源县",4416,44
441702,"江城区",4417,44
441704,"阳东区",4417,44
441721,"阳西县",4417,44
441781,"阳春市",4417,44
441802,"清城区",4418,44
441803,"清新区",4418,44
441821,"佛冈县",4418,44
441823,"阳山县",4418,44
441825,"连山壮族瑶族自治县",4418,44
441826,"连南瑶族自治县",4418,44
441881,"英德市",4418,44
441882,"连州市",4418,44
441900,"东莞市",4419,44
442000,"中山市",4420,44
445102,"湘桥区",4451,44
445103,"潮安区",4451,44
445122,"饶平县",4451,44
445202,"榕城区",4452,44
445203,"揭东区",4452,44
445222,"揭西县",4452,44
445224,"惠来县",4452,44
445281,"普宁市",4452,44
445302,"云城区",4453,44
445303,"云安区",4453,44
445321,"新兴县",4453,44
445322,"郁南县",4453,44
445381,"罗定市",4453,44
450102,"兴宁区",4501,45
450103,"青秀区",4501,45
450105,"江南区",4501,45
450107,"西乡塘区",4501,45
450108,"良庆区",4501,45
450109,"邕宁区",4501,45
450110,"武鸣区",4501,45
450123,"隆安县",4501,45
450124,"马山县",4501,45
450125,"上林县",4501,45
450126,"宾阳县",4501,45
450181,"横州市",4501,45
450202,"城中区",4502,45
450203,"鱼峰区",4502,45
450204,"柳南区",4502,45
450205,"柳北区",4502,45
450206,"柳江区",4502,45
450222,"柳城县",4502,45
450223,"鹿寨县",4502,45
450224,"融安县",4502,45
450225,"融水苗族自治县",4502,45
450226,"三江侗族自治县",4502,45
450302,"秀峰区",4503,45
450303,"叠彩区",4503,45
450304,"象山区",4503,45
450305,"七星区",4503,45
450311,"雁山区",4503,45
450312,"临桂区",4503,45
450321,"阳朔县",4503,45
450323,"灵川县",4503,45
450324,"全州县",4503,45
450325,"兴安县",4503,45
450326,"永福县",4503,45
450327,"灌阳县",4503,45
450328,"龙胜各族自治县",4503,45
450329,"资源县",4503,45
450330,"平乐县",4503,45
450332,"恭城瑶族自治县",4503,45
450381,"荔浦市",4503,45
450403,"万秀区",4504,45
450405,"长洲区",4504,45
450406,"龙圩区",4504,45
450421,"苍梧县",4504,45
450422,"藤县",4504,45
450423,"蒙山县",4504,45
450481,"岑溪市",4504,45
450502,"海城区",4505,45
450503,"银海区",4505,45
450512,"铁山港区",4505,45
450521,"合浦县",4505,45
450602,"港口区",4506,45
450603,"防城区",4506,45
450621,"上思县",4506,45
450681,"东兴市",4506,45
450702,"钦南区",4507,45
450703,"钦北区",4507,45
450721,"灵山县",4507,45
450722,"浦北县",4507,45
450802,"港北区",4508,45
450803,"港南区",4508,45
450804,"覃塘区",4508,45
450821,"平南县",4508,45
450881,"桂平市",4508,45
450902,"玉州区",4509,45
450903,"福绵区",4509,45
450921,"容县",4509,45
450922,"陆川县",4509,45
450923,"博白县",4509,45
450924,"兴业县",4509,45
450981,"北流市",4509,45
451002,"右江区",4510,45
451003,"田阳区",4510,45
451022,"田东县",4510,45
451024,"德保县",4510,45
451026,"那坡县",4510,45
451027,"凌云县",4510,45
451028,"乐业县",4510,45
451029,"田林县",4510,45
451030,"西林县",4510,45
451031,"隆林各族自治县",4510,45
451081,"靖西市",4510,45
451082,"平果市",4510,45
451102,"八步区",4511,45
451103,"平桂区",4511,45
451121,"昭平县",4511,45
451122,"钟山县",4511,45
451123,"富川瑶族自治县",4511,45
451202,"金城江区",4512,45
451203,"宜州区",4512,45
451221,"南丹县",4512,45
451222,"天峨县",4512,45
451223,"凤山县",4512,45
451224,"东兰县",4512,45
451225,"罗城仫佬族自治县",4512,45
451226,"环江毛南族自治县",4512,45
451227,"巴马瑶族自治县",4512,45
451228,"都安瑶族自治县",4512,45
451229,"大化瑶族自治县",4512,45
451302,"兴宾区",4513,45
451321,"忻城县",4513,45
451322,"象州县",4513,45
451323,"武宣县",4513,45
451324,"金秀瑶族自治县",4513,45
451381,"合山市",4513,45
451402,"江州区",4514,45
451421,"扶绥县",4514,45
451422,"宁明县",4514,45
451423,"龙州县",4514,45
451424,"大新县",4514,45
451425,"天等县",4514,45
451481,"凭祥市",4514,45
460105,"秀英区",4601,46
460106,"龙华区",4601,46
460107,"琼山区",4601,46
460108,"美兰区",4601,46
460202,"海棠区",4602,46
460203,"吉阳区",4602,46
460204,"天涯区",4602,46
460205,"崖州区",4602,46
460321,"西沙群岛",4603,46
460322,"南沙群岛",4603,46
460323,"中沙群岛的岛礁及其海域",4603,46
460400,"儋州市",4604,46
469001,"五指山市",4690,46
469002,"琼海市",4690,46
469005,"文昌市",4690,46
469006,"万宁市",4690,46
469007,"东方市",4690,46
469021,"定安县",4690,46
469022,"屯昌县",4690,46
469023,"澄迈县",4690,46
469024,"临高县",4690,46
469025,"白沙黎族自治县",4690,46
469026,"昌江黎族自治县",4690,46
469027,"乐东黎族自治县",4690,46
469028,"陵水黎族自治县",4690,46
469029,"保亭黎族苗族自治县",4690,46
469030,"琼中黎族苗族自治县",4690,46
500101,"万州区",5001,50
500102,"涪陵区",5001,50
500103,"渝中区",5001,50
500104,"大渡口区",5001,50
500105,"江北区",5001,50
500106,"沙坪坝区",5001,50
500107,"九龙坡区",5001,50
500108,"南岸区",5001,50
500109,"北碚区",5001,50
500110,"綦江区",5001,50
500111,"大足区",5001,50
500112,"渝北区",5001,50
500113,"巴南区",5001,50
500114,"黔江区",5001,50
500115,"长寿区",5001,50
500116,"江津区",5001,50
500117,"合川区",5001,50
500118,"永川区",5001,50
500119,"南川区",5001,50
500120,"璧山区",5001,50
500151,"铜梁区",5001,50
500152,"潼南区",5001,50
500153,"荣昌区",5001,50
500154,"开州区",5001,50
500155,"梁平区",5001,50
500156,"武隆区",5001,50
500229,"城口县",5002,50
500230,"丰都县",5002,50
500231,"垫江县",5002,50
500233,"忠县",5002,50
500235,"云阳县",5002,50
500236,"奉节县",5002,50
500237,"巫山县",5002,50
500238,"巫溪县",5002,50
500240,"石柱土家族自治县",5002,50
500241,"秀山土家族苗族自治县",5002,50
500242,"酉阳土家族苗族自治县",5002,50
500243,"彭水苗族土家族自治县",5002,50
510104,"锦江区",5101,51
510105,"青羊区",5101,51
510106,"金牛区",5101,51
510107,"武侯区",5101,51
510108,"成华区",5101,51
510112,"龙泉驿区",5101,51
510113,"青白江区",5101,51
510114,"新都区",5101,51
510115,"温江区",5101,51
510116,"双流区",5101,51
510117,"郫都区",5101,51
510118,"新津区",5101,51
510121,"金堂县",5101,51
510129,"大邑县",5101,51
510131,"蒲江县",5101,51
510181,"都江堰市",5101,51
510182,"彭州市",5101,51
510183,"邛崃市",5101,51
510184,"崇州市",5101,51
510185,"简阳市",5101,51
510302,"自流井区",5103,51
510303,"贡井区",5103,51
510304,"大安区",5103,51
510311,"沿滩区",5103,51
510321,"荣县",5103,51
510322,"富顺县",5103,51
510402,"东区",5104,51
510403,"西区",5104,51
510411,"仁和区",5104,51
510421,"米易县",5104,51
510422,"盐边县",5104,51
510502,"江阳区",5105,51
510503,"纳溪区",5105,51
510504,"龙马潭区",5105,51
510521,"泸县",5105,51
510522,"合江县",5105,51
510524,"叙永县",5105,51
510525,"古蔺县",5105,51
510603,"旌阳区",5106,51
510604,"罗江区",5106,51
510623,"中江县",5106,51
510681,"广汉市",5106,51
510682,"什邡市",5106,51
510683,"绵竹市",5106,51
510703,"涪城区",5107,51
510704,"游仙区",5107,51
510705,"安州区",5107,51
510722,"三台县",5107,51
510723,"盐亭县",5107,51
510725,"梓潼县",5107,51
510726,"北川羌族自治县",5107,51
510727,"平武县",5107,51
510781,"江油市",5107,51
510802,"利州区",5108,51
510811,"昭化区",5108,51
510812,"朝天区",5108,51
510821,"旺苍县",5108,51
510822,"青川县",5108,51
510823,"剑阁县",5108,51
510824,"苍溪县",5108,51
510903,"船山区",5109,51
510904,"安居区",5109,51
510921,"蓬溪县",5109,51
510923,"大英县",5109,51
510981,"射洪市",5109,51
511002,"市中区",5110,51
511011,"东兴区",5110,51
511024,"威远县",5110,51
511025,"资中县",5110,51
511083,"隆昌市",5110,51
511102,"市中区",5111,51
511111,"沙湾区",5111,51
511112,"五通桥区",5111,51
511113,"金口河区",5111,51
511123,"犍为县",5111,51
511124,"井研县",5111,51
511126,"夹江县",5111,51
511129,"沐川县",5111,51
511132,"峨边彝族自治县",5111,51
511133,"马边彝族自治县",5111,51
511181,"峨眉山市",5111,51
511302,"顺庆区",5113,51
511303,"高坪区",5113,51
511304,"嘉陵区",5113,51
511321,"南部县",5113,51
511322,"营山县",5113,51
511323,"蓬安县",5113,51
511324,"仪陇县",5113,51
511325,"西充县",5113,51
511381,"阆中市",5113,51
511402,"东坡区",5114,51
511403,"彭山区",5114,51
511421,"仁寿县",5114,51
511423,"洪雅县",5114,51
511424,"丹棱县",5114,51
511425,"青神县",5114,51
511502,"翠屏区",5115,51
511503,"南溪区",5115,51
511504,"叙州区",5115,51
511523,"江安县",5115,51
511524,"长宁县",5115,51
511525,"高县",5115,51
511526,"珙县",5115,51
511527,"筠连县",5115,51
511528,"兴文县",5115,51
511529,"屏山县",5115,51
511602,"广安区",5116,51
511603,"前锋区",5116,51
511621,"岳池县",5116,51
511622,"武胜县",5116,51
511623,"邻水县",5116,51
511681,"华蓥市",5116,51
511702,"通川区",5117,51
511703,"达川区",5117,51
511722,"宣汉县",5117,51
511723,"开江县",5117,51
511724,"大竹县",5117,51
511725,"渠县",5117,51
511781,"万源市",5117,51
511802,"雨城区",5118,51
511803,"名山区",5118,51
511822,"荥经县",5118,51
511823,"汉源县",5118,51
511824,"石棉县",5118,51
511825,"天全县",5118,51
511826,"芦山县",5118,51
511827,"宝兴县",5118,51
511902,"巴州区",5119,51
511903,"恩阳区",5119,51
511921,"通江县",5119,51
511922,"南江县",5119,51
511923,"平昌县",5119,51
512002,"雁江区",5120,51
512021,"安岳县",5120,51
512022,"乐至县",5120,51
513201,"马尔康市",5132,51
513221,"汶川县",5132,51
513222,"理县",5132,51
513223,"茂县",5132,51
513224,"松潘县",5132,51
513225,"九寨沟县",5132,51
513226,"金川县",5132,51
513227,"小金县",5132,51
513228,"黑水县",5132,51
513230,"壤塘县",5132,51
513231,"阿坝县",5132,51
513232,"若尔盖县",5132,51
513233,"红原县",5132,51
513301,"康定市",5133,51
513322,"泸定县",5133,51
513323,"丹巴县",5133,51
513324,"九龙县",5133,51
513325,"雅江县",5133,51
513326,"道孚县",5133,51
513327,"炉霍县",5133,51
513328,"甘孜县",5133,51
513329,"新龙县",5133,51
513330,"德格县",5133,51
513331,"白玉县",5133,51
513332,"石渠县",5133,51
513333,"色达县",5133,51
513334,"理塘县",5133,51
513335,"巴塘县",5133,51
513336,"乡城县",5133,51
513337,"稻城县",5133,51
513338,"得荣县",5133,51
513401,"西昌市",5134,51
513402,"会理市",5134,51
513422,"木里藏族自治县",5134,51
513423,"盐源县",5134,51
513424,"德昌县",5134,51
513426,"会东县",5134,51
513427,"宁南县",5134,51
513428,"普格县",5134,51
513429,"布拖县",5134,51
513430,"金阳县",5134,51
513431,"昭觉县",5134,51
513432,"喜德县",5134,51
513433,"冕宁县",5134,51
513434,"越西县",5134,51
513435,"甘洛县",5134,51
513436,"美姑县",5134,51
513437,"雷波县",5134,51
520102,"南明区",5201,52
520103,"云岩区",5201,52
520111,"花溪区",5201,52
520112,"乌当区",5201,52
520113,"白云区",5201,52
520115,"观山湖区",5201,52
520121,"开阳县",5201,52
520122,"息烽县",5201,52
520123,"修文县",5201,52
520181,"清镇市",5201,52
520201,"钟山区",5202,52
520203,"六枝特区",5202,52
520204,"水城区",5202,52
520281,"盘州市",5202,52
520302,"红花岗区",5203,52
520303,"汇川区",5203,52
520304,"播州区",5203,52
520322,"桐梓县",5203,52
520323,"绥阳县",5203,52
520324,"正安县",5203,52
520325,"道真仡佬族苗族自治县",5203,52
520326,"务川仡佬族苗族自治县",5203,52
520327,"凤冈县",5203,52
520328,"湄潭县",5203,52
520329,"余庆县",5203,52
520330,"习水县",5203,52
520381,"赤水市",5203,52
520382,"仁怀市",5203,52
520402,"西秀区",5204,52
520403,"平坝区",5204,52
520422,"普定县",5204,52
520423,"镇宁布依族苗族自治县",5204,52
520424,"关岭布依族苗族自治县",5204,52
520425,"紫云苗族布依族自治县",5204,52
520502,"七星关区",5205,52
520521,"大方县",5205,52
520523,"金沙县",5205,52
520524,"织金县",5205,52
520525,"纳雍县",5205,52
520526,"威宁彝族回族苗族自治县",5205,52
520527,"赫章县",5205,52
520581,"黔西市",5205,52
520602,"碧江区",5206,52
520603,"万山区",5206,52
520621,"江口县",5206,52
520622,"玉屏侗族自治县",5206,52
520623,"石阡县",5206,52
520624,"思南县",5206,52
520625,"印江土家族苗族自治县",5206,52
520626,"德江县",5206,52
520627,"沿河土家族自治县",5206,52
520628,"松桃苗族自治县",5206,52
522301,"兴义市",5223,52
522302,"兴仁市",5223,52
522323,"普安县",5223,52
522324,"晴隆县",5223,52
522325,"贞丰县",5223,52
522326,"望谟县",5223,52
522327,"册亨县",5223,52
522328,"安龙县",5223,52
522601,"凯里市",5226,52
522622,"黄平县",5226,52
522623,"施秉县",5226,52
522624,"三穗县",5226,52
522625,"镇远县",5226,52
522626,"岑巩县",5226,52
522627,"天柱县",5226,52
522628,"锦屏县",5226,52
522629,"剑河县",5226,52
522630,"台江县",5226,52
522631,"黎平县",5226,52
522632,"榕江县",5226,52
522633,"从江县",5226,52
522634,"雷山县",5226,52
522635,"麻江县",5226,52
522636,"丹寨县",5226,52
522701,"都匀市",5227,52
522702,"福泉市",5227,52
522722,"荔波县",5227,52
522723,"贵定县",5227,52
522725,"瓮安县",5227,52
522726,"独山县",5227,52
522727,"平塘县",5227,52
522728,"罗甸县",5227,52
522729,"长顺县",5227,52
522730,"龙里县",5227,52
522731,"惠水县",5227,52
522732,"三都水族自治县",5227,52
530102,"五华区",5301,53
530103,"盘龙区",5301,53
530111,"官渡区",5301,53
530112,"西山区",5301,53
530113,"东川区",5301,53
530114,"呈贡区",5301,53
530115,"晋宁区",5301,53
530124,"富民县",5301,53
530125,"宜良县",5301,53
530126,"石林彝族自治县",5301,53
530127,"嵩明县",5301,53
530128,"禄劝彝族苗族自治县",5301,53
530129,"寻甸回族彝族自治县",5301,53
530181,"安宁市",5301,53
530302,"麒麟区",5303,53
530303,"沾益区",5303,53
530304,"马龙区",5303,53
530322,"陆良县",5303,53
530323,"师宗县",5303,53
530324,"罗平县",5303,53
530325,"富源县",5303,53
530326,"会泽县",5303,53
530381,"宣威市",5303,53
530402,"红塔区",5304,53
530403,"江川区",5304,53
530423,"通海县",5304,53
530424,"华宁县",5304,53
530425,"易门县",5304,53
530426,"峨山彝族自治县",5304,53
530427,"新平彝族傣族自治县",5304,53
530428,"元江哈尼族彝族傣族自治县",5304,53
530481,"澄江市",5304,53
530502,"隆阳区",5305,53
530521,"施甸县",5305,53
530523,"龙陵县",5305,53
530524,"昌宁县",5305,53
530581,"腾冲市",5305,53
530602,"昭阳区",5306,53
530621,"鲁甸县",5306,53
530622,"巧家县",5306,53
530623,"盐津县",5306,53
530624,"大关县",5306,53
530625,"永善县",5306,53
530626,"绥江县",5306,53
530627,"镇雄县",5306,53
530628,"彝良县",5306,53
530629,"威信县",5306,53
530681,"水富市",5306,53
530702,"古城区",5307,53
530721,"玉龙纳西族自治县",5307,53
530722,"永胜县",5307,53
530723,"华坪县",5307,53
530724,"宁蒗彝族自治县",5307,53
530802,"思茅区",5308,53
530821,"宁洱哈尼族彝族自治县",5308,53
530822,"墨江哈尼族自治县",5308,53
530823,"景东彝族自治县",5308,53
530824,"景谷傣族彝族自治县",5308,53
530825,"镇沅彝族哈尼族拉祜族自治县",5308,53
530826,"江城哈尼族彝族自治县",5308,53
530827,"孟连傣族拉祜族佤族自治县",5308,53
530828,"澜沧拉祜族自治县",5308,53
530829,"西盟佤族自治县",5308,53
530902,"临翔区",5309,53
530921,"凤庆县",5309,53
530922,"云县",5309,53
530923,"永德县",5309,53
530924,"镇康县",5309,53
530925,"双江拉祜族佤族布朗族傣族自治县",5309,53
530926,"耿马傣族佤族自治县",5309,53
530927,"沧源佤族自治县",5309,53
532301,"楚雄市",5323,53
532302,"禄丰市",5323,53
532322,"双柏县",5323,53
532323,"牟定县",5323,53
532324,"南华县",5323,53
532325,"姚安县",5323,53
532326,"大姚县",5323,53
532327,"永仁县",5323,53
532328,"元谋县",5323,53
532329,"武定县",5323,53
532501,"个旧市",5325,53
532502,"开远市",5325,53
532503,"蒙自市",5325,53
532504,"弥勒市",5325,53
532523,"屏边苗族自治县",5325,53
532524,"建水县",5325,53
532525,"石屏县",5325,53
532527,"泸西县",5325,53
532528,"元阳县",5325,53
532529,"红河县",5325,53
532530,"金平苗族瑶族傣族自治县",5325,53
532531,"绿春县",5325,53
532532,"河口瑶族自治县",5325,53
532601,"文山市",5326,53
532622,"砚山县",5326,53
532623,"西畴县",5326,53
532624,"麻栗坡县",5326,53
532625,"马关县",5326,53
532626,"丘北县",5326,53
532627,"广南县",5326,53
532628,"富宁县",5326,53
532801,"景洪市",5328,53
532822,"勐海县",5328,53
532823,"勐腊县",5328,53
532901,"大理市",5329,53
532922,"漾濞彝族自治县",5329,53
532923,"祥云县",5329,53
532924,"宾川县",5329,53
532925,"弥渡县",5329,53
532926,"南涧彝族自治县",5329,53
532927,"巍山彝族回族自治县",5329,53
532928,"永平县",5329,53
532929,"云龙县",5329,53
532930,"洱源县",5329,53
532931,"剑川县",5329,53
532932,"鹤庆县",5329,53
533102,"瑞丽市",5331,53
533103,"芒市",5331,53
533122,"梁河县",5331,53
533123,"盈江县",5331,53
533124,"陇川县",5331,53
533301,"泸水市",5333,53
533323,"福贡县",5333,53
533324,"贡山独龙族怒族自治县",5333,53
533325,"兰坪白族普米族自治县",5333,53
533401,"香格里拉市",5334,53
533422,"德钦县",5334,53
533423,"维西傈僳族自治县",5334,53
540102,"城关区",5401,54
540103,"堆龙德庆区",5401,54
540104,"达孜区",5401,54
540121,"林周县",5401,54
540122,"当雄县",5401,54
540123,"尼木县",5401,54
540124,"曲水县",5401,54
540127,"墨竹工卡县",5401,54
540171,"格尔木藏青工业园区",5401,54
540172,"拉萨经济技术开发区",5401,54
540173,"西藏文化旅游创意园区",5401,54
540174,"达孜工业园区",5401,54
540202,"桑珠孜区",5402,54
540221,"南木林县",5402,54
540222,"江孜县",5402,54
540223,"定日县",5402,54
540224,"萨迦县",5402,54
540225,"拉孜县",5402,54
540226,"昂仁县",5402,54
540227,"谢通门县",5402,54
540228,"白朗县",5402,54
540229,"仁布县",5402,54
540230,"康马县",5402,54
540231,"定结县",5402,54
540232,"仲巴县",5402,54
540233,"亚东县",5402,54
540234,"吉隆县",5402,54
540235,"聂拉木县",5402,54
540236,"萨嘎县",5402,54
540237,"岗巴县",5402,54
540302,"卡若区",5403,54
540321,"江达县",5403,54
540322,"贡觉县",5403,54
540323,"类乌齐县",5403,54
540324,"丁青县",5403,54
540325,"察雅县",5403,54
540326,"八宿县",5403,54
540327,"左贡县",5403,54
540328,"芒康县",5403,54
540329,"洛隆县",5403,54
540330,"边坝县",5403,54
540402,"巴宜区",5404,54
540421,"工布江达县",5404,54
540423,"墨脱县",5404,54
540424,"波密县",5404,54
540425,"察隅县",5404,54
540426,"朗县",5404,54
540481,"米林市",5404,54
540502,"乃东区",5405,54
540521,"扎囊县",5405,54
540522,"贡嘎县",5405,54
540523,"桑日县",5405,54
540524,"琼结县",5405,54
540525,"曲松县",5405,54
540526,"措美县",5405,54
540527,"洛扎县",5405,54
540528,"加查县",5405,54
540529,"隆子县",5405,54
540531,"浪卡子县",5405,54
540581,"错那市",5405,54
540602,"色尼区",5406,54
540621,"嘉黎县",5406,54
540622,"比如县",5406,54
540623,"聂荣县",5406,54
540624,"安多县",5406,54
540625,"申扎县",5406,54
540626,"索县",5406,54
540627,"班戈县",5406,54
540628,"巴青县",5406,54
540629,"尼玛县",5406,54
540630,"双湖县",5406,54
542521,"普兰县",5425,54
542522,"札达县",5425,54
542523,"噶尔县",5425,54
542524,"日土县",5425,54
542525,"革吉县",5425,54
542526,"改则县",5425,54
542527,"措勤县",5425,54
610102,"新城区",6101,61
610103,"碑林区",6101,61
610104,"莲湖区",6101,61
610111,"灞桥区",6101,61
610112,"未央区",6101,61
610113,"雁塔区",6101,61
610114,"阎良区",6101,61
610115,"临潼区",6101,61
610116,"长安区",6101,61
610117,"高陵区",6101,61
610118,"鄠邑区",6101,61
610122,"蓝田县",6101,61
610124,"周至县",6101,61
610202,"王益区",6102,61
610203,"印台区",6102,61
610204,"耀州区",6102,61
610222,"宜君县",6102,61
610302,"渭滨区",6103,61
610303,"金台区",6103,61
610304,"陈仓区",6103,61
610305,"凤翔区",6103,61
610323,"岐山县",6103,61
610324,"扶风县",6103,61
610326,"眉县",6103,61
610327,"陇县",6103,61
610328,"千阳县",6103,61
610329,"麟游县",6103,61
610330,"凤县",6103,61
610331,"太白县",6103,61
610402,"秦都区",6104,61
610403,"杨陵区",6104,61
610404,"渭城区",6104,61
610422,"三原县",6104,61
610423,"泾阳县",6104,61
610424,"乾县",6104,61
610425,"礼泉县",6104,61
610426,"永寿县",6104,61
610428,"长武县",6104,61
610429,"旬邑县",6104,61
610430,"淳化县",6104,61
610431,"武功县",6104,61
610481,"兴平市",6104,61
610482,"彬州市",6104,61
610502,"临渭区",6105,61
610503,"华州区",6105,61
610522,"潼关县",6105,61
610523,"大荔县",6105,61
610524,"合阳县",6105,61
610525,"澄城县",6105,61
610526,"蒲城县",6105,61
610527,"白水县",6105,61
610528,"富平县",6105,61
610581,"韩城市",6105,61
610582,"华阴市",6105,61
610602,"宝塔区",6106,61
610603,"安塞区",6106,61
610621,"延长县",6106,61
610622,"延川县",6106,61
610625,"志丹县",6106,61
610626,"吴起县",6106,61
610627,"甘泉县",6106,61
610628,"富县",6106,61
610629,"洛川县",6106,61
610630,"宜川县",6106,61
610631,"黄龙县",6106,61
610632,"黄陵县",6106,61
610681,"子长市",6106,61
610702,"汉台区",6107,61
610703,"南郑区",6107,61
610722,"城固县",6107,61
610723,"洋县",6107,61
610724,"西乡县",6107,61
610725,"勉县",6107,61
610726,"宁强县",6107,61
610727,"略阳县",6107,61
610728,"镇巴县",6107,61
610729,"留坝县",6107,61
610730,"佛坪县",6107,61
610802,"榆阳区",6108,61
610803,"横山区",6108,61
610822,"府谷县",6108,61
610824,"靖边县",6108,61
610825,"定边县",6108,61
610826,"绥德县",6108,61
610827,"米脂县",6108,61
610828,"佳县",6108,61
610829,"吴堡县",6108,61
610830,"清涧县",6108,61
610831,"子洲县",6108,61
610881,"神木市",6108,61
610902,"汉滨区",6109,61
610921,"汉阴县",6109,61
610922,"石泉县",6109,61
610923,"宁陕县",6109,61
610924,"紫阳县",6109,61
610925,"岚皋县",6109,61
610926,"平利县",6109,61
610927,"镇坪县",6109,61
610929,"白河县",6109,61
610981,"旬阳市",6109,61
611002,"商州区",6110,61
611021,"洛南县",6110,61
611022,"丹凤县",6110,61
611023,"商南县",6110,61
611024,"山阳县",6110,61
611025,"镇安县",6110,61
611026,"柞水县",6110,61
620102,"城关区",6201,62
620103,"七里河区",6201,62
620104,"西固区",6201,62
620105,"安宁区",6201,62
620111,"红古区",6201,62
620121,"永登县",6201,62
620122,"皋兰县",6201,62
620123,"榆中县",6201,62
620171,"兰州新区",6201,62
620201,"嘉峪关市",6202,62
620302,"金川区",6203,62
620321,"永昌县",6203,62
620402,"白银区",6204,62
620403,"平川区",6204,62
620421,"靖远县",6204,62
620422,"会宁县",6204,62
620423,"景泰县",6204,62
620502,"秦州区",6205,62
620503,"麦积区",6205,62
620521,"清水县",6205,62
620522,"秦安县",6205,62
620523,"甘谷县",6205,62
620524,"武山县",6205,62
620525,"张家川回族自治县",6205,62
620602,"凉州区",6206,62
620621,"民勤县",6206,62
620622,"古浪县",6206,62
620623,"天祝藏族自治县",6206,62
620702,"甘州区",6207,62
620721,"肃南裕固族自治县",6207,62
620722,"民乐县",6207,62
620723,"临泽县",6207,62
620724,"高台县",6207,62
620725,"山丹县",6207,62
620802,"崆峒区",6208,62
620821,"泾川县",6208,62
620822,"灵台县",6208,62
620823,"崇信县",6208,62
620825,"庄浪县",6208,62
620826,"静宁县",6208,62
620881,"华亭市",6208,62
620902,"肃州区",6209,62
620921,"金塔县",6209,62
620922,"瓜州县",6209,62
620923,"肃北蒙古族自治县",6209,62
620924,"阿克塞哈萨克族自治县",6209,62
620981,"玉门市",6209,62
620982,"敦煌市",6209,62
621002,"西峰区",6210,62
621021,"庆城县",6210,62
621022,"环县",6210,62
621023,"华池县",6210,62
621024,"合水县",6210,62
621025,"正宁县",6210,62
621026,"宁县",6210,62
621027,"镇原县",6210,62
621102,"安定区",6211,62
621121,"通渭县",6211,62
621122,"陇西县",6211,62
621123,"渭源县",6211,62
621124,"临洮县",6211,62
621125,"漳县",6211,62
621126,"岷县",6211,62
621202,"武都区",6212,62
621221,"成县",6212,62
621222,"文县",6212,62
621223,"宕昌县",6212,62
621224,"康县",6212,62
621225,"西和县",6212,62
621226,"礼县",6212,62
621227,"徽县",6212,62
621228,"两当县",6212,62
622901,"临夏市",6229,62
622921,"临夏县",6229,62
622922,"康乐县",6229,62
622923,"永靖县",6229,62
622924,"广河县",6229,62
622925,"和政县",6229,62
622926,"东乡族自治县",6229,62
622927,"积石山保安族东乡族撒拉族自治县",6229,62
623001,"合作市",6230,62
623021,"临潭县",6230,62
623022,"卓尼县",6230,62
623023,"舟曲县",6230,62
623024,"迭部县",6230,62
623025,"玛曲县",6230,62
623026,"碌曲县",6230,62
623027,"夏河县",6230,62
630102,"城东区",6301,63
630103,"城中区",6301,63
630104,"城西区",6301,63
630105,"城北区",6301,63
630106,"湟中区",6301,63
630121,"大通回族土族自治县",6301,63
630123,"湟源县",6301,63
630202,"乐都区",6302,63
630203,"平安区",6302,63
630222,"民和回族土族自治县",6302,63
630223,"互助土族自治县",6302,63
630224,"化隆回族自治县",6302,63
630225,"循化撒拉族自治县",6302,63
632221,"门源回族自治县",6322,63
632222,"祁连县",6322,63
632223,"海晏县",6322,63
632224,"刚察县",6322,63
632301,"同仁市",6323,63
632322,"尖扎县",6323,63
632323,"泽库县",6323,63
632324,"河南蒙古族自治县",6323,63
632521,"共和县",6325,63
632522,"同德县",6325,63
632523,"贵德县",6325,63
632524,"兴海县",6325,63
632525,"贵南县",6325,63
632621,"玛沁县",6326,63
632622,"班玛县",6326,63
632623,"甘德县",6326,63
632624,"达日县",6326,63
632625,"久治县",6326,63
632626,"玛多县",6326,63
632701,"玉树市",6327,63
632722,"杂多县",6327,63
632723,"称多县",6327,63
632724,"治多县",6327,63
632725,"囊谦县",6327,63
632726,"曲麻莱县",6327,63
632801,"格尔木市",6328,63
632802,"德令哈市",6328,63
632803,"茫崖市",6328,63
632821,"乌兰县",6328,63
632822,"都兰县",6328,63
632823,"天峻县",6328,63
632857,"大柴旦行政委员会",6328,63
640104,"兴庆区",6401,64
640105,"西夏区",6401,64
640106,"金凤区",6401,64
640121,"永宁县",6401,64
640122,"贺兰县",6401,64
640181,"灵武市",6401,64
640202,"大武口区",6402,64
640205,"惠农区",6402,64
640221,"平罗县",6402,64
640302,"利通区",6403,64
640303,"红寺堡区",6403,64
640323,"盐池县",6403,64
640324,"同心县",6403,64
640381,"青铜峡市",6403,64
640402,"原州区",6404,64
640422,"西吉县",6404,64
640423,"隆德县",6404,64
640424,"泾源县",6404,64
640425,"彭阳县",6404,64
640502,"沙坡头区",6405,64
640521,"中宁县",6405,64
640522,"海原县",6405,64
650102,"天山区",6501,65
650103,"沙依巴克区",6501,65
650104,"新市区",6501,65
650105,"水磨沟区",6501,65
650106,"头屯河区",6501,65
650107,"达坂城区",6501,65
650109,"米东区",6501,65
650121,"乌鲁木齐县",6501,65
650202,"独山子区",6502,65
650203,"克拉玛依区",6502,65
650204,"白碱滩区",6502,65
650205,"乌尔禾区",6502,65
650402,"高昌区",6504,65
650421,"鄯善县",6504,65
650422,"托克逊县",6504,65
650502,"伊州区",6505,65
650521,"巴里坤哈萨克自治县",6505,65
650522,"伊吾县",6505,65
652301,"昌吉市",6523,65
652302,"阜康市",6523,65
652323,"呼图壁县",6523,65
652324,"玛纳斯县",6523,65
652325,"奇台县",6523,65
652327,"吉木萨尔县",6523,65
652328,"木垒哈萨克自治县",6523,65
652701,"博乐市",6527,65
652702,"阿拉山口市",6527,65
652722,"精河县",6527,65
652723,"温泉县",6527,65
652801,"库尔勒市",6528,65
652822,"轮台县",6528,65
652823,"尉犁县",6528,65
652824,"若羌县",6528,65
652825,"且末县",6528,65
652826,"焉耆回族自治县",6528,65
652827,"和静县",6528,65
652828,"和硕县",6528,65
652829,"博湖县",6528,65
652901,"阿克苏市",6529,65
652902,"库车市",6529,65
652922,"温宿县",6529,65
652924,"沙雅县",6529,65
652925,"新和县",6529,65
652926,"拜城县",6529,65
652927,"乌什县",6529,65
652928,"阿瓦提县",6529,65
652929,"柯坪县",6529,65
653001,"阿图什市",6530,65
653022,"阿克陶县",6530,65
653023,"阿合奇县",6530,65
653024,"乌恰县",6530,65
653101,"喀什市",6531,65
653121,"疏附县",6531,65
653122,"疏勒县",6531,65
653123,"英吉沙县",6531,65
653124,"泽普县",6531,65
653125,"莎车县",6531,65
653126,"叶城县",6531,65
653127,"麦盖提县",6531,65
653128,"岳普湖县",6531,65
653129,"伽师县",6531,65
653130,"巴楚县",6531,65
653131,"塔什库尔干塔吉克自治县",6531,65
653201,"和田市",6532,65
653221,"和田县",6532,65
653222,"墨玉县",6532,65
653223,"皮山县",6532,65
653224,"洛浦县",6532,65
653225,"策勒县",6532,65
653226,"于田县",6532,65
653227,"民丰县",6532,65
654002,"伊宁市",6540,65
654003,"奎屯市",6540,65
654004,"霍尔果斯市",6540,65
654021,"伊宁县",6540,65
654022,"察布查尔锡伯自治县",6540,65
654023,"霍城县",6540,65
654024,"巩留县",6540,65
654025,"新源县",6540,65
654026,"昭苏县",6540,65
654027,"特克斯县",6540,65
654028,"尼勒克县",6540,65
654201,"塔城市",6542,65
654202,"乌苏市",6542,65
654203,"沙湾市",6542,65
654221,"额敏县",6542,65
654224,"托里县",6542,65
654225,"裕民县",6542,65
654226,"和布克赛尔蒙古自治县",6542,65
654301,"阿勒泰市",6543,65
654321,"布尔津县",6543,65
654322,"富蕴县",6543,65
654323,"福海县",6543,65
654324,"哈巴河县",6543,65
654325,"青河县",6543,65
654326,"吉木乃县",6543,65
659001,"石河子市",6590,65
659002,"阿拉尔市",6590,65
659003,"图木舒克市",6590,65
659004,"五家渠市",6590,65
659005,"北屯市",6590,65
659006,"铁门关市",6590,65
659007,"双河市",6590,65
659008,"可克达拉市",6590,65
659009,"昆玉市",6590,65
659010,"胡杨河市",6590,65
659011,"新星市",6590,65
659012,"白杨市",6590,65
`;
function parseProvices(provicesText2) {
  const lines = provicesText2.split("\n");
  const provices2 = [];
  const [_, ...data] = lines;
  for (const line of data) {
    const [code, name] = line.split(",");
    if (code && name) {
      provices2.push({ code, name: name.replace(/"/g, "") });
    }
  }
  return provices2;
}
function parseCities(citiesText2) {
  const lines = citiesText2.split("\n");
  const cities2 = [];
  const [_, ...data] = lines;
  for (const line of data) {
    const [code, name, provinceCode] = line.split(",");
    if (code && name && provinceCode) {
      cities2.push({ code, name: name.replace(/"/g, ""), provinceCode });
    }
  }
  return cities2;
}
function parseAreas(areasText2) {
  const lines = areasText2.split("\n");
  const areas2 = [];
  const [_, ...data] = lines;
  for (const line of data) {
    const [code, name, cityCode, provinceCode] = line.split(",");
    if (code && name && cityCode && provinceCode) {
      areas2.push({ code, name: name.replace(/"/g, ""), cityCode, provinceCode });
    }
  }
  return areas2;
}
const provices = parseProvices(provicesText);
const cities = parseCities(citiesText);
const areas = parseAreas(areasText);
function bind(fn, thisArg) {
  return function wrap() {
    return fn.apply(thisArg, arguments);
  };
}
const { toString } = Object.prototype;
const { getPrototypeOf } = Object;
const { iterator, toStringTag } = Symbol;
const kindOf = /* @__PURE__ */ ((cache) => (thing) => {
  const str = toString.call(thing);
  return cache[str] || (cache[str] = str.slice(8, -1).toLowerCase());
})(/* @__PURE__ */ Object.create(null));
const kindOfTest = (type) => {
  type = type.toLowerCase();
  return (thing) => kindOf(thing) === type;
};
const typeOfTest = (type) => (thing) => typeof thing === type;
const { isArray } = Array;
const isUndefined = typeOfTest("undefined");
function isBuffer(val) {
  return val !== null && !isUndefined(val) && val.constructor !== null && !isUndefined(val.constructor) && isFunction(val.constructor.isBuffer) && val.constructor.isBuffer(val);
}
const isArrayBuffer = kindOfTest("ArrayBuffer");
function isArrayBufferView(val) {
  let result;
  if (typeof ArrayBuffer !== "undefined" && ArrayBuffer.isView) {
    result = ArrayBuffer.isView(val);
  } else {
    result = val && val.buffer && isArrayBuffer(val.buffer);
  }
  return result;
}
const isString = typeOfTest("string");
const isFunction = typeOfTest("function");
const isNumber = typeOfTest("number");
const isObject = (thing) => thing !== null && typeof thing === "object";
const isBoolean = (thing) => thing === true || thing === false;
const isPlainObject = (val) => {
  if (kindOf(val) !== "object") {
    return false;
  }
  const prototype2 = getPrototypeOf(val);
  return (prototype2 === null || prototype2 === Object.prototype || Object.getPrototypeOf(prototype2) === null) && !(toStringTag in val) && !(iterator in val);
};
const isDate = kindOfTest("Date");
const isFile = kindOfTest("File");
const isBlob = kindOfTest("Blob");
const isFileList = kindOfTest("FileList");
const isStream = (val) => isObject(val) && isFunction(val.pipe);
const isFormData = (thing) => {
  let kind;
  return thing && (typeof FormData === "function" && thing instanceof FormData || isFunction(thing.append) && ((kind = kindOf(thing)) === "formdata" || // detect form-data instance
  kind === "object" && isFunction(thing.toString) && thing.toString() === "[object FormData]"));
};
const isURLSearchParams = kindOfTest("URLSearchParams");
const [isReadableStream, isRequest, isResponse, isHeaders] = ["ReadableStream", "Request", "Response", "Headers"].map(kindOfTest);
const trim = (str) => str.trim ? str.trim() : str.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g, "");
function forEach(obj, fn, { allOwnKeys = false } = {}) {
  if (obj === null || typeof obj === "undefined") {
    return;
  }
  let i;
  let l;
  if (typeof obj !== "object") {
    obj = [obj];
  }
  if (isArray(obj)) {
    for (i = 0, l = obj.length; i < l; i++) {
      fn.call(null, obj[i], i, obj);
    }
  } else {
    const keys = allOwnKeys ? Object.getOwnPropertyNames(obj) : Object.keys(obj);
    const len = keys.length;
    let key;
    for (i = 0; i < len; i++) {
      key = keys[i];
      fn.call(null, obj[key], key, obj);
    }
  }
}
function findKey(obj, key) {
  key = key.toLowerCase();
  const keys = Object.keys(obj);
  let i = keys.length;
  let _key;
  while (i-- > 0) {
    _key = keys[i];
    if (key === _key.toLowerCase()) {
      return _key;
    }
  }
  return null;
}
const _global = (() => {
  if (typeof globalThis !== "undefined") return globalThis;
  return typeof self !== "undefined" ? self : typeof window !== "undefined" ? window : global;
})();
const isContextDefined = (context) => !isUndefined(context) && context !== _global;
function merge() {
  const { caseless } = isContextDefined(this) && this || {};
  const result = {};
  const assignValue = (val, key) => {
    const targetKey = caseless && findKey(result, key) || key;
    if (isPlainObject(result[targetKey]) && isPlainObject(val)) {
      result[targetKey] = merge(result[targetKey], val);
    } else if (isPlainObject(val)) {
      result[targetKey] = merge({}, val);
    } else if (isArray(val)) {
      result[targetKey] = val.slice();
    } else {
      result[targetKey] = val;
    }
  };
  for (let i = 0, l = arguments.length; i < l; i++) {
    arguments[i] && forEach(arguments[i], assignValue);
  }
  return result;
}
const extend = (a, b, thisArg, { allOwnKeys } = {}) => {
  forEach(b, (val, key) => {
    if (thisArg && isFunction(val)) {
      a[key] = bind(val, thisArg);
    } else {
      a[key] = val;
    }
  }, { allOwnKeys });
  return a;
};
const stripBOM = (content) => {
  if (content.charCodeAt(0) === 65279) {
    content = content.slice(1);
  }
  return content;
};
const inherits = (constructor, superConstructor, props, descriptors2) => {
  constructor.prototype = Object.create(superConstructor.prototype, descriptors2);
  constructor.prototype.constructor = constructor;
  Object.defineProperty(constructor, "super", {
    value: superConstructor.prototype
  });
  props && Object.assign(constructor.prototype, props);
};
const toFlatObject = (sourceObj, destObj, filter2, propFilter) => {
  let props;
  let i;
  let prop;
  const merged = {};
  destObj = destObj || {};
  if (sourceObj == null) return destObj;
  do {
    props = Object.getOwnPropertyNames(sourceObj);
    i = props.length;
    while (i-- > 0) {
      prop = props[i];
      if ((!propFilter || propFilter(prop, sourceObj, destObj)) && !merged[prop]) {
        destObj[prop] = sourceObj[prop];
        merged[prop] = true;
      }
    }
    sourceObj = filter2 !== false && getPrototypeOf(sourceObj);
  } while (sourceObj && (!filter2 || filter2(sourceObj, destObj)) && sourceObj !== Object.prototype);
  return destObj;
};
const endsWith = (str, searchString, position) => {
  str = String(str);
  if (position === void 0 || position > str.length) {
    position = str.length;
  }
  position -= searchString.length;
  const lastIndex = str.indexOf(searchString, position);
  return lastIndex !== -1 && lastIndex === position;
};
const toArray = (thing) => {
  if (!thing) return null;
  if (isArray(thing)) return thing;
  let i = thing.length;
  if (!isNumber(i)) return null;
  const arr = new Array(i);
  while (i-- > 0) {
    arr[i] = thing[i];
  }
  return arr;
};
const isTypedArray = /* @__PURE__ */ ((TypedArray) => {
  return (thing) => {
    return TypedArray && thing instanceof TypedArray;
  };
})(typeof Uint8Array !== "undefined" && getPrototypeOf(Uint8Array));
const forEachEntry = (obj, fn) => {
  const generator = obj && obj[iterator];
  const _iterator = generator.call(obj);
  let result;
  while ((result = _iterator.next()) && !result.done) {
    const pair = result.value;
    fn.call(obj, pair[0], pair[1]);
  }
};
const matchAll = (regExp, str) => {
  let matches;
  const arr = [];
  while ((matches = regExp.exec(str)) !== null) {
    arr.push(matches);
  }
  return arr;
};
const isHTMLForm = kindOfTest("HTMLFormElement");
const toCamelCase = (str) => {
  return str.toLowerCase().replace(
    /[-_\s]([a-z\d])(\w*)/g,
    function replacer(m, p1, p2) {
      return p1.toUpperCase() + p2;
    }
  );
};
const hasOwnProperty = (({ hasOwnProperty: hasOwnProperty2 }) => (obj, prop) => hasOwnProperty2.call(obj, prop))(Object.prototype);
const isRegExp = kindOfTest("RegExp");
const reduceDescriptors = (obj, reducer) => {
  const descriptors2 = Object.getOwnPropertyDescriptors(obj);
  const reducedDescriptors = {};
  forEach(descriptors2, (descriptor, name) => {
    let ret;
    if ((ret = reducer(descriptor, name, obj)) !== false) {
      reducedDescriptors[name] = ret || descriptor;
    }
  });
  Object.defineProperties(obj, reducedDescriptors);
};
const freezeMethods = (obj) => {
  reduceDescriptors(obj, (descriptor, name) => {
    if (isFunction(obj) && ["arguments", "caller", "callee"].indexOf(name) !== -1) {
      return false;
    }
    const value = obj[name];
    if (!isFunction(value)) return;
    descriptor.enumerable = false;
    if ("writable" in descriptor) {
      descriptor.writable = false;
      return;
    }
    if (!descriptor.set) {
      descriptor.set = () => {
        throw Error("Can not rewrite read-only method '" + name + "'");
      };
    }
  });
};
const toObjectSet = (arrayOrString, delimiter) => {
  const obj = {};
  const define = (arr) => {
    arr.forEach((value) => {
      obj[value] = true;
    });
  };
  isArray(arrayOrString) ? define(arrayOrString) : define(String(arrayOrString).split(delimiter));
  return obj;
};
const noop = () => {
};
const toFiniteNumber = (value, defaultValue) => {
  return value != null && Number.isFinite(value = +value) ? value : defaultValue;
};
function isSpecCompliantForm(thing) {
  return !!(thing && isFunction(thing.append) && thing[toStringTag] === "FormData" && thing[iterator]);
}
const toJSONObject = (obj) => {
  const stack = new Array(10);
  const visit = (source, i) => {
    if (isObject(source)) {
      if (stack.indexOf(source) >= 0) {
        return;
      }
      if (!("toJSON" in source)) {
        stack[i] = source;
        const target = isArray(source) ? [] : {};
        forEach(source, (value, key) => {
          const reducedValue = visit(value, i + 1);
          !isUndefined(reducedValue) && (target[key] = reducedValue);
        });
        stack[i] = void 0;
        return target;
      }
    }
    return source;
  };
  return visit(obj, 0);
};
const isAsyncFn = kindOfTest("AsyncFunction");
const isThenable = (thing) => thing && (isObject(thing) || isFunction(thing)) && isFunction(thing.then) && isFunction(thing.catch);
const _setImmediate = ((setImmediateSupported, postMessageSupported) => {
  if (setImmediateSupported) {
    return setImmediate;
  }
  return postMessageSupported ? ((token, callbacks) => {
    _global.addEventListener("message", ({ source, data }) => {
      if (source === _global && data === token) {
        callbacks.length && callbacks.shift()();
      }
    }, false);
    return (cb) => {
      callbacks.push(cb);
      _global.postMessage(token, "*");
    };
  })(`axios@${Math.random()}`, []) : (cb) => setTimeout(cb);
})(
  typeof setImmediate === "function",
  isFunction(_global.postMessage)
);
const asap = typeof queueMicrotask !== "undefined" ? queueMicrotask.bind(_global) : typeof process !== "undefined" && process.nextTick || _setImmediate;
const isIterable = (thing) => thing != null && isFunction(thing[iterator]);
const utils$1 = {
  isArray,
  isArrayBuffer,
  isBuffer,
  isFormData,
  isArrayBufferView,
  isString,
  isNumber,
  isBoolean,
  isObject,
  isPlainObject,
  isReadableStream,
  isRequest,
  isResponse,
  isHeaders,
  isUndefined,
  isDate,
  isFile,
  isBlob,
  isRegExp,
  isFunction,
  isStream,
  isURLSearchParams,
  isTypedArray,
  isFileList,
  forEach,
  merge,
  extend,
  trim,
  stripBOM,
  inherits,
  toFlatObject,
  kindOf,
  kindOfTest,
  endsWith,
  toArray,
  forEachEntry,
  matchAll,
  isHTMLForm,
  hasOwnProperty,
  hasOwnProp: hasOwnProperty,
  // an alias to avoid ESLint no-prototype-builtins detection
  reduceDescriptors,
  freezeMethods,
  toObjectSet,
  toCamelCase,
  noop,
  toFiniteNumber,
  findKey,
  global: _global,
  isContextDefined,
  isSpecCompliantForm,
  toJSONObject,
  isAsyncFn,
  isThenable,
  setImmediate: _setImmediate,
  asap,
  isIterable
};
function AxiosError$1(message, code, config, request, response) {
  Error.call(this);
  if (Error.captureStackTrace) {
    Error.captureStackTrace(this, this.constructor);
  } else {
    this.stack = new Error().stack;
  }
  this.message = message;
  this.name = "AxiosError";
  code && (this.code = code);
  config && (this.config = config);
  request && (this.request = request);
  if (response) {
    this.response = response;
    this.status = response.status ? response.status : null;
  }
}
utils$1.inherits(AxiosError$1, Error, {
  toJSON: function toJSON() {
    return {
      // Standard
      message: this.message,
      name: this.name,
      // Microsoft
      description: this.description,
      number: this.number,
      // Mozilla
      fileName: this.fileName,
      lineNumber: this.lineNumber,
      columnNumber: this.columnNumber,
      stack: this.stack,
      // Axios
      config: utils$1.toJSONObject(this.config),
      code: this.code,
      status: this.status
    };
  }
});
const prototype$1 = AxiosError$1.prototype;
const descriptors = {};
[
  "ERR_BAD_OPTION_VALUE",
  "ERR_BAD_OPTION",
  "ECONNABORTED",
  "ETIMEDOUT",
  "ERR_NETWORK",
  "ERR_FR_TOO_MANY_REDIRECTS",
  "ERR_DEPRECATED",
  "ERR_BAD_RESPONSE",
  "ERR_BAD_REQUEST",
  "ERR_CANCELED",
  "ERR_NOT_SUPPORT",
  "ERR_INVALID_URL"
  // eslint-disable-next-line func-names
].forEach((code) => {
  descriptors[code] = { value: code };
});
Object.defineProperties(AxiosError$1, descriptors);
Object.defineProperty(prototype$1, "isAxiosError", { value: true });
AxiosError$1.from = (error, code, config, request, response, customProps) => {
  const axiosError = Object.create(prototype$1);
  utils$1.toFlatObject(error, axiosError, function filter2(obj) {
    return obj !== Error.prototype;
  }, (prop) => {
    return prop !== "isAxiosError";
  });
  AxiosError$1.call(axiosError, error.message, code, config, request, response);
  axiosError.cause = error;
  axiosError.name = error.name;
  customProps && Object.assign(axiosError, customProps);
  return axiosError;
};
const httpAdapter = null;
function isVisitable(thing) {
  return utils$1.isPlainObject(thing) || utils$1.isArray(thing);
}
function removeBrackets(key) {
  return utils$1.endsWith(key, "[]") ? key.slice(0, -2) : key;
}
function renderKey(path, key, dots) {
  if (!path) return key;
  return path.concat(key).map(function each2(token, i) {
    token = removeBrackets(token);
    return !dots && i ? "[" + token + "]" : token;
  }).join(dots ? "." : "");
}
function isFlatArray(arr) {
  return utils$1.isArray(arr) && !arr.some(isVisitable);
}
const predicates = utils$1.toFlatObject(utils$1, {}, null, function filter(prop) {
  return /^is[A-Z]/.test(prop);
});
function toFormData$1(obj, formData, options) {
  if (!utils$1.isObject(obj)) {
    throw new TypeError("target must be an object");
  }
  formData = formData || new FormData();
  options = utils$1.toFlatObject(options, {
    metaTokens: true,
    dots: false,
    indexes: false
  }, false, function defined(option, source) {
    return !utils$1.isUndefined(source[option]);
  });
  const metaTokens = options.metaTokens;
  const visitor = options.visitor || defaultVisitor;
  const dots = options.dots;
  const indexes = options.indexes;
  const _Blob = options.Blob || typeof Blob !== "undefined" && Blob;
  const useBlob = _Blob && utils$1.isSpecCompliantForm(formData);
  if (!utils$1.isFunction(visitor)) {
    throw new TypeError("visitor must be a function");
  }
  function convertValue(value) {
    if (value === null) return "";
    if (utils$1.isDate(value)) {
      return value.toISOString();
    }
    if (!useBlob && utils$1.isBlob(value)) {
      throw new AxiosError$1("Blob is not supported. Use a Buffer instead.");
    }
    if (utils$1.isArrayBuffer(value) || utils$1.isTypedArray(value)) {
      return useBlob && typeof Blob === "function" ? new Blob([value]) : Buffer.from(value);
    }
    return value;
  }
  function defaultVisitor(value, key, path) {
    let arr = value;
    if (value && !path && typeof value === "object") {
      if (utils$1.endsWith(key, "{}")) {
        key = metaTokens ? key : key.slice(0, -2);
        value = JSON.stringify(value);
      } else if (utils$1.isArray(value) && isFlatArray(value) || (utils$1.isFileList(value) || utils$1.endsWith(key, "[]")) && (arr = utils$1.toArray(value))) {
        key = removeBrackets(key);
        arr.forEach(function each2(el, index) {
          !(utils$1.isUndefined(el) || el === null) && formData.append(
            // eslint-disable-next-line no-nested-ternary
            indexes === true ? renderKey([key], index, dots) : indexes === null ? key : key + "[]",
            convertValue(el)
          );
        });
        return false;
      }
    }
    if (isVisitable(value)) {
      return true;
    }
    formData.append(renderKey(path, key, dots), convertValue(value));
    return false;
  }
  const stack = [];
  const exposedHelpers = Object.assign(predicates, {
    defaultVisitor,
    convertValue,
    isVisitable
  });
  function build(value, path) {
    if (utils$1.isUndefined(value)) return;
    if (stack.indexOf(value) !== -1) {
      throw Error("Circular reference detected in " + path.join("."));
    }
    stack.push(value);
    utils$1.forEach(value, function each2(el, key) {
      const result = !(utils$1.isUndefined(el) || el === null) && visitor.call(
        formData,
        el,
        utils$1.isString(key) ? key.trim() : key,
        path,
        exposedHelpers
      );
      if (result === true) {
        build(el, path ? path.concat(key) : [key]);
      }
    });
    stack.pop();
  }
  if (!utils$1.isObject(obj)) {
    throw new TypeError("data must be an object");
  }
  build(obj);
  return formData;
}
function encode$1(str) {
  const charMap = {
    "!": "%21",
    "'": "%27",
    "(": "%28",
    ")": "%29",
    "~": "%7E",
    "%20": "+",
    "%00": "\0"
  };
  return encodeURIComponent(str).replace(/[!'()~]|%20|%00/g, function replacer(match) {
    return charMap[match];
  });
}
function AxiosURLSearchParams(params, options) {
  this._pairs = [];
  params && toFormData$1(params, this, options);
}
const prototype = AxiosURLSearchParams.prototype;
prototype.append = function append(name, value) {
  this._pairs.push([name, value]);
};
prototype.toString = function toString2(encoder) {
  const _encode = encoder ? function(value) {
    return encoder.call(this, value, encode$1);
  } : encode$1;
  return this._pairs.map(function each2(pair) {
    return _encode(pair[0]) + "=" + _encode(pair[1]);
  }, "").join("&");
};
function encode(val) {
  return encodeURIComponent(val).replace(/%3A/gi, ":").replace(/%24/g, "$").replace(/%2C/gi, ",").replace(/%20/g, "+").replace(/%5B/gi, "[").replace(/%5D/gi, "]");
}
function buildURL(url, params, options) {
  if (!params) {
    return url;
  }
  const _encode = options && options.encode || encode;
  if (utils$1.isFunction(options)) {
    options = {
      serialize: options
    };
  }
  const serializeFn = options && options.serialize;
  let serializedParams;
  if (serializeFn) {
    serializedParams = serializeFn(params, options);
  } else {
    serializedParams = utils$1.isURLSearchParams(params) ? params.toString() : new AxiosURLSearchParams(params, options).toString(_encode);
  }
  if (serializedParams) {
    const hashmarkIndex = url.indexOf("#");
    if (hashmarkIndex !== -1) {
      url = url.slice(0, hashmarkIndex);
    }
    url += (url.indexOf("?") === -1 ? "?" : "&") + serializedParams;
  }
  return url;
}
class InterceptorManager {
  constructor() {
    this.handlers = [];
  }
  /**
   * Add a new interceptor to the stack
   *
   * @param {Function} fulfilled The function to handle `then` for a `Promise`
   * @param {Function} rejected The function to handle `reject` for a `Promise`
   *
   * @return {Number} An ID used to remove interceptor later
   */
  use(fulfilled, rejected, options) {
    this.handlers.push({
      fulfilled,
      rejected,
      synchronous: options ? options.synchronous : false,
      runWhen: options ? options.runWhen : null
    });
    return this.handlers.length - 1;
  }
  /**
   * Remove an interceptor from the stack
   *
   * @param {Number} id The ID that was returned by `use`
   *
   * @returns {Boolean} `true` if the interceptor was removed, `false` otherwise
   */
  eject(id2) {
    if (this.handlers[id2]) {
      this.handlers[id2] = null;
    }
  }
  /**
   * Clear all interceptors from the stack
   *
   * @returns {void}
   */
  clear() {
    if (this.handlers) {
      this.handlers = [];
    }
  }
  /**
   * Iterate over all the registered interceptors
   *
   * This method is particularly useful for skipping over any
   * interceptors that may have become `null` calling `eject`.
   *
   * @param {Function} fn The function to call for each interceptor
   *
   * @returns {void}
   */
  forEach(fn) {
    utils$1.forEach(this.handlers, function forEachHandler(h) {
      if (h !== null) {
        fn(h);
      }
    });
  }
}
const transitionalDefaults = {
  silentJSONParsing: true,
  forcedJSONParsing: true,
  clarifyTimeoutError: false
};
const URLSearchParams$1 = typeof URLSearchParams !== "undefined" ? URLSearchParams : AxiosURLSearchParams;
const FormData$1 = typeof FormData !== "undefined" ? FormData : null;
const Blob$1 = typeof Blob !== "undefined" ? Blob : null;
const platform$1 = {
  isBrowser: true,
  classes: {
    URLSearchParams: URLSearchParams$1,
    FormData: FormData$1,
    Blob: Blob$1
  },
  protocols: ["http", "https", "file", "blob", "url", "data"]
};
const hasBrowserEnv = typeof window !== "undefined" && typeof document !== "undefined";
const _navigator = typeof navigator === "object" && navigator || void 0;
const hasStandardBrowserEnv = hasBrowserEnv && (!_navigator || ["ReactNative", "NativeScript", "NS"].indexOf(_navigator.product) < 0);
const hasStandardBrowserWebWorkerEnv = (() => {
  return typeof WorkerGlobalScope !== "undefined" && // eslint-disable-next-line no-undef
  self instanceof WorkerGlobalScope && typeof self.importScripts === "function";
})();
const origin = hasBrowserEnv && window.location.href || "http://localhost";
const utils = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  hasBrowserEnv,
  hasStandardBrowserEnv,
  hasStandardBrowserWebWorkerEnv,
  navigator: _navigator,
  origin
}, Symbol.toStringTag, { value: "Module" }));
const platform = {
  ...utils,
  ...platform$1
};
function toURLEncodedForm(data, options) {
  return toFormData$1(data, new platform.classes.URLSearchParams(), Object.assign({
    visitor: function(value, key, path, helpers) {
      if (platform.isNode && utils$1.isBuffer(value)) {
        this.append(key, value.toString("base64"));
        return false;
      }
      return helpers.defaultVisitor.apply(this, arguments);
    }
  }, options));
}
function parsePropPath(name) {
  return utils$1.matchAll(/\w+|\[(\w*)]/g, name).map((match) => {
    return match[0] === "[]" ? "" : match[1] || match[0];
  });
}
function arrayToObject(arr) {
  const obj = {};
  const keys = Object.keys(arr);
  let i;
  const len = keys.length;
  let key;
  for (i = 0; i < len; i++) {
    key = keys[i];
    obj[key] = arr[key];
  }
  return obj;
}
function formDataToJSON(formData) {
  function buildPath(path, value, target, index) {
    let name = path[index++];
    if (name === "__proto__") return true;
    const isNumericKey = Number.isFinite(+name);
    const isLast = index >= path.length;
    name = !name && utils$1.isArray(target) ? target.length : name;
    if (isLast) {
      if (utils$1.hasOwnProp(target, name)) {
        target[name] = [target[name], value];
      } else {
        target[name] = value;
      }
      return !isNumericKey;
    }
    if (!target[name] || !utils$1.isObject(target[name])) {
      target[name] = [];
    }
    const result = buildPath(path, value, target[name], index);
    if (result && utils$1.isArray(target[name])) {
      target[name] = arrayToObject(target[name]);
    }
    return !isNumericKey;
  }
  if (utils$1.isFormData(formData) && utils$1.isFunction(formData.entries)) {
    const obj = {};
    utils$1.forEachEntry(formData, (name, value) => {
      buildPath(parsePropPath(name), value, obj, 0);
    });
    return obj;
  }
  return null;
}
function stringifySafely(rawValue, parser, encoder) {
  if (utils$1.isString(rawValue)) {
    try {
      (parser || JSON.parse)(rawValue);
      return utils$1.trim(rawValue);
    } catch (e) {
      if (e.name !== "SyntaxError") {
        throw e;
      }
    }
  }
  return (encoder || JSON.stringify)(rawValue);
}
const defaults = {
  transitional: transitionalDefaults,
  adapter: ["xhr", "http", "fetch"],
  transformRequest: [function transformRequest(data, headers) {
    const contentType = headers.getContentType() || "";
    const hasJSONContentType = contentType.indexOf("application/json") > -1;
    const isObjectPayload = utils$1.isObject(data);
    if (isObjectPayload && utils$1.isHTMLForm(data)) {
      data = new FormData(data);
    }
    const isFormData2 = utils$1.isFormData(data);
    if (isFormData2) {
      return hasJSONContentType ? JSON.stringify(formDataToJSON(data)) : data;
    }
    if (utils$1.isArrayBuffer(data) || utils$1.isBuffer(data) || utils$1.isStream(data) || utils$1.isFile(data) || utils$1.isBlob(data) || utils$1.isReadableStream(data)) {
      return data;
    }
    if (utils$1.isArrayBufferView(data)) {
      return data.buffer;
    }
    if (utils$1.isURLSearchParams(data)) {
      headers.setContentType("application/x-www-form-urlencoded;charset=utf-8", false);
      return data.toString();
    }
    let isFileList2;
    if (isObjectPayload) {
      if (contentType.indexOf("application/x-www-form-urlencoded") > -1) {
        return toURLEncodedForm(data, this.formSerializer).toString();
      }
      if ((isFileList2 = utils$1.isFileList(data)) || contentType.indexOf("multipart/form-data") > -1) {
        const _FormData = this.env && this.env.FormData;
        return toFormData$1(
          isFileList2 ? { "files[]": data } : data,
          _FormData && new _FormData(),
          this.formSerializer
        );
      }
    }
    if (isObjectPayload || hasJSONContentType) {
      headers.setContentType("application/json", false);
      return stringifySafely(data);
    }
    return data;
  }],
  transformResponse: [function transformResponse(data) {
    const transitional2 = this.transitional || defaults.transitional;
    const forcedJSONParsing = transitional2 && transitional2.forcedJSONParsing;
    const JSONRequested = this.responseType === "json";
    if (utils$1.isResponse(data) || utils$1.isReadableStream(data)) {
      return data;
    }
    if (data && utils$1.isString(data) && (forcedJSONParsing && !this.responseType || JSONRequested)) {
      const silentJSONParsing = transitional2 && transitional2.silentJSONParsing;
      const strictJSONParsing = !silentJSONParsing && JSONRequested;
      try {
        return JSON.parse(data);
      } catch (e) {
        if (strictJSONParsing) {
          if (e.name === "SyntaxError") {
            throw AxiosError$1.from(e, AxiosError$1.ERR_BAD_RESPONSE, this, null, this.response);
          }
          throw e;
        }
      }
    }
    return data;
  }],
  /**
   * A timeout in milliseconds to abort a request. If set to 0 (default) a
   * timeout is not created.
   */
  timeout: 0,
  xsrfCookieName: "XSRF-TOKEN",
  xsrfHeaderName: "X-XSRF-TOKEN",
  maxContentLength: -1,
  maxBodyLength: -1,
  env: {
    FormData: platform.classes.FormData,
    Blob: platform.classes.Blob
  },
  validateStatus: function validateStatus(status) {
    return status >= 200 && status < 300;
  },
  headers: {
    common: {
      "Accept": "application/json, text/plain, */*",
      "Content-Type": void 0
    }
  }
};
utils$1.forEach(["delete", "get", "head", "post", "put", "patch"], (method) => {
  defaults.headers[method] = {};
});
const ignoreDuplicateOf = utils$1.toObjectSet([
  "age",
  "authorization",
  "content-length",
  "content-type",
  "etag",
  "expires",
  "from",
  "host",
  "if-modified-since",
  "if-unmodified-since",
  "last-modified",
  "location",
  "max-forwards",
  "proxy-authorization",
  "referer",
  "retry-after",
  "user-agent"
]);
const parseHeaders = (rawHeaders) => {
  const parsed = {};
  let key;
  let val;
  let i;
  rawHeaders && rawHeaders.split("\n").forEach(function parser(line) {
    i = line.indexOf(":");
    key = line.substring(0, i).trim().toLowerCase();
    val = line.substring(i + 1).trim();
    if (!key || parsed[key] && ignoreDuplicateOf[key]) {
      return;
    }
    if (key === "set-cookie") {
      if (parsed[key]) {
        parsed[key].push(val);
      } else {
        parsed[key] = [val];
      }
    } else {
      parsed[key] = parsed[key] ? parsed[key] + ", " + val : val;
    }
  });
  return parsed;
};
const $internals = Symbol("internals");
function normalizeHeader(header) {
  return header && String(header).trim().toLowerCase();
}
function normalizeValue(value) {
  if (value === false || value == null) {
    return value;
  }
  return utils$1.isArray(value) ? value.map(normalizeValue) : String(value);
}
function parseTokens(str) {
  const tokens = /* @__PURE__ */ Object.create(null);
  const tokensRE = /([^\s,;=]+)\s*(?:=\s*([^,;]+))?/g;
  let match;
  while (match = tokensRE.exec(str)) {
    tokens[match[1]] = match[2];
  }
  return tokens;
}
const isValidHeaderName = (str) => /^[-_a-zA-Z0-9^`|~,!#$%&'*+.]+$/.test(str.trim());
function matchHeaderValue(context, value, header, filter2, isHeaderNameFilter) {
  if (utils$1.isFunction(filter2)) {
    return filter2.call(this, value, header);
  }
  if (isHeaderNameFilter) {
    value = header;
  }
  if (!utils$1.isString(value)) return;
  if (utils$1.isString(filter2)) {
    return value.indexOf(filter2) !== -1;
  }
  if (utils$1.isRegExp(filter2)) {
    return filter2.test(value);
  }
}
function formatHeader(header) {
  return header.trim().toLowerCase().replace(/([a-z\d])(\w*)/g, (w, char, str) => {
    return char.toUpperCase() + str;
  });
}
function buildAccessors(obj, header) {
  const accessorName = utils$1.toCamelCase(" " + header);
  ["get", "set", "has"].forEach((methodName) => {
    Object.defineProperty(obj, methodName + accessorName, {
      value: function(arg1, arg2, arg3) {
        return this[methodName].call(this, header, arg1, arg2, arg3);
      },
      configurable: true
    });
  });
}
let AxiosHeaders$1 = class AxiosHeaders {
  constructor(headers) {
    headers && this.set(headers);
  }
  set(header, valueOrRewrite, rewrite) {
    const self2 = this;
    function setHeader(_value, _header, _rewrite) {
      const lHeader = normalizeHeader(_header);
      if (!lHeader) {
        throw new Error("header name must be a non-empty string");
      }
      const key = utils$1.findKey(self2, lHeader);
      if (!key || self2[key] === void 0 || _rewrite === true || _rewrite === void 0 && self2[key] !== false) {
        self2[key || _header] = normalizeValue(_value);
      }
    }
    const setHeaders = (headers, _rewrite) => utils$1.forEach(headers, (_value, _header) => setHeader(_value, _header, _rewrite));
    if (utils$1.isPlainObject(header) || header instanceof this.constructor) {
      setHeaders(header, valueOrRewrite);
    } else if (utils$1.isString(header) && (header = header.trim()) && !isValidHeaderName(header)) {
      setHeaders(parseHeaders(header), valueOrRewrite);
    } else if (utils$1.isObject(header) && utils$1.isIterable(header)) {
      let obj = {}, dest, key;
      for (const entry of header) {
        if (!utils$1.isArray(entry)) {
          throw TypeError("Object iterator must return a key-value pair");
        }
        obj[key = entry[0]] = (dest = obj[key]) ? utils$1.isArray(dest) ? [...dest, entry[1]] : [dest, entry[1]] : entry[1];
      }
      setHeaders(obj, valueOrRewrite);
    } else {
      header != null && setHeader(valueOrRewrite, header, rewrite);
    }
    return this;
  }
  get(header, parser) {
    header = normalizeHeader(header);
    if (header) {
      const key = utils$1.findKey(this, header);
      if (key) {
        const value = this[key];
        if (!parser) {
          return value;
        }
        if (parser === true) {
          return parseTokens(value);
        }
        if (utils$1.isFunction(parser)) {
          return parser.call(this, value, key);
        }
        if (utils$1.isRegExp(parser)) {
          return parser.exec(value);
        }
        throw new TypeError("parser must be boolean|regexp|function");
      }
    }
  }
  has(header, matcher) {
    header = normalizeHeader(header);
    if (header) {
      const key = utils$1.findKey(this, header);
      return !!(key && this[key] !== void 0 && (!matcher || matchHeaderValue(this, this[key], key, matcher)));
    }
    return false;
  }
  delete(header, matcher) {
    const self2 = this;
    let deleted = false;
    function deleteHeader(_header) {
      _header = normalizeHeader(_header);
      if (_header) {
        const key = utils$1.findKey(self2, _header);
        if (key && (!matcher || matchHeaderValue(self2, self2[key], key, matcher))) {
          delete self2[key];
          deleted = true;
        }
      }
    }
    if (utils$1.isArray(header)) {
      header.forEach(deleteHeader);
    } else {
      deleteHeader(header);
    }
    return deleted;
  }
  clear(matcher) {
    const keys = Object.keys(this);
    let i = keys.length;
    let deleted = false;
    while (i--) {
      const key = keys[i];
      if (!matcher || matchHeaderValue(this, this[key], key, matcher, true)) {
        delete this[key];
        deleted = true;
      }
    }
    return deleted;
  }
  normalize(format) {
    const self2 = this;
    const headers = {};
    utils$1.forEach(this, (value, header) => {
      const key = utils$1.findKey(headers, header);
      if (key) {
        self2[key] = normalizeValue(value);
        delete self2[header];
        return;
      }
      const normalized = format ? formatHeader(header) : String(header).trim();
      if (normalized !== header) {
        delete self2[header];
      }
      self2[normalized] = normalizeValue(value);
      headers[normalized] = true;
    });
    return this;
  }
  concat(...targets) {
    return this.constructor.concat(this, ...targets);
  }
  toJSON(asStrings) {
    const obj = /* @__PURE__ */ Object.create(null);
    utils$1.forEach(this, (value, header) => {
      value != null && value !== false && (obj[header] = asStrings && utils$1.isArray(value) ? value.join(", ") : value);
    });
    return obj;
  }
  [Symbol.iterator]() {
    return Object.entries(this.toJSON())[Symbol.iterator]();
  }
  toString() {
    return Object.entries(this.toJSON()).map(([header, value]) => header + ": " + value).join("\n");
  }
  getSetCookie() {
    return this.get("set-cookie") || [];
  }
  get [Symbol.toStringTag]() {
    return "AxiosHeaders";
  }
  static from(thing) {
    return thing instanceof this ? thing : new this(thing);
  }
  static concat(first, ...targets) {
    const computed = new this(first);
    targets.forEach((target) => computed.set(target));
    return computed;
  }
  static accessor(header) {
    const internals = this[$internals] = this[$internals] = {
      accessors: {}
    };
    const accessors = internals.accessors;
    const prototype2 = this.prototype;
    function defineAccessor(_header) {
      const lHeader = normalizeHeader(_header);
      if (!accessors[lHeader]) {
        buildAccessors(prototype2, _header);
        accessors[lHeader] = true;
      }
    }
    utils$1.isArray(header) ? header.forEach(defineAccessor) : defineAccessor(header);
    return this;
  }
};
AxiosHeaders$1.accessor(["Content-Type", "Content-Length", "Accept", "Accept-Encoding", "User-Agent", "Authorization"]);
utils$1.reduceDescriptors(AxiosHeaders$1.prototype, ({ value }, key) => {
  let mapped = key[0].toUpperCase() + key.slice(1);
  return {
    get: () => value,
    set(headerValue) {
      this[mapped] = headerValue;
    }
  };
});
utils$1.freezeMethods(AxiosHeaders$1);
function transformData(fns, response) {
  const config = this || defaults;
  const context = response || config;
  const headers = AxiosHeaders$1.from(context.headers);
  let data = context.data;
  utils$1.forEach(fns, function transform(fn) {
    data = fn.call(config, data, headers.normalize(), response ? response.status : void 0);
  });
  headers.normalize();
  return data;
}
function isCancel$1(value) {
  return !!(value && value.__CANCEL__);
}
function CanceledError$1(message, config, request) {
  AxiosError$1.call(this, message == null ? "canceled" : message, AxiosError$1.ERR_CANCELED, config, request);
  this.name = "CanceledError";
}
utils$1.inherits(CanceledError$1, AxiosError$1, {
  __CANCEL__: true
});
function settle(resolve, reject, response) {
  const validateStatus2 = response.config.validateStatus;
  if (!response.status || !validateStatus2 || validateStatus2(response.status)) {
    resolve(response);
  } else {
    reject(new AxiosError$1(
      "Request failed with status code " + response.status,
      [AxiosError$1.ERR_BAD_REQUEST, AxiosError$1.ERR_BAD_RESPONSE][Math.floor(response.status / 100) - 4],
      response.config,
      response.request,
      response
    ));
  }
}
function parseProtocol(url) {
  const match = /^([-+\w]{1,25})(:?\/\/|:)/.exec(url);
  return match && match[1] || "";
}
function speedometer(samplesCount, min) {
  samplesCount = samplesCount || 10;
  const bytes = new Array(samplesCount);
  const timestamps = new Array(samplesCount);
  let head = 0;
  let tail = 0;
  let firstSampleTS;
  min = min !== void 0 ? min : 1e3;
  return function push(chunkLength) {
    const now = Date.now();
    const startedAt = timestamps[tail];
    if (!firstSampleTS) {
      firstSampleTS = now;
    }
    bytes[head] = chunkLength;
    timestamps[head] = now;
    let i = tail;
    let bytesCount = 0;
    while (i !== head) {
      bytesCount += bytes[i++];
      i = i % samplesCount;
    }
    head = (head + 1) % samplesCount;
    if (head === tail) {
      tail = (tail + 1) % samplesCount;
    }
    if (now - firstSampleTS < min) {
      return;
    }
    const passed = startedAt && now - startedAt;
    return passed ? Math.round(bytesCount * 1e3 / passed) : void 0;
  };
}
function throttle(fn, freq) {
  let timestamp = 0;
  let threshold = 1e3 / freq;
  let lastArgs;
  let timer;
  const invoke = (args, now = Date.now()) => {
    timestamp = now;
    lastArgs = null;
    if (timer) {
      clearTimeout(timer);
      timer = null;
    }
    fn.apply(null, args);
  };
  const throttled = (...args) => {
    const now = Date.now();
    const passed = now - timestamp;
    if (passed >= threshold) {
      invoke(args, now);
    } else {
      lastArgs = args;
      if (!timer) {
        timer = setTimeout(() => {
          timer = null;
          invoke(lastArgs);
        }, threshold - passed);
      }
    }
  };
  const flush = () => lastArgs && invoke(lastArgs);
  return [throttled, flush];
}
const progressEventReducer = (listener, isDownloadStream, freq = 3) => {
  let bytesNotified = 0;
  const _speedometer = speedometer(50, 250);
  return throttle((e) => {
    const loaded = e.loaded;
    const total = e.lengthComputable ? e.total : void 0;
    const progressBytes = loaded - bytesNotified;
    const rate = _speedometer(progressBytes);
    const inRange = loaded <= total;
    bytesNotified = loaded;
    const data = {
      loaded,
      total,
      progress: total ? loaded / total : void 0,
      bytes: progressBytes,
      rate: rate ? rate : void 0,
      estimated: rate && total && inRange ? (total - loaded) / rate : void 0,
      event: e,
      lengthComputable: total != null,
      [isDownloadStream ? "download" : "upload"]: true
    };
    listener(data);
  }, freq);
};
const progressEventDecorator = (total, throttled) => {
  const lengthComputable = total != null;
  return [(loaded) => throttled[0]({
    lengthComputable,
    total,
    loaded
  }), throttled[1]];
};
const asyncDecorator = (fn) => (...args) => utils$1.asap(() => fn(...args));
const isURLSameOrigin = platform.hasStandardBrowserEnv ? /* @__PURE__ */ ((origin2, isMSIE) => (url) => {
  url = new URL(url, platform.origin);
  return origin2.protocol === url.protocol && origin2.host === url.host && (isMSIE || origin2.port === url.port);
})(
  new URL(platform.origin),
  platform.navigator && /(msie|trident)/i.test(platform.navigator.userAgent)
) : () => true;
const cookies = platform.hasStandardBrowserEnv ? (
  // Standard browser envs support document.cookie
  {
    write(name, value, expires, path, domain, secure) {
      const cookie = [name + "=" + encodeURIComponent(value)];
      utils$1.isNumber(expires) && cookie.push("expires=" + new Date(expires).toGMTString());
      utils$1.isString(path) && cookie.push("path=" + path);
      utils$1.isString(domain) && cookie.push("domain=" + domain);
      secure === true && cookie.push("secure");
      document.cookie = cookie.join("; ");
    },
    read(name) {
      const match = document.cookie.match(new RegExp("(^|;\\s*)(" + name + ")=([^;]*)"));
      return match ? decodeURIComponent(match[3]) : null;
    },
    remove(name) {
      this.write(name, "", Date.now() - 864e5);
    }
  }
) : (
  // Non-standard browser env (web workers, react-native) lack needed support.
  {
    write() {
    },
    read() {
      return null;
    },
    remove() {
    }
  }
);
function isAbsoluteURL(url) {
  return /^([a-z][a-z\d+\-.]*:)?\/\//i.test(url);
}
function combineURLs(baseURL, relativeURL) {
  return relativeURL ? baseURL.replace(/\/?\/$/, "") + "/" + relativeURL.replace(/^\/+/, "") : baseURL;
}
function buildFullPath(baseURL, requestedURL, allowAbsoluteUrls) {
  let isRelativeUrl = !isAbsoluteURL(requestedURL);
  if (baseURL && (isRelativeUrl || allowAbsoluteUrls == false)) {
    return combineURLs(baseURL, requestedURL);
  }
  return requestedURL;
}
const headersToObject = (thing) => thing instanceof AxiosHeaders$1 ? { ...thing } : thing;
function mergeConfig$1(config1, config2) {
  config2 = config2 || {};
  const config = {};
  function getMergedValue(target, source, prop, caseless) {
    if (utils$1.isPlainObject(target) && utils$1.isPlainObject(source)) {
      return utils$1.merge.call({ caseless }, target, source);
    } else if (utils$1.isPlainObject(source)) {
      return utils$1.merge({}, source);
    } else if (utils$1.isArray(source)) {
      return source.slice();
    }
    return source;
  }
  function mergeDeepProperties(a, b, prop, caseless) {
    if (!utils$1.isUndefined(b)) {
      return getMergedValue(a, b, prop, caseless);
    } else if (!utils$1.isUndefined(a)) {
      return getMergedValue(void 0, a, prop, caseless);
    }
  }
  function valueFromConfig2(a, b) {
    if (!utils$1.isUndefined(b)) {
      return getMergedValue(void 0, b);
    }
  }
  function defaultToConfig2(a, b) {
    if (!utils$1.isUndefined(b)) {
      return getMergedValue(void 0, b);
    } else if (!utils$1.isUndefined(a)) {
      return getMergedValue(void 0, a);
    }
  }
  function mergeDirectKeys(a, b, prop) {
    if (prop in config2) {
      return getMergedValue(a, b);
    } else if (prop in config1) {
      return getMergedValue(void 0, a);
    }
  }
  const mergeMap = {
    url: valueFromConfig2,
    method: valueFromConfig2,
    data: valueFromConfig2,
    baseURL: defaultToConfig2,
    transformRequest: defaultToConfig2,
    transformResponse: defaultToConfig2,
    paramsSerializer: defaultToConfig2,
    timeout: defaultToConfig2,
    timeoutMessage: defaultToConfig2,
    withCredentials: defaultToConfig2,
    withXSRFToken: defaultToConfig2,
    adapter: defaultToConfig2,
    responseType: defaultToConfig2,
    xsrfCookieName: defaultToConfig2,
    xsrfHeaderName: defaultToConfig2,
    onUploadProgress: defaultToConfig2,
    onDownloadProgress: defaultToConfig2,
    decompress: defaultToConfig2,
    maxContentLength: defaultToConfig2,
    maxBodyLength: defaultToConfig2,
    beforeRedirect: defaultToConfig2,
    transport: defaultToConfig2,
    httpAgent: defaultToConfig2,
    httpsAgent: defaultToConfig2,
    cancelToken: defaultToConfig2,
    socketPath: defaultToConfig2,
    responseEncoding: defaultToConfig2,
    validateStatus: mergeDirectKeys,
    headers: (a, b, prop) => mergeDeepProperties(headersToObject(a), headersToObject(b), prop, true)
  };
  utils$1.forEach(Object.keys(Object.assign({}, config1, config2)), function computeConfigValue(prop) {
    const merge2 = mergeMap[prop] || mergeDeepProperties;
    const configValue = merge2(config1[prop], config2[prop], prop);
    utils$1.isUndefined(configValue) && merge2 !== mergeDirectKeys || (config[prop] = configValue);
  });
  return config;
}
const resolveConfig = (config) => {
  const newConfig = mergeConfig$1({}, config);
  let { data, withXSRFToken, xsrfHeaderName, xsrfCookieName, headers, auth } = newConfig;
  newConfig.headers = headers = AxiosHeaders$1.from(headers);
  newConfig.url = buildURL(buildFullPath(newConfig.baseURL, newConfig.url, newConfig.allowAbsoluteUrls), config.params, config.paramsSerializer);
  if (auth) {
    headers.set(
      "Authorization",
      "Basic " + btoa((auth.username || "") + ":" + (auth.password ? unescape(encodeURIComponent(auth.password)) : ""))
    );
  }
  let contentType;
  if (utils$1.isFormData(data)) {
    if (platform.hasStandardBrowserEnv || platform.hasStandardBrowserWebWorkerEnv) {
      headers.setContentType(void 0);
    } else if ((contentType = headers.getContentType()) !== false) {
      const [type, ...tokens] = contentType ? contentType.split(";").map((token) => token.trim()).filter(Boolean) : [];
      headers.setContentType([type || "multipart/form-data", ...tokens].join("; "));
    }
  }
  if (platform.hasStandardBrowserEnv) {
    withXSRFToken && utils$1.isFunction(withXSRFToken) && (withXSRFToken = withXSRFToken(newConfig));
    if (withXSRFToken || withXSRFToken !== false && isURLSameOrigin(newConfig.url)) {
      const xsrfValue = xsrfHeaderName && xsrfCookieName && cookies.read(xsrfCookieName);
      if (xsrfValue) {
        headers.set(xsrfHeaderName, xsrfValue);
      }
    }
  }
  return newConfig;
};
const isXHRAdapterSupported = typeof XMLHttpRequest !== "undefined";
const xhrAdapter = isXHRAdapterSupported && function(config) {
  return new Promise(function dispatchXhrRequest(resolve, reject) {
    const _config = resolveConfig(config);
    let requestData = _config.data;
    const requestHeaders = AxiosHeaders$1.from(_config.headers).normalize();
    let { responseType, onUploadProgress, onDownloadProgress } = _config;
    let onCanceled;
    let uploadThrottled, downloadThrottled;
    let flushUpload, flushDownload;
    function done() {
      flushUpload && flushUpload();
      flushDownload && flushDownload();
      _config.cancelToken && _config.cancelToken.unsubscribe(onCanceled);
      _config.signal && _config.signal.removeEventListener("abort", onCanceled);
    }
    let request = new XMLHttpRequest();
    request.open(_config.method.toUpperCase(), _config.url, true);
    request.timeout = _config.timeout;
    function onloadend() {
      if (!request) {
        return;
      }
      const responseHeaders = AxiosHeaders$1.from(
        "getAllResponseHeaders" in request && request.getAllResponseHeaders()
      );
      const responseData = !responseType || responseType === "text" || responseType === "json" ? request.responseText : request.response;
      const response = {
        data: responseData,
        status: request.status,
        statusText: request.statusText,
        headers: responseHeaders,
        config,
        request
      };
      settle(function _resolve(value) {
        resolve(value);
        done();
      }, function _reject(err) {
        reject(err);
        done();
      }, response);
      request = null;
    }
    if ("onloadend" in request) {
      request.onloadend = onloadend;
    } else {
      request.onreadystatechange = function handleLoad() {
        if (!request || request.readyState !== 4) {
          return;
        }
        if (request.status === 0 && !(request.responseURL && request.responseURL.indexOf("file:") === 0)) {
          return;
        }
        setTimeout(onloadend);
      };
    }
    request.onabort = function handleAbort() {
      if (!request) {
        return;
      }
      reject(new AxiosError$1("Request aborted", AxiosError$1.ECONNABORTED, config, request));
      request = null;
    };
    request.onerror = function handleError() {
      reject(new AxiosError$1("Network Error", AxiosError$1.ERR_NETWORK, config, request));
      request = null;
    };
    request.ontimeout = function handleTimeout() {
      let timeoutErrorMessage = _config.timeout ? "timeout of " + _config.timeout + "ms exceeded" : "timeout exceeded";
      const transitional2 = _config.transitional || transitionalDefaults;
      if (_config.timeoutErrorMessage) {
        timeoutErrorMessage = _config.timeoutErrorMessage;
      }
      reject(new AxiosError$1(
        timeoutErrorMessage,
        transitional2.clarifyTimeoutError ? AxiosError$1.ETIMEDOUT : AxiosError$1.ECONNABORTED,
        config,
        request
      ));
      request = null;
    };
    requestData === void 0 && requestHeaders.setContentType(null);
    if ("setRequestHeader" in request) {
      utils$1.forEach(requestHeaders.toJSON(), function setRequestHeader(val, key) {
        request.setRequestHeader(key, val);
      });
    }
    if (!utils$1.isUndefined(_config.withCredentials)) {
      request.withCredentials = !!_config.withCredentials;
    }
    if (responseType && responseType !== "json") {
      request.responseType = _config.responseType;
    }
    if (onDownloadProgress) {
      [downloadThrottled, flushDownload] = progressEventReducer(onDownloadProgress, true);
      request.addEventListener("progress", downloadThrottled);
    }
    if (onUploadProgress && request.upload) {
      [uploadThrottled, flushUpload] = progressEventReducer(onUploadProgress);
      request.upload.addEventListener("progress", uploadThrottled);
      request.upload.addEventListener("loadend", flushUpload);
    }
    if (_config.cancelToken || _config.signal) {
      onCanceled = (cancel) => {
        if (!request) {
          return;
        }
        reject(!cancel || cancel.type ? new CanceledError$1(null, config, request) : cancel);
        request.abort();
        request = null;
      };
      _config.cancelToken && _config.cancelToken.subscribe(onCanceled);
      if (_config.signal) {
        _config.signal.aborted ? onCanceled() : _config.signal.addEventListener("abort", onCanceled);
      }
    }
    const protocol = parseProtocol(_config.url);
    if (protocol && platform.protocols.indexOf(protocol) === -1) {
      reject(new AxiosError$1("Unsupported protocol " + protocol + ":", AxiosError$1.ERR_BAD_REQUEST, config));
      return;
    }
    request.send(requestData || null);
  });
};
const composeSignals = (signals, timeout) => {
  const { length } = signals = signals ? signals.filter(Boolean) : [];
  if (timeout || length) {
    let controller = new AbortController();
    let aborted;
    const onabort = function(reason) {
      if (!aborted) {
        aborted = true;
        unsubscribe();
        const err = reason instanceof Error ? reason : this.reason;
        controller.abort(err instanceof AxiosError$1 ? err : new CanceledError$1(err instanceof Error ? err.message : err));
      }
    };
    let timer = timeout && setTimeout(() => {
      timer = null;
      onabort(new AxiosError$1(`timeout ${timeout} of ms exceeded`, AxiosError$1.ETIMEDOUT));
    }, timeout);
    const unsubscribe = () => {
      if (signals) {
        timer && clearTimeout(timer);
        timer = null;
        signals.forEach((signal2) => {
          signal2.unsubscribe ? signal2.unsubscribe(onabort) : signal2.removeEventListener("abort", onabort);
        });
        signals = null;
      }
    };
    signals.forEach((signal2) => signal2.addEventListener("abort", onabort));
    const { signal } = controller;
    signal.unsubscribe = () => utils$1.asap(unsubscribe);
    return signal;
  }
};
const streamChunk = function* (chunk, chunkSize) {
  let len = chunk.byteLength;
  if (len < chunkSize) {
    yield chunk;
    return;
  }
  let pos = 0;
  let end;
  while (pos < len) {
    end = pos + chunkSize;
    yield chunk.slice(pos, end);
    pos = end;
  }
};
const readBytes = async function* (iterable, chunkSize) {
  for await (const chunk of readStream(iterable)) {
    yield* streamChunk(chunk, chunkSize);
  }
};
const readStream = async function* (stream) {
  if (stream[Symbol.asyncIterator]) {
    yield* stream;
    return;
  }
  const reader = stream.getReader();
  try {
    for (; ; ) {
      const { done, value } = await reader.read();
      if (done) {
        break;
      }
      yield value;
    }
  } finally {
    await reader.cancel();
  }
};
const trackStream = (stream, chunkSize, onProgress, onFinish) => {
  const iterator2 = readBytes(stream, chunkSize);
  let bytes = 0;
  let done;
  let _onFinish = (e) => {
    if (!done) {
      done = true;
      onFinish && onFinish(e);
    }
  };
  return new ReadableStream({
    async pull(controller) {
      try {
        const { done: done2, value } = await iterator2.next();
        if (done2) {
          _onFinish();
          controller.close();
          return;
        }
        let len = value.byteLength;
        if (onProgress) {
          let loadedBytes = bytes += len;
          onProgress(loadedBytes);
        }
        controller.enqueue(new Uint8Array(value));
      } catch (err) {
        _onFinish(err);
        throw err;
      }
    },
    cancel(reason) {
      _onFinish(reason);
      return iterator2.return();
    }
  }, {
    highWaterMark: 2
  });
};
const isFetchSupported = typeof fetch === "function" && typeof Request === "function" && typeof Response === "function";
const isReadableStreamSupported = isFetchSupported && typeof ReadableStream === "function";
const encodeText = isFetchSupported && (typeof TextEncoder === "function" ? /* @__PURE__ */ ((encoder) => (str) => encoder.encode(str))(new TextEncoder()) : async (str) => new Uint8Array(await new Response(str).arrayBuffer()));
const test = (fn, ...args) => {
  try {
    return !!fn(...args);
  } catch (e) {
    return false;
  }
};
const supportsRequestStream = isReadableStreamSupported && test(() => {
  let duplexAccessed = false;
  const hasContentType = new Request(platform.origin, {
    body: new ReadableStream(),
    method: "POST",
    get duplex() {
      duplexAccessed = true;
      return "half";
    }
  }).headers.has("Content-Type");
  return duplexAccessed && !hasContentType;
});
const DEFAULT_CHUNK_SIZE = 64 * 1024;
const supportsResponseStream = isReadableStreamSupported && test(() => utils$1.isReadableStream(new Response("").body));
const resolvers = {
  stream: supportsResponseStream && ((res) => res.body)
};
isFetchSupported && ((res) => {
  ["text", "arrayBuffer", "blob", "formData", "stream"].forEach((type) => {
    !resolvers[type] && (resolvers[type] = utils$1.isFunction(res[type]) ? (res2) => res2[type]() : (_, config) => {
      throw new AxiosError$1(`Response type '${type}' is not supported`, AxiosError$1.ERR_NOT_SUPPORT, config);
    });
  });
})(new Response());
const getBodyLength = async (body) => {
  if (body == null) {
    return 0;
  }
  if (utils$1.isBlob(body)) {
    return body.size;
  }
  if (utils$1.isSpecCompliantForm(body)) {
    const _request = new Request(platform.origin, {
      method: "POST",
      body
    });
    return (await _request.arrayBuffer()).byteLength;
  }
  if (utils$1.isArrayBufferView(body) || utils$1.isArrayBuffer(body)) {
    return body.byteLength;
  }
  if (utils$1.isURLSearchParams(body)) {
    body = body + "";
  }
  if (utils$1.isString(body)) {
    return (await encodeText(body)).byteLength;
  }
};
const resolveBodyLength = async (headers, body) => {
  const length = utils$1.toFiniteNumber(headers.getContentLength());
  return length == null ? getBodyLength(body) : length;
};
const fetchAdapter = isFetchSupported && (async (config) => {
  let {
    url,
    method,
    data,
    signal,
    cancelToken,
    timeout,
    onDownloadProgress,
    onUploadProgress,
    responseType,
    headers,
    withCredentials = "same-origin",
    fetchOptions
  } = resolveConfig(config);
  responseType = responseType ? (responseType + "").toLowerCase() : "text";
  let composedSignal = composeSignals([signal, cancelToken && cancelToken.toAbortSignal()], timeout);
  let request;
  const unsubscribe = composedSignal && composedSignal.unsubscribe && (() => {
    composedSignal.unsubscribe();
  });
  let requestContentLength;
  try {
    if (onUploadProgress && supportsRequestStream && method !== "get" && method !== "head" && (requestContentLength = await resolveBodyLength(headers, data)) !== 0) {
      let _request = new Request(url, {
        method: "POST",
        body: data,
        duplex: "half"
      });
      let contentTypeHeader;
      if (utils$1.isFormData(data) && (contentTypeHeader = _request.headers.get("content-type"))) {
        headers.setContentType(contentTypeHeader);
      }
      if (_request.body) {
        const [onProgress, flush] = progressEventDecorator(
          requestContentLength,
          progressEventReducer(asyncDecorator(onUploadProgress))
        );
        data = trackStream(_request.body, DEFAULT_CHUNK_SIZE, onProgress, flush);
      }
    }
    if (!utils$1.isString(withCredentials)) {
      withCredentials = withCredentials ? "include" : "omit";
    }
    const isCredentialsSupported = "credentials" in Request.prototype;
    request = new Request(url, {
      ...fetchOptions,
      signal: composedSignal,
      method: method.toUpperCase(),
      headers: headers.normalize().toJSON(),
      body: data,
      duplex: "half",
      credentials: isCredentialsSupported ? withCredentials : void 0
    });
    let response = await fetch(request);
    const isStreamResponse = supportsResponseStream && (responseType === "stream" || responseType === "response");
    if (supportsResponseStream && (onDownloadProgress || isStreamResponse && unsubscribe)) {
      const options = {};
      ["status", "statusText", "headers"].forEach((prop) => {
        options[prop] = response[prop];
      });
      const responseContentLength = utils$1.toFiniteNumber(response.headers.get("content-length"));
      const [onProgress, flush] = onDownloadProgress && progressEventDecorator(
        responseContentLength,
        progressEventReducer(asyncDecorator(onDownloadProgress), true)
      ) || [];
      response = new Response(
        trackStream(response.body, DEFAULT_CHUNK_SIZE, onProgress, () => {
          flush && flush();
          unsubscribe && unsubscribe();
        }),
        options
      );
    }
    responseType = responseType || "text";
    let responseData = await resolvers[utils$1.findKey(resolvers, responseType) || "text"](response, config);
    !isStreamResponse && unsubscribe && unsubscribe();
    return await new Promise((resolve, reject) => {
      settle(resolve, reject, {
        data: responseData,
        headers: AxiosHeaders$1.from(response.headers),
        status: response.status,
        statusText: response.statusText,
        config,
        request
      });
    });
  } catch (err) {
    unsubscribe && unsubscribe();
    if (err && err.name === "TypeError" && /Load failed|fetch/i.test(err.message)) {
      throw Object.assign(
        new AxiosError$1("Network Error", AxiosError$1.ERR_NETWORK, config, request),
        {
          cause: err.cause || err
        }
      );
    }
    throw AxiosError$1.from(err, err && err.code, config, request);
  }
});
const knownAdapters = {
  http: httpAdapter,
  xhr: xhrAdapter,
  fetch: fetchAdapter
};
utils$1.forEach(knownAdapters, (fn, value) => {
  if (fn) {
    try {
      Object.defineProperty(fn, "name", { value });
    } catch (e) {
    }
    Object.defineProperty(fn, "adapterName", { value });
  }
});
const renderReason = (reason) => `- ${reason}`;
const isResolvedHandle = (adapter) => utils$1.isFunction(adapter) || adapter === null || adapter === false;
const adapters = {
  getAdapter: (adapters2) => {
    adapters2 = utils$1.isArray(adapters2) ? adapters2 : [adapters2];
    const { length } = adapters2;
    let nameOrAdapter;
    let adapter;
    const rejectedReasons = {};
    for (let i = 0; i < length; i++) {
      nameOrAdapter = adapters2[i];
      let id2;
      adapter = nameOrAdapter;
      if (!isResolvedHandle(nameOrAdapter)) {
        adapter = knownAdapters[(id2 = String(nameOrAdapter)).toLowerCase()];
        if (adapter === void 0) {
          throw new AxiosError$1(`Unknown adapter '${id2}'`);
        }
      }
      if (adapter) {
        break;
      }
      rejectedReasons[id2 || "#" + i] = adapter;
    }
    if (!adapter) {
      const reasons = Object.entries(rejectedReasons).map(
        ([id2, state]) => `adapter ${id2} ` + (state === false ? "is not supported by the environment" : "is not available in the build")
      );
      let s = length ? reasons.length > 1 ? "since :\n" + reasons.map(renderReason).join("\n") : " " + renderReason(reasons[0]) : "as no adapter specified";
      throw new AxiosError$1(
        `There is no suitable adapter to dispatch the request ` + s,
        "ERR_NOT_SUPPORT"
      );
    }
    return adapter;
  },
  adapters: knownAdapters
};
function throwIfCancellationRequested(config) {
  if (config.cancelToken) {
    config.cancelToken.throwIfRequested();
  }
  if (config.signal && config.signal.aborted) {
    throw new CanceledError$1(null, config);
  }
}
function dispatchRequest(config) {
  throwIfCancellationRequested(config);
  config.headers = AxiosHeaders$1.from(config.headers);
  config.data = transformData.call(
    config,
    config.transformRequest
  );
  if (["post", "put", "patch"].indexOf(config.method) !== -1) {
    config.headers.setContentType("application/x-www-form-urlencoded", false);
  }
  const adapter = adapters.getAdapter(config.adapter || defaults.adapter);
  return adapter(config).then(function onAdapterResolution(response) {
    throwIfCancellationRequested(config);
    response.data = transformData.call(
      config,
      config.transformResponse,
      response
    );
    response.headers = AxiosHeaders$1.from(response.headers);
    return response;
  }, function onAdapterRejection(reason) {
    if (!isCancel$1(reason)) {
      throwIfCancellationRequested(config);
      if (reason && reason.response) {
        reason.response.data = transformData.call(
          config,
          config.transformResponse,
          reason.response
        );
        reason.response.headers = AxiosHeaders$1.from(reason.response.headers);
      }
    }
    return Promise.reject(reason);
  });
}
const VERSION$1 = "1.9.0";
const validators$1 = {};
["object", "boolean", "number", "function", "string", "symbol"].forEach((type, i) => {
  validators$1[type] = function validator2(thing) {
    return typeof thing === type || "a" + (i < 1 ? "n " : " ") + type;
  };
});
const deprecatedWarnings = {};
validators$1.transitional = function transitional(validator2, version, message) {
  function formatMessage(opt, desc) {
    return "[Axios v" + VERSION$1 + "] Transitional option '" + opt + "'" + desc + (message ? ". " + message : "");
  }
  return (value, opt, opts) => {
    if (validator2 === false) {
      throw new AxiosError$1(
        formatMessage(opt, " has been removed" + (version ? " in " + version : "")),
        AxiosError$1.ERR_DEPRECATED
      );
    }
    if (version && !deprecatedWarnings[opt]) {
      deprecatedWarnings[opt] = true;
      console.warn(
        formatMessage(
          opt,
          " has been deprecated since v" + version + " and will be removed in the near future"
        )
      );
    }
    return validator2 ? validator2(value, opt, opts) : true;
  };
};
validators$1.spelling = function spelling(correctSpelling) {
  return (value, opt) => {
    console.warn(`${opt} is likely a misspelling of ${correctSpelling}`);
    return true;
  };
};
function assertOptions(options, schema, allowUnknown) {
  if (typeof options !== "object") {
    throw new AxiosError$1("options must be an object", AxiosError$1.ERR_BAD_OPTION_VALUE);
  }
  const keys = Object.keys(options);
  let i = keys.length;
  while (i-- > 0) {
    const opt = keys[i];
    const validator2 = schema[opt];
    if (validator2) {
      const value = options[opt];
      const result = value === void 0 || validator2(value, opt, options);
      if (result !== true) {
        throw new AxiosError$1("option " + opt + " must be " + result, AxiosError$1.ERR_BAD_OPTION_VALUE);
      }
      continue;
    }
    if (allowUnknown !== true) {
      throw new AxiosError$1("Unknown option " + opt, AxiosError$1.ERR_BAD_OPTION);
    }
  }
}
const validator = {
  assertOptions,
  validators: validators$1
};
const validators = validator.validators;
let Axios$1 = class Axios {
  constructor(instanceConfig) {
    this.defaults = instanceConfig || {};
    this.interceptors = {
      request: new InterceptorManager(),
      response: new InterceptorManager()
    };
  }
  /**
   * Dispatch a request
   *
   * @param {String|Object} configOrUrl The config specific for this request (merged with this.defaults)
   * @param {?Object} config
   *
   * @returns {Promise} The Promise to be fulfilled
   */
  async request(configOrUrl, config) {
    try {
      return await this._request(configOrUrl, config);
    } catch (err) {
      if (err instanceof Error) {
        let dummy = {};
        Error.captureStackTrace ? Error.captureStackTrace(dummy) : dummy = new Error();
        const stack = dummy.stack ? dummy.stack.replace(/^.+\n/, "") : "";
        try {
          if (!err.stack) {
            err.stack = stack;
          } else if (stack && !String(err.stack).endsWith(stack.replace(/^.+\n.+\n/, ""))) {
            err.stack += "\n" + stack;
          }
        } catch (e) {
        }
      }
      throw err;
    }
  }
  _request(configOrUrl, config) {
    if (typeof configOrUrl === "string") {
      config = config || {};
      config.url = configOrUrl;
    } else {
      config = configOrUrl || {};
    }
    config = mergeConfig$1(this.defaults, config);
    const { transitional: transitional2, paramsSerializer, headers } = config;
    if (transitional2 !== void 0) {
      validator.assertOptions(transitional2, {
        silentJSONParsing: validators.transitional(validators.boolean),
        forcedJSONParsing: validators.transitional(validators.boolean),
        clarifyTimeoutError: validators.transitional(validators.boolean)
      }, false);
    }
    if (paramsSerializer != null) {
      if (utils$1.isFunction(paramsSerializer)) {
        config.paramsSerializer = {
          serialize: paramsSerializer
        };
      } else {
        validator.assertOptions(paramsSerializer, {
          encode: validators.function,
          serialize: validators.function
        }, true);
      }
    }
    if (config.allowAbsoluteUrls !== void 0) ;
    else if (this.defaults.allowAbsoluteUrls !== void 0) {
      config.allowAbsoluteUrls = this.defaults.allowAbsoluteUrls;
    } else {
      config.allowAbsoluteUrls = true;
    }
    validator.assertOptions(config, {
      baseUrl: validators.spelling("baseURL"),
      withXsrfToken: validators.spelling("withXSRFToken")
    }, true);
    config.method = (config.method || this.defaults.method || "get").toLowerCase();
    let contextHeaders = headers && utils$1.merge(
      headers.common,
      headers[config.method]
    );
    headers && utils$1.forEach(
      ["delete", "get", "head", "post", "put", "patch", "common"],
      (method) => {
        delete headers[method];
      }
    );
    config.headers = AxiosHeaders$1.concat(contextHeaders, headers);
    const requestInterceptorChain = [];
    let synchronousRequestInterceptors = true;
    this.interceptors.request.forEach(function unshiftRequestInterceptors(interceptor) {
      if (typeof interceptor.runWhen === "function" && interceptor.runWhen(config) === false) {
        return;
      }
      synchronousRequestInterceptors = synchronousRequestInterceptors && interceptor.synchronous;
      requestInterceptorChain.unshift(interceptor.fulfilled, interceptor.rejected);
    });
    const responseInterceptorChain = [];
    this.interceptors.response.forEach(function pushResponseInterceptors(interceptor) {
      responseInterceptorChain.push(interceptor.fulfilled, interceptor.rejected);
    });
    let promise;
    let i = 0;
    let len;
    if (!synchronousRequestInterceptors) {
      const chain = [dispatchRequest.bind(this), void 0];
      chain.unshift.apply(chain, requestInterceptorChain);
      chain.push.apply(chain, responseInterceptorChain);
      len = chain.length;
      promise = Promise.resolve(config);
      while (i < len) {
        promise = promise.then(chain[i++], chain[i++]);
      }
      return promise;
    }
    len = requestInterceptorChain.length;
    let newConfig = config;
    i = 0;
    while (i < len) {
      const onFulfilled = requestInterceptorChain[i++];
      const onRejected = requestInterceptorChain[i++];
      try {
        newConfig = onFulfilled(newConfig);
      } catch (error) {
        onRejected.call(this, error);
        break;
      }
    }
    try {
      promise = dispatchRequest.call(this, newConfig);
    } catch (error) {
      return Promise.reject(error);
    }
    i = 0;
    len = responseInterceptorChain.length;
    while (i < len) {
      promise = promise.then(responseInterceptorChain[i++], responseInterceptorChain[i++]);
    }
    return promise;
  }
  getUri(config) {
    config = mergeConfig$1(this.defaults, config);
    const fullPath = buildFullPath(config.baseURL, config.url, config.allowAbsoluteUrls);
    return buildURL(fullPath, config.params, config.paramsSerializer);
  }
};
utils$1.forEach(["delete", "get", "head", "options"], function forEachMethodNoData(method) {
  Axios$1.prototype[method] = function(url, config) {
    return this.request(mergeConfig$1(config || {}, {
      method,
      url,
      data: (config || {}).data
    }));
  };
});
utils$1.forEach(["post", "put", "patch"], function forEachMethodWithData(method) {
  function generateHTTPMethod(isForm) {
    return function httpMethod(url, data, config) {
      return this.request(mergeConfig$1(config || {}, {
        method,
        headers: isForm ? {
          "Content-Type": "multipart/form-data"
        } : {},
        url,
        data
      }));
    };
  }
  Axios$1.prototype[method] = generateHTTPMethod();
  Axios$1.prototype[method + "Form"] = generateHTTPMethod(true);
});
let CancelToken$1 = class CancelToken {
  constructor(executor) {
    if (typeof executor !== "function") {
      throw new TypeError("executor must be a function.");
    }
    let resolvePromise;
    this.promise = new Promise(function promiseExecutor(resolve) {
      resolvePromise = resolve;
    });
    const token = this;
    this.promise.then((cancel) => {
      if (!token._listeners) return;
      let i = token._listeners.length;
      while (i-- > 0) {
        token._listeners[i](cancel);
      }
      token._listeners = null;
    });
    this.promise.then = (onfulfilled) => {
      let _resolve;
      const promise = new Promise((resolve) => {
        token.subscribe(resolve);
        _resolve = resolve;
      }).then(onfulfilled);
      promise.cancel = function reject() {
        token.unsubscribe(_resolve);
      };
      return promise;
    };
    executor(function cancel(message, config, request) {
      if (token.reason) {
        return;
      }
      token.reason = new CanceledError$1(message, config, request);
      resolvePromise(token.reason);
    });
  }
  /**
   * Throws a `CanceledError` if cancellation has been requested.
   */
  throwIfRequested() {
    if (this.reason) {
      throw this.reason;
    }
  }
  /**
   * Subscribe to the cancel signal
   */
  subscribe(listener) {
    if (this.reason) {
      listener(this.reason);
      return;
    }
    if (this._listeners) {
      this._listeners.push(listener);
    } else {
      this._listeners = [listener];
    }
  }
  /**
   * Unsubscribe from the cancel signal
   */
  unsubscribe(listener) {
    if (!this._listeners) {
      return;
    }
    const index = this._listeners.indexOf(listener);
    if (index !== -1) {
      this._listeners.splice(index, 1);
    }
  }
  toAbortSignal() {
    const controller = new AbortController();
    const abort = (err) => {
      controller.abort(err);
    };
    this.subscribe(abort);
    controller.signal.unsubscribe = () => this.unsubscribe(abort);
    return controller.signal;
  }
  /**
   * Returns an object that contains a new `CancelToken` and a function that, when called,
   * cancels the `CancelToken`.
   */
  static source() {
    let cancel;
    const token = new CancelToken(function executor(c) {
      cancel = c;
    });
    return {
      token,
      cancel
    };
  }
};
function spread$1(callback) {
  return function wrap(arr) {
    return callback.apply(null, arr);
  };
}
function isAxiosError$1(payload) {
  return utils$1.isObject(payload) && payload.isAxiosError === true;
}
const HttpStatusCode$1 = {
  Continue: 100,
  SwitchingProtocols: 101,
  Processing: 102,
  EarlyHints: 103,
  Ok: 200,
  Created: 201,
  Accepted: 202,
  NonAuthoritativeInformation: 203,
  NoContent: 204,
  ResetContent: 205,
  PartialContent: 206,
  MultiStatus: 207,
  AlreadyReported: 208,
  ImUsed: 226,
  MultipleChoices: 300,
  MovedPermanently: 301,
  Found: 302,
  SeeOther: 303,
  NotModified: 304,
  UseProxy: 305,
  Unused: 306,
  TemporaryRedirect: 307,
  PermanentRedirect: 308,
  BadRequest: 400,
  Unauthorized: 401,
  PaymentRequired: 402,
  Forbidden: 403,
  NotFound: 404,
  MethodNotAllowed: 405,
  NotAcceptable: 406,
  ProxyAuthenticationRequired: 407,
  RequestTimeout: 408,
  Conflict: 409,
  Gone: 410,
  LengthRequired: 411,
  PreconditionFailed: 412,
  PayloadTooLarge: 413,
  UriTooLong: 414,
  UnsupportedMediaType: 415,
  RangeNotSatisfiable: 416,
  ExpectationFailed: 417,
  ImATeapot: 418,
  MisdirectedRequest: 421,
  UnprocessableEntity: 422,
  Locked: 423,
  FailedDependency: 424,
  TooEarly: 425,
  UpgradeRequired: 426,
  PreconditionRequired: 428,
  TooManyRequests: 429,
  RequestHeaderFieldsTooLarge: 431,
  UnavailableForLegalReasons: 451,
  InternalServerError: 500,
  NotImplemented: 501,
  BadGateway: 502,
  ServiceUnavailable: 503,
  GatewayTimeout: 504,
  HttpVersionNotSupported: 505,
  VariantAlsoNegotiates: 506,
  InsufficientStorage: 507,
  LoopDetected: 508,
  NotExtended: 510,
  NetworkAuthenticationRequired: 511
};
Object.entries(HttpStatusCode$1).forEach(([key, value]) => {
  HttpStatusCode$1[value] = key;
});
function createInstance(defaultConfig) {
  const context = new Axios$1(defaultConfig);
  const instance = bind(Axios$1.prototype.request, context);
  utils$1.extend(instance, Axios$1.prototype, context, { allOwnKeys: true });
  utils$1.extend(instance, context, null, { allOwnKeys: true });
  instance.create = function create(instanceConfig) {
    return createInstance(mergeConfig$1(defaultConfig, instanceConfig));
  };
  return instance;
}
const axios = createInstance(defaults);
axios.Axios = Axios$1;
axios.CanceledError = CanceledError$1;
axios.CancelToken = CancelToken$1;
axios.isCancel = isCancel$1;
axios.VERSION = VERSION$1;
axios.toFormData = toFormData$1;
axios.AxiosError = AxiosError$1;
axios.Cancel = axios.CanceledError;
axios.all = function all(promises) {
  return Promise.all(promises);
};
axios.spread = spread$1;
axios.isAxiosError = isAxiosError$1;
axios.mergeConfig = mergeConfig$1;
axios.AxiosHeaders = AxiosHeaders$1;
axios.formToJSON = (thing) => formDataToJSON(utils$1.isHTMLForm(thing) ? new FormData(thing) : thing);
axios.getAdapter = adapters.getAdapter;
axios.HttpStatusCode = HttpStatusCode$1;
axios.default = axios;
const {
  Axios: Axios2,
  AxiosError,
  CanceledError,
  isCancel,
  CancelToken: CancelToken2,
  VERSION,
  all: all2,
  Cancel,
  isAxiosError,
  spread,
  toFormData,
  AxiosHeaders: AxiosHeaders2,
  HttpStatusCode,
  formToJSON,
  getAdapter,
  mergeConfig
} = axios;
class HHBaseAPI {
  constructor(ops, tokenValue) {
    __publicField(this, "_instance");
    this._instance = axios.create({
      baseURL: ops.baseURL,
      timeout: 1e3 * 60 * 10
    });
    this._instance.interceptors.request.use(
      (config) => {
        let token = tokenValue.getToken();
        if (!token) {
          if (globalThis && globalThis["localStorage"]) {
            token = globalThis["localStorage"].getItem("auth-token") || "";
          }
        }
        if (token) {
          config.headers = {
            ...config.headers,
            Authorization: `Bearer ${token}`
          };
        }
        return config;
      }
    );
    this._instance.interceptors.response.use(
      (response) => response,
      (error) => {
        var _a, _b;
        if (((_a = error.response) == null ? void 0 : _a.status) === 401) {
          (_b = ops.onUnauthorized) == null ? void 0 : _b.call(ops);
        }
        return Promise.reject(error);
      }
    );
  }
  get instance() {
    return this._instance;
  }
  async get(opts) {
    try {
      const { url, config, schema } = opts;
      const res = await this._instance.get(url, config);
      this.checkResponse(res);
      const data = schema.parse(res.data.result);
      return Result.ok(data);
    } catch (ex) {
      console.error(ex);
      return Result.err(new Error(`get ${opts.url} error`));
    }
  }
  async post(opts) {
    var _a, _b;
    try {
      const { url, config, schema } = opts;
      const res = await this._instance.post(url, opts.data, config);
      this.checkResponse(res);
      const data = schema.parse(res.data.result);
      return Result.ok(data);
    } catch (ex) {
      let msg = ((_b = (_a = ex.response) == null ? void 0 : _a.data) == null ? void 0 : _b.detail) || "";
      if (msg) {
        return Result.err(new Error(msg));
      }
      return Result.err(ex);
    }
  }
  async delete(opts) {
    var _a, _b;
    try {
      const { url, config, schema } = opts;
      const res = await this._instance.delete(url, config);
      this.checkResponse(res);
      const data = schema.parse(res.data.result);
      return Result.ok(data);
    } catch (ex) {
      let msg = ((_b = (_a = ex.response) == null ? void 0 : _a.data) == null ? void 0 : _b.detail) || "";
      if (msg) {
        return Result.err(new Error(msg));
      }
      return Result.err(ex);
    }
  }
  checkResponse(res) {
    console.log(res);
    if (res.status != 200) {
      throw new Error(res.data.msg);
    }
    if (res.data.code !== 0) {
      console.log(res.data);
      throw new BusinessError(res.data.code, res.data.msg, res.data.result);
    }
  }
}
class HHAPI extends HHBaseAPI {
  constructor(ops, token) {
    super(ops, token);
  }
  /// 获取知识点列表
  async listkPoints(req) {
    return this.get({ url: "/v1/kpoint/", config: { params: req }, schema: APIPointsResponse });
  }
  async listPointRoots(req) {
    return this.get({ url: "/v1/kpoint/root", config: { params: req }, schema: APIPointsRootResponse });
  }
  /**
   * 获取Flow授权
   */
  async getFlowTokens() {
    return this.get({ url: "/v1/langflow/get_token", schema: GetFlowTokenResponse });
  }
  /**
   * 获取Flow配置
   */
  async getFlowConfig() {
    return this.get({ url: "/v1/langflow/config", schema: GetFlowConfigResponse });
  }
  /**
   * 生成试题的知识点列表
   */
  async generateQuizKnowledges(req) {
    const res = await this.post({
      url: "/v1/qa/get_kpoint_list",
      data: {
        qa_id: Number(req.quizId)
      },
      config: {},
      schema: z.string()
    });
    return res.map((r) => JSON.parse(r));
  }
  async getUserInfo(req) {
    console.debug(req);
    return this.get({
      url: "/v1/userinfo",
      schema: GetUserInfoResponse
    });
  }
  /// 获取问题列表
  async listQuizzes(req) {
    const res = await this.get({
      url: "/v1/qa/",
      config: {
        params: {
          skip: req.skip,
          limit: req.limit,
          subject: req.subject,
          state: req.state,
          qa_type_id: req.quizTypeId
        }
      },
      schema: ListQuestionsRes
    });
    return res.map((r) => {
      return {
        items: r.items.map((item) => {
          return {
            kpoints: item.kpoints.map((kp) => ({ id: kp.id.toString(), name: kp.name, parent_id: kp.parent_id.toString() })),
            quiz: apiQuestionToQuiz(item.qa)
          };
        }),
        limit: r.limit,
        skip: r.skip,
        total: r.total
      };
    });
  }
  async getExamPaperConfig() {
    const res = await this.get({ url: "/v1/qa/config", schema: QAConfig });
    return res.map(qaConfigToExamPaperConfig);
  }
  /**
   * 获取七牛上传配置
   */
  async getQiNiuUploadConfig() {
    return this.get({ url: "/v1/qiniu/getQiniuConfig", schema: APIQiniuUploadConfig });
  }
  /**
   * 创建试卷
   */
  async createExamPaper(req) {
    const data = quizConfigToCreateExamPaperObject(req.config);
    return this.post({ url: "/v1/exampaper/create", data, config: {}, schema: CreateExamPaperResponse });
  }
  /**
   * 删除试卷
   */
  async deleteExamPaper(req) {
    return this.post({
      url: `/v1/exampaper/delete/${req.id}`,
      data: {},
      schema: z.any()
    });
  }
  /**
   * 更新试卷
   */
  async updateExamPaper(req) {
    const examPaper = quizConfigToCreateExamPaperObject(req.config);
    const res = await this.post({
      url: "/v1/exampaper/update",
      data: {
        exampaper_id: req.paperId,
        exampaper: examPaper
      },
      schema: z.any()
    });
    return res;
  }
  async listExamPapers(req) {
    if (!req.subject) {
      delete req.subject;
    }
    const res = await this.get({ url: "/v1/exampaper/", config: { params: req }, schema: ListExamPapersResponse });
    return res.map((data) => {
      return {
        total: data.total,
        tasks: data.exampapers.map((exampaper) => {
          return {
            quizConfig: examPaperToQuizConfig(exampaper),
            completedQuizCount: exampaper.qa_num_completed,
            quizCount: exampaper.qa_num,
            quizAdditions: [],
            updatedAt: exampaper.created_at,
            id: exampaper.id.toString()
          };
        })
      };
    });
  }
  async getExamPaperDetail(req) {
    const res = await this.get({
      url: `/v1/exampaper/detail/${req.id}`,
      schema: GetExamPaperDetailResponse
    });
    return res.map(examPaperDetailToTask);
  }
  async batchListExamPaperQuiz(req) {
    const res = await this.post({
      url: "/v1/exampaper/qa/batch_list",
      data: {
        ids: req.ids
      },
      schema: BatchListExamPaperQuizResponse
    });
    return res;
  }
  async addExamPaperQA(req) {
    var _a;
    const { taskId, quizAddition } = req;
    const res = await this.post({
      url: "/v1/exampaper/qa/add",
      data: {
        exampaper_id: Number(taskId),
        state: quizAddition.state,
        qa_id: (_a = quizAddition.quiz) == null ? void 0 : _a.id,
        type: quizAddition.type,
        bg_runflow: req.bgRunFlow ? 1 : 0,
        data_info: {
          content: quizAddition.prompt.text,
          images: quizAddition.prompt.files.map((f) => ({ src: f.fileUrl })),
          content_images: quizAddition.prompt.bodyFile ? [{ src: quizAddition.prompt.bodyFile.fileUrl }] : []
        }
      },
      config: {},
      schema: z.object({ exampaper_qa_id: z.number(), qa_inputrecord_id: z.number() })
    });
    return res.map((r) => ({ ...quizAddition, id: r.exampaper_qa_id.toString(), quizInputRecordId: r.qa_inputrecord_id.toString() }));
  }
  async deleteExamPaperQA(req) {
    const res = await this.delete({ url: `/v1/exampaper/qa/remove/${req.id}`, data: {}, config: {}, schema: z.any() });
    return res;
  }
  /**
   * 更新试卷
   * update_student_info: 如果是true表示会更新学情管理相关信息
   */
  async updateExamPaperQA(req) {
    var _a;
    const { taskId, quizAddition, update_student_info } = req;
    const res = await this.post({
      url: "/v1/exampaper/qa/update",
      data: {
        exampaper_id: taskId,
        exampaper_qa_id: quizAddition.id,
        is_student_relation: update_student_info ? 1 : 0,
        state: quizAddition.state,
        list_qa_ids: quizAddition.quizNumList,
        type: quizAddition.type,
        already_exists: quizAddition.alreadyExists ? 1 : 0,
        qa_id: ((_a = quizAddition.quiz) == null ? void 0 : _a.id) || 0
      },
      schema: z.any()
    });
    return res.map(() => quizAddition);
  }
  /**
   * 上传文件到七牛云,返回图片URL以及id
   */
  async uploadMedia(file) {
    const config = (await this.getQiNiuUploadConfig()).unwrap();
    let data;
    let headers = {};
    const formDdata = new FormData();
    formDdata.append("file", file);
    formDdata.append("token", config.token);
    formDdata.append("filename", file.name);
    data = formDdata;
    headers = { "Content-Type": "multipart/form-data" };
    const res = await this.instance.post("https://upload-z2.qiniup.com/", data, {
      headers: {
        ...headers
      }
    });
    return Result.ok({
      id: res.data.key,
      fileUrl: `${config.domain}/${res.data.key}`,
      coverUrl: `${config.domain}/${res.data.key}`,
      type: FileType.Image
    });
  }
  async listTextBooks(req) {
    const res = await this.get({
      url: "/v1/textbook/book_info",
      config: {
        params: { ...req }
      },
      schema: ListTextBooksRes
    });
    return res.map((r) => r.catalog_data.map(textBookToBook));
  }
  // 获取Course列表
  async listCourses(req) {
    return this.get({
      url: "/v1/textbook/text_courses",
      config: {
        params: {
          subject: req.subject,
          skip: 0,
          limit: 1e3
        }
      },
      schema: ListCoursesResponse
    });
  }
  // 获取Textbooks
  async listTextbooks(req) {
    return this.get({
      url: "/v1/textbook/textbooks",
      config: {
        params: {
          course_id: req.course_id,
          skip: 0,
          limit: 1e3
        }
      },
      schema: ListTextbooksResponse
    });
  }
  async listCatalogs(req) {
    return this.get({
      url: "/v1/textbook/textbook_catalogs",
      config: {
        params: {
          textbook_id: req.textbook_id,
          skip: 0,
          limit: 1e3
        }
      },
      schema: ListCatalogsResponse
    });
  }
  async listCatalogsWithKpoints(req) {
    return this.get({
      url: "/v1/textbook/total_textbook_catalogs",
      config: {
        params: req
      },
      schema: ListCatalogsWithKpointsResponse
    });
  }
  async listExams(req) {
    const res = await this.get({
      url: "/v1/honghu/exam/student/stat/",
      config: {
        params: {
          skip: req.skip,
          limit: req.limit
        }
      },
      schema: ListExamsResponse
    });
    return res;
  }
  async getExamDetail(req) {
    const res = await this.get({
      url: "/v1/honghu/exam/student/stat/detail/",
      config: {
        params: {
          ep_no: req.ep_no,
          student_no: req.student_no
        }
      },
      schema: GetExamDetailResponse
    });
    return res;
  }
  async listStudentsExam(req) {
    const ListStudentResponseSchema = z.object({
      total: z.number(),
      items: z.array(z.object({
        class_id: z.number(),
        history_num_cross: z.coerce.number(),
        history_num_total: z.coerce.number(),
        student_nickname: z.string(),
        student_no: z.string()
      })),
      class_info: z.array(z.object({
        class_id: z.number(),
        class_name: z.string(),
        grade: z.string()
      }))
    });
    const res = await this.get({
      url: "/v1/honghu/exam/student/list",
      config: {
        params: {
          subject: req.subject
        }
      },
      schema: ListStudentResponseSchema
    });
    if (res.isErr()) {
      return Result.err(res.error);
    }
    const data = res.value;
    const studentExams = [];
    for (const d of data.items) {
      studentExams.push({
        student: {
          class_id: d.class_id.toString(),
          nickname: d.student_nickname,
          no: d.student_no
        },
        examInfo: {
          crossNum: d.history_num_cross,
          total: d.history_num_total
        }
      });
    }
    return Result.ok({
      exams: studentExams,
      classes: data.class_info.map((c) => ({ id: c.class_id.toString(), name: c.class_name, grade: c.grade })),
      total: data.total
    });
  }
  async listStudentKPoints(req) {
    const ResSchema = z.array(z.object({
      kpoints_id: z.number(),
      count: z.number(),
      name: z.string(),
      parent_id: z.number(),
      proficiency: ProficiencyEnum
    }));
    const res = await this.get({
      url: "/v1/honghu/exam/student/kpoints/cross",
      config: {
        params: {
          student_no: req.student_no,
          subject: req.subject,
          top_n: 20
        }
      },
      schema: ResSchema
    });
    return res.map((r) => {
      return {
        items: r.map((m) => ({
          kpoint: {
            id: m.kpoints_id.toString(),
            name: m.name,
            parent_id: m.parent_id.toString()
          },
          proficiency: m.proficiency
        }))
      };
    });
  }
  async getStudentInfo(req) {
    const ResAPI = z.object({
      class_id: z.number(),
      class_name: z.string(),
      id: z.number(),
      student_name: z.string(),
      student_number: z.string()
    });
    const res = await this.get({
      url: `/v1/honghu/exam/student/info/${req.stuNo}`,
      schema: ResAPI
    });
    return res.map((r) => ({
      class_id: r.class_id.toString(),
      nickname: r.student_name,
      no: r.student_number
    }));
  }
  async listStudentQuizzes(req) {
    const APIRes = z.object({
      qa_list: z.array(APIQuestion),
      kpoints_to_name: z.array(z.object({
        kpoints_id: z.number(),
        kpoints_name: z.string()
      })),
      qa_id_to_kpoints_id: z.array(z.object({
        qa_id: z.number(),
        kpoints_id: z.array(z.number())
      }))
    });
    const res = await this.get({
      url: "/v1/honghu/exam/student/qa/cross",
      config: {
        params: {
          student_no: req.student_no,
          subject: req.subject,
          skip: req.skip,
          limit: req.limit
        }
      },
      schema: APIRes
    });
    return res.map((r) => ({
      quizzsWithKnowledges: r.qa_list.map((qa) => {
        const knowledgePoints = [];
        const kid = r.qa_id_to_kpoints_id.find((x) => x.qa_id == qa.id);
        if (kid) {
          kid.kpoints_id.forEach((id2) => {
            const point = r.kpoints_to_name.find((x) => x.kpoints_id == id2);
            if (point) {
              knowledgePoints.push({ id: point.kpoints_id.toString(), name: point.kpoints_name, parent_id: "0" });
            }
          });
        }
        const d = {
          quiz: apiQuestionToQuiz(qa),
          knowledgePoints
        };
        return d;
      })
    }));
  }
  /**
  * 获取学生考试列表
  */
  async listStudentExamPapers(req) {
    console.log(req);
    const ListStudentExamPapersResponseSchema = z.object({
      limit: z.coerce.number(),
      skip: z.coerce.number(),
      items: z.array(z.object({
        // 试卷编号
        ep_no: z.string(),
        // 试卷标题
        exampaper_title: z.string(),
        // 总题数量
        num_total: z.coerce.number(),
        // 错误数量
        num_cross: z.coerce.number(),
        // 预估时间(分钟)
        estimated_time: z.coerce.number(),
        // 总分
        total_score: z.coerce.number(),
        // 得分
        student_score: z.coerce.number(),
        // 学生姓名
        student_nickname: z.string(),
        // 学生编号
        student_no: z.string(),
        // 难度
        difficulty: z.coerce.string(),
        // 考试时间
        exam_time: z.string(),
        // 知识点
        kpoints: z.array(z.object({ id: z.number(), name: z.string() }))
      }))
    });
    const res = await this.get({
      url: "/v1/honghu/exam/student/exampapers",
      config: {
        params: {
          student_no: req.student_no,
          subject: req.subject,
          skip: req.skip,
          limit: req.limit
        }
      },
      schema: ListStudentExamPapersResponseSchema
    });
    return res.map((data) => {
      return {
        items: data.items.map((item) => {
          const r = {
            student: {
              no: item.student_no,
              nickname: item.student_nickname,
              class_id: "0"
            },
            exam: {
              correctNum: item.num_total - item.num_cross,
              incorrectNum: item.num_cross,
              score: item.student_score,
              examTime: item.exam_time
            },
            examPaper: {
              score: item.total_score,
              no: item.ep_no,
              difficulty: item.difficulty,
              title: item.exampaper_title,
              estimatedTimeMinutes: item.estimated_time,
              quizNum: item.num_total,
              subject: "EN",
              sourceImageUrls: []
            },
            knowpoints: item.kpoints.map((x) => ({ id: x.id.toString(), name: x.name, parent_id: "" }))
          };
          return r;
        })
      };
    });
  }
  async getStudentExamPaperDetail(req) {
    const ResSchema = z.object({
      exampaper_info: z.object({
        num_total: z.number(),
        id: z.number(),
        ep_no: z.string(),
        exampaper_title: z.string(),
        subject: SubjectEnum,
        estimated_time: z.coerce.string(),
        student_score: z.number(),
        student_nickname: z.string(),
        student_no: z.string(),
        exam_time: z.string(),
        num_cross: z.number(),
        total_score: z.number(),
        difficulty: z.string(),
        kpoints: z.array(z.object({ id: z.number(), name: z.string() }))
      }),
      exampaper_stats: z.array(z.object({
        qa_id: z.number(),
        list_qa_ids: z.array(z.object({
          qa_no: z.string(),
          mark: z.string()
        }))
      })),
      qa_id_to_kpoints: z.record(z.array(z.number())),
      img_list: z.array(z.string()),
      qa_list: z.array(APIQuestion)
    });
    const res = await this.get({
      url: "/v1/honghu/exam/exampaper/detail",
      config: {
        params: {
          ep_no: req.ep_no,
          student_no: req.student_no
        }
      },
      schema: ResSchema
    });
    if (res.isErr()) {
      return Result.err(res.error);
    }
    const v = res.value;
    const knowledgePoints = v.exampaper_info.kpoints.map((p) => ({
      parent_id: "0",
      name: p.name,
      id: p.id.toString()
    }));
    const resData = {
      exam: {
        correctNum: v.exampaper_info.num_total - v.exampaper_info.num_cross,
        examTime: v.exampaper_info.exam_time,
        incorrectNum: v.exampaper_info.num_cross,
        score: v.exampaper_info.student_score
      },
      examPaper: {
        score: v.exampaper_info.total_score,
        difficulty: v.exampaper_info.difficulty,
        estimatedTimeMinutes: Number(v.exampaper_info.estimated_time),
        no: v.exampaper_info.ep_no,
        quizNum: v.exampaper_info.num_total,
        title: v.exampaper_info.exampaper_title,
        subject: v.exampaper_info.subject,
        sourceImageUrls: v.img_list
      },
      knowledgePoints,
      quizzes: v.qa_list.map((qa) => {
        const kpointIds = v.qa_id_to_kpoints[qa.id.toString()] || [];
        const kpoints = kpointIds.map((id2) => {
          return knowledgePoints.find((x) => x.id == id2.toString());
        }).filter((x) => !!x);
        let isCheck = false;
        const exam = v.exampaper_stats.find((x) => x.qa_id.toString() == qa.id.toString());
        if (exam) {
          if (exam.list_qa_ids.find((x) => x.mark == "❌")) {
            isCheck = false;
          } else {
            isCheck = true;
          }
        }
        return {
          quiz: apiQuestionToQuiz(qa),
          knowledgePoints: kpoints,
          isCheck
        };
      }),
      student: {
        no: v.exampaper_info.student_no,
        class_id: "0",
        nickname: v.exampaper_info.student_nickname
      }
    };
    return Result.ok(resData);
  }
  async listStudentQuizzesByKnowledges(req) {
    const res = await this.post({
      url: "v1/honghu/exam/student/kpoints/qa",
      data: {
        student_no: req.student_no,
        kpoints_ids: req.kpoint_ids
      },
      schema: z.array(z.object({ qa: APIQuestion, qa_kpoints_id_to_name: z.array(z.object({ id: z.number(), name: z.string() })) }))
    });
    return res.map((r) => ({
      quizzes: r.map((m) => ({
        quiz: apiQuestionToQuiz(m.qa),
        knowledgePoints: m.qa_kpoints_id_to_name.map((x) => ({ id: x.id.toString(), name: x.name, parent_id: "" }))
      }))
    }));
  }
  async getCozeToken() {
    const res = await this.post({
      url: "v1/userinfo/coze/token",
      data: {},
      schema: CozeToken
    });
    return res;
  }
}
function paperQAToTaskAddition(q, subject, knowledges, quiz) {
  var _a;
  const contentImg = q.data_info.content_images ? q.data_info.content_images[0] : null;
  return {
    id: q.id.toString(),
    prompt: {
      text: q.data_info.content,
      files: q.data_info.images.map((img) => ({ id: img.src, fileUrl: img.src, coverUrl: img.src, type: FileType.Image })),
      bodyFile: contentImg ? { id: contentImg.src, fileUrl: contentImg.src, coverUrl: contentImg.src, type: FileType.Image } : void 0
    },
    quizInputRecordId: ((_a = q.qa_inputrecord_id) == null ? void 0 : _a.toString()) || null,
    subject,
    knowledges,
    quizNumList: q.list_qa_ids,
    alreadyExists: q.already_exists == 0 ? false : true,
    state: q.state,
    type: q.type,
    quiz,
    splitQuiz: false
  };
}
function examPaperDetailToTask(detail) {
  const examqalist = detail.exampaper_qa_list;
  const qalist = detail.qa_list;
  return {
    quizConfig: examPaperToQuizConfig(detail.exampaper),
    quizAdditions: examqalist.map((q) => {
      const qa_id = q.qa_id;
      const qa = qalist.find((x) => x.id == qa_id);
      const knowledgePoints = [];
      for (const kd of detail.kpoints_detail) {
        if (kd.qa_id == q.qa_id) {
          if (!knowledgePoints.find((x) => x.id == kd.kpoints.id.toString())) {
            knowledgePoints.push({
              id: kd.kpoints.id.toString(),
              name: kd.kpoints.name.toString(),
              parent_id: kd.kpoints.parent_id.toString()
            });
          }
        }
      }
      const quiz = qa ? apiQuestionToQuiz(qa) : null;
      return paperQAToTaskAddition(q, detail.exampaper.subject, knowledgePoints, quiz);
    }),
    updatedAt: detail.exampaper.updated_at,
    completedQuizCount: detail.exampaper.qa_num_completed,
    quizCount: detail.exampaper.qa_num,
    id: detail.exampaper.id.toString()
  };
}
function examPaperToQuizConfig(exampaper) {
  return {
    paperFrom: exampaper.exampaper_source == "鸿鹄内部" ? "Internal" : "External",
    // paperNum: exampaper.exampaper_number,
    area: exampaper.district,
    examType: exampaper.source_type,
    scene: exampaper.apply_scene,
    grade: exampaper.grade_name,
    lession: exampaper.catalog_ids,
    subject: exampaper.subject,
    title: exampaper.title,
    year: exampaper.year.toString(),
    totalQuizCount: exampaper.qa_num_init,
    applyScope: exampaper.apply_scope || ""
  };
}
function quizConfigToCreateExamPaperObject(config) {
  let source = "";
  if (config.paperFrom == "Internal") {
    source = "鸿鹄内部";
  } else if (config.paperFrom == "External") {
    source = "外部";
  } else {
    throw new Error("试卷来源不支持");
  }
  return {
    subject: config.subject,
    apply_scene: config.scene || "",
    catalog_ids: config.lession || [],
    district: config.area || "",
    exampaper_source: source,
    grade_name: config.grade || "",
    source_type: config.examType || "",
    title: config.title,
    year: Number(config.year),
    qa_num_init: config.totalQuizCount,
    apply_scope: config.applyScope
  };
}
function apiQuestionToQuiz(qa) {
  var _a, _b;
  const sections = qa.sections_info.sections.map((section) => {
    const answers = section.answers.map((answer) => answer.values);
    let _ans = (answers.length > 0 ? answers[0] : []) || [];
    if (_ans.length == 0) {
      _ans = [""];
    }
    return {
      id: genId(),
      content: section.content,
      choices: section.choices.map((choice) => ({
        key: choice.value,
        uid: genId(),
        value: choice.desc
      })),
      type: section.sec_type,
      answers: _ans,
      solution: section.solution
    };
  });
  return {
    content: qa.content,
    id: qa.id.toString(),
    state: qa.state,
    rate: 4,
    quizTypeId: qa.qa_type_id,
    quizTypeName: qa.extra_info.qa_type_name,
    sourceFrom: qa.source_from,
    subject: qa.subject,
    difficulty: ((_a = qa.sections_info) == null ? void 0 : _a.difficulty_name) || "一般",
    imageUrls: [],
    knowledgePointIds: [],
    knowledgePoints: [],
    tags: ((_b = qa.source_info.tags) == null ? void 0 : _b.filter((x) => x.type).map((t) => ({ type: t.type, name: t.name, id: genId() }))) || [],
    sections,
    aiAnalysis: qa.extra_info.ai_q_analysis,
    updatedAt: qa.updated_at,
    source: qa.source_info
  };
}
function quizToAPIQuestion(quiz) {
  const kids = [];
  if (quiz.knowledgePointIds) {
    kids.push(...quiz.knowledgePointIds.map((kid) => Number(kid)));
  }
  const q = {
    content: quiz.content,
    qa_type_id: quiz.quizTypeId,
    subject: quiz.subject,
    source_info: { ...quiz.source, tags: quiz.tags.map((x) => ({ name: x.name, type: x.type })) },
    state: quiz.state,
    source_from: quiz.sourceFrom,
    kpoint_ids: kids,
    sections_info: {
      difficulty_name: quiz.difficulty,
      images: [],
      sections: quiz.sections.map((section) => {
        const sectionType = section.choices.length > 0 ? "single_choice" : "single_blank";
        return {
          content: section.content,
          answers: [{
            values: section.answers
          }],
          choices: section.choices.map((choice) => {
            return {
              desc: choice.value,
              value: choice.key
            };
          }),
          mark: { raw: "", value: "" },
          sec_type: sectionType,
          solution: section.solution
        };
      })
    }
  };
  return q;
}
function textBookToBook(qaBook) {
  const book = {
    id: qaBook.textbook_id.toString(),
    name: qaBook.textbook_name,
    units: qaBook.catalogs.map((qaUnit) => {
      return {
        id: qaUnit.id.toString(),
        name: qaUnit.name,
        lessions: qaUnit.children.map((qaLession) => {
          return {
            id: qaLession.id.toString(),
            name: qaLession.name
          };
        })
      };
    })
  };
  return book;
}
function qaConfigToExamPaperConfig(config) {
  const qaBooks = config.catalog_data;
  const books = qaBooks.map(textBookToBook);
  return {
    grades: config.grade_names,
    areas,
    cities,
    provices,
    examType: config.exam_types,
    scenes: config.application_scenarios,
    subjects: config.subjects,
    years: config.years,
    books,
    difficulties: [{ name: "简单(0.9-0.99)", value: "简单" }, { name: "较易(0.8-0.9)", value: "较易" }, { name: "一般(0.5-0.8)", value: "一般" }, { name: "较难(0.3-0.5)", value: "较难" }, { name: "困难(0-0.3)", value: "困难" }],
    quizTypes: config.qa_types,
    applyScopes: ["通用", "培优"]
  };
}
const FlowChoice = z.object({
  key: z.string(),
  value: z.string()
});
const FlowQuizAnswer = z.object({
  detail: z.string().or(z.null()).optional(),
  number: z.coerce.string().optional(),
  type: z.string(),
  value: z.string()
});
const FlowSection = z.object({
  // 题目对应的ID(卷子上)
  id: z.string().optional(),
  // 题干
  content: z.string().default(""),
  // 选项
  choices: z.array(FlowChoice).optional(),
  // 答案
  answer: FlowQuizAnswer.optional().or(FlowQuizAnswer.array())
});
const FlowQuiz = z.object({
  // 类型
  type: z.string().default(""),
  // 题干
  content: z.string().default(""),
  // 选项
  choices: z.array(FlowChoice).optional(),
  // 子题
  sections: z.array(FlowSection).optional(),
  // 答案
  answer: z.array(FlowQuizAnswer).or(FlowQuizAnswer).optional(),
  // 试题的ID (卷子上)
  id: z.string().optional()
});
z.object({
  qa: FlowQuiz
});
z.object({
  qa: FlowQuiz,
  knowledge: z.array(z.object({
    id: z.coerce.string(),
    name: z.string()
  })).optional()
});
z.array(z.object({
  question: z.string().optional(),
  answer: z.string().optional(),
  explanation: z.string().optional(),
  knowledge_points: z.array(z.string()).optional(),
  knowledge_list: z.array(z.object({ id: z.number(), name: z.string() })).optional()
}));
class QuizAPI extends HHBaseAPI {
  /**
   * 获取试题
   */
  async getQuiz(quizId) {
    const res = await this.get({ url: `/v1/qa/detail/${quizId}`, schema: QuestionDetail });
    return res.map((r) => {
      const quiz = apiQuestionToQuiz(r.qa);
      const kpoints = r.kpoints.map((k) => ({ isMain: k.is_main == 1, name: k.name, id: k.id.toString() }));
      quiz.knowledgePoints = kpoints;
      console.log(quiz, r.kpoints);
      return {
        quiz,
        kpoints: r.kpoints.map((kp) => ({ id: kp.id.toString(), name: kp.name, parent_id: kp.parent_id.toString(), is_main: kp.is_main == 1 ? true : false }))
      };
    });
  }
  /**
   * 新增试题
   */
  async addQuiz(quiz) {
    const question = quizToAPIQuestion(quiz);
    const res = await this.post({
      url: "/v1/qa/insert",
      data: {
        qa: {
          ...question,
          extra_info: {
            note: ""
          }
        },
        kpoint_ids: question.kpoint_ids
      },
      config: {},
      schema: APIAddQuestionRes
    });
    if (res.isOk()) {
      return Result.ok({ quizId: res.value.qa_id });
    } else {
      const err = res.error;
      if (err instanceof BusinessError) {
        if (err.code == 1) {
          const qid = err.object.qa_id;
          return Result.err(new DuplicatedQuizError(qid, err.object.fail_reason));
        }
      }
      return Result.err(err);
    }
  }
  /**
   * 更新试题
   */
  async updateQuiz(quiz) {
    const question = quizToAPIQuestion(quiz);
    const kpoints = quiz.knowledgePoints;
    await this.post({
      url: "/v1/qa/update",
      data: {
        qa_id: quiz.id,
        qa: {
          ...question,
          extra_info: {
            note: ""
          }
        },
        // kpoint_ids: question.kpoint_ids,
        kpoints: kpoints.map((x) => ({ id: x.id, is_main: x.isMain ? 1 : 0 }))
      },
      config: {},
      schema: z.any()
    });
    return Result.ok({ quizId: quiz.id });
  }
  /**
   * 记录问题的提示词
   */
  async addQuizPrompt(req) {
    this.post({
      url: "/v1/qa/input/insert",
      data: {
        qa_id: Number(req.qid),
        data: req.prompt
      },
      config: {},
      schema: z.void()
    });
  }
  /**
  * 记录学科网试题
  */
  async addXKWQuiz(req) {
    const res = await this.post({
      url: "/v1/qa/insert/xkw_info",
      data: req,
      config: {},
      schema: z.object({ xkw_info_id: z.number() })
    });
    return res;
  }
  /**
  * 更新学科网试题的quizId
  */
  async updateXKWQuizId(req) {
    const res = await this.post({
      url: "/v1/qa/update/xkw_info",
      data: {
        id: req.id,
        qa_id: req.qa_id
      },
      schema: z.void()
    });
    return res;
  }
  async getXKWQuizByXKWQuizId(req) {
    const res = await this.get({
      url: `/v1/qa/detail/xkw_info/xkw_qa_id/${req.xkwQuizId}`,
      schema: GetXKWQuizByXKWQuizIdResponse
    });
    return res;
  }
  /*
   * 获取录题记录
   */
  async listQuizRecords(req) {
    const res = await this.get({
      url: "/v1/qa/inputrecord",
      config: {
        params: {
          skip: req.skip,
          limit: req.limit,
          subject: req.subject,
          state: req.state
        }
      },
      schema: z.object({ inputrecord_list: z.array(QuizInputRecord) })
    });
    return res;
  }
  /**
   * 增加录题记录
   */
  async addQuizRecord(req) {
    const data = {
      subject: req.subject,
      from_type: req.fromType,
      bg_runflow: 1,
      failed_reason: "",
      flow_data: "",
      from_id: req.fromId,
      input_data: {
        content: req.prompt.text,
        content_images: (req.prompt.bodyFile ? [req.prompt.bodyFile] : []).map((m) => ({ src: m.fileUrl })),
        input_type: req.action,
        images: req.prompt.files.map((m) => ({ src: m.fileUrl })),
        extra: req.extra
      },
      qa_id: 0,
      state: "initial"
    };
    const res = await this.post({
      url: "/v1/qa/inputrecord/insert",
      data,
      schema: z.object({ qa_input_record_id: z.number() })
    });
    return res.map((r) => {
      return { id: r.qa_input_record_id.toString() };
    });
  }
  /**
   * 批量获取录题记录
   */
  async batchListQuizRecord(req) {
    const res = await this.post({
      url: "/v1/qa/inputrecord/batch_list",
      data: {
        ids: req.ids
      },
      schema: BatchListQuizRecordsResponse
    });
    return res;
  }
  async deleteQuizRecord(req) {
    const res = await this.delete({
      url: "/v1/qa/inputrecord/delete/" + req.id,
      data: {},
      schema: z.any()
    });
    return res;
  }
  async refundQuizRecord(req) {
    const res = await this.post({
      url: "/v1/qa/inputrecord/update",
      data: {
        id: req.id,
        state: "review-failed",
        review_comment: req.reason
      },
      schema: z.any()
    });
    return res;
  }
  /**
  * 重试录题
  */
  async retryQuizInput(req) {
    const res = await this.post({
      url: `/v1/qa/inputrecord/runflow`,
      schema: z.any(),
      data: {
        id: req.id,
        bg_runflow: req.runBackground ? 1 : 0
      }
    });
    return res;
  }
  async completeQuizInput(req) {
    const listRes = await this.batchListQuizRecord({ ids: [req.id] });
    let source = null;
    if (listRes.isOk()) {
      const list = listRes.value.inputrecord_list;
      if (list.length > 0) {
        source = list[0];
      }
    } else {
      return Result.err(listRes.error);
    }
    const res = await this.post({
      url: "/v1/qa/inputrecord/update",
      data: {
        ...source,
        state: "completed"
      },
      schema: z.any()
    });
    return res;
  }
  async listXKWQuizes(req) {
    const res = await this.get({
      url: "/v1/qa/batch/xkw_info",
      config: {
        params: {
          limit: req.limit,
          skip: req.skip,
          has_qa: 2
        }
      },
      schema: z.object({ items: z.array(XKWQuiz), total: z.number() })
    });
    if (res.isErr()) {
      return Result.err(res.error);
    }
    return Result.ok({
      xkwQuizzes: res.value.items,
      total: res.value.total
    });
  }
  async listQuizTags(req) {
    return this.get({
      url: "/v1/qa/tagdetail/search",
      config: {
        params: {
          subject: req.subject,
          name_like: req.name_like,
          skip: 0,
          limit: 50
        }
      },
      schema: ListQuizTagsResponse
    });
  }
}
class HHToken {
  constructor(token) {
    this.token = token;
  }
  setToken(token) {
    this.token = token;
  }
  getToken() {
    return this.token;
  }
  clearToken() {
    this.token = "";
  }
}
const Section = z.object({
  // 现有的试题QA
  qa: APIQuestion,
  // 预计分钟数
  estimated_minutes: z.number(),
  // 分数
  score: z.number()
});
const Group = z.object({
  // 分组ID
  id: z.number(),
  // 分组名词
  title: z.string(),
  // 试题列表
  // 试题列表有顺序，前端会直接按照数组的顺序渲染
  // 调整也直接修改数组
  sections: z.array(Section)
});
const Paper = z.object({
  id: z.number(),
  // 试卷标题
  title: z.string(),
  // 总分
  total_score: z.number(),
  // 预计分钟数
  // 卷子的预计耗时是所有章节下面的小题的耗时之和
  // estimated_minutes: z.number(),
  // 创建时间
  created_at: z.string(),
  // 试卷的难易度
  difficulty: z.string(),
  // 试卷的知识点
  kpoints: z.array(KnowledgePoint),
  // 分组
  // 有顺序，前端会直接按照数组顺序渲染
  // 调整也是直接修改数组
  groups: z.array(Group)
});
const FormulationState = z.enum(["initial", "draft", "published", "archived"]);
const FormulationStateMap = {
  archived: "已归档",
  draft: "草稿",
  initial: "初始中",
  published: "已发布"
};
const FormulationStateOptions = FormulationState.options.map((key) => ({
  value: key,
  label: FormulationStateMap[key]
}));
[{ value: "", label: "全部" }, ...FormulationStateOptions];
const FormulationType = z.enum(["manual", "agent"]);
const FormulationTypeMap = {
  agent: "AI组卷",
  manual: "手动组卷"
};
const FormulationTypeOptions = FormulationType.options.map((key) => ({
  label: FormulationTypeMap[key],
  value: key
}));
[{ value: "", label: "全部" }, ...FormulationTypeOptions];
const AgentExecution = z.object({
  // 当前的行为例如：AI正在组卷中
  action: z.string().optional(),
  description: z.string().optional(),
  // Agent处理状态，如果时completed表示已经处理完成
  // state: z.enum(['running', 'failed', 'completed']),
  // 如果Agent处理完了，会产出一个试题列表
  qa_ids: z.array(z.number()),
  // Agent的分析
  analysis: z.object({
    qa: z.array(z.object({
      // 试题ID
      qa_id: z.number(),
      // 选择原因
      select_reason: z.string()
    }))
  }).optional(),
  // Agent执行的步骤
  steps: z.array(z.object({
    // 步骤名词
    name: z.string(),
    // 开始处理时间
    start_time: z.string(),
    // 处理完成时间
    end_time: z.string().optional().nullable(),
    // 状态
    state: z.enum(["running", "failed", "completed"])
  }))
});
const ManualExecution = z.object({
  // 备选的试题列表
  qa_ids: z.array(z.number())
});
const ExamFrequency = z.enum(["one_year", "three_years", "five_years"]);
const SIKPoint = z.object({
  // 知识点基础信息
  kpoint: KnowledgePoint,
  // 学生与知识点信息
  sutdent_overview: z.object({
    // 熟练度
    mastery: z.number(),
    // 答题次数
    attempts: z.number(),
    // 正确次数
    correct: z.number(),
    // 最后一次答题日期
    last_attempt_date: z.string()
  }),
  // exam_frequency: {
  //   one_year: 1,
  //   three_years: 2,
  //   five_years: 4
  // }
  // 考频数据
  // exam_frequency: {'one_year': 10, 'three_years': 20, 'three_years': 50}
  exam_frequency: z.record(ExamFrequency, z.number()).optional().nullable()
});
const AgentStudentInfo = z.object({
  // 学号 [学号跟班级ID至少有一个不为空]
  student_no: z.string().optional().nullable(),
  // 班级ID [学号跟班级ID至少有一个不为空]
  class_id: z.number().optional().nullable(),
  // 开始时间[unix时间戳] 精确到秒(10位)
  start_time: z.number().optional().nullable(),
  // 结束时间[unix时间戳] 精确到秒(10位)
  end_time: z.number().optional().nullable(),
  // 学情知识点列表
  kpoints: z.array(SIKPoint.omit({ exam_frequency: true }))
});
const AgentFormulationSettings = z.object({
  // 知识点列表
  kpoints: z.array(KnowledgePoint),
  // 章节选题
  book_section_ids: z.array(z.number()),
  // 难度
  difficulty: z.string(),
  // 学情 可以为空
  student_info: AgentStudentInfo.optional().nullable(),
  // 试题配置
  questions: z.array(z.object({
    // 题目类型
    qa_type_id: z.number(),
    // 题目名称
    qa_type_name: z.string(),
    // 数量
    count: z.number()
  }))
});
const PaperAnalysis = z.object({
  // 打分: 组卷优异指数 [0-100]
  score: z.number().min(0).max(100),
  // 分析总结
  summary: z.string(),
  questions: z.array(z.object({
    // 试题ID
    qa_id: z.number(),
    // 试题分析
    analysis: z.string()
  }))
});
const Formulation = z.object({
  // 组卷ID
  id: z.number(),
  // 用于展示的一个ID, 前端会展示: 20250714-1024
  no: z.string(),
  // 学科
  subject: SubjectEnum,
  // 组卷名词
  title: z.string(),
  // 组卷类型：手动组卷或者Agent组卷
  type: FormulationType,
  // 如果type是agent时才有数据
  // 表示Agent组卷
  agent: AgentExecution.optional().nullable(),
  // 如果type是manual时才有的数据
  // 表示手动组卷
  manual: ManualExecution.optional().nullable(),
  // 如果type是agent时才有数据
  // Agent 组卷设置
  agent_formulation_settings: AgentFormulationSettings.optional().nullable(),
  // 试卷
  paper: Paper.optional().nullable(),
  // 试卷分析
  paper_analysis: PaperAnalysis.optional().nullable(),
  // 状态
  state: FormulationState,
  // 标签列表
  // 章节、分类、学情等等
  tags: z.array(z.string()),
  // 难易度
  difficulty: z.string(),
  // 预计耗时分钟数
  // 卷子上每道题的耗时数相加
  estimated_minutes: z.number(),
  // 试题数量
  qa_count: z.number(),
  // 知识点列表
  // 去掉，使用试卷里面的知识点
  kpoints: z.array(KnowledgePoint),
  // 创建时间
  created_at: z.string()
});
z.object({
  // kpoints: z.array(KnowledgePoint),
  // 学情知识点列表
  si_kpoints: z.array(SIKPoint)
});
z.object({
  // 试题列表
  items: z.array(z.object({ qa: APIQuestion, kpoints: z.array(KnowledgePoint) })),
  limit: z.number(),
  skip: z.number(),
  total: z.number()
});
z.object({
  formulation: Formulation
});
z.object({
  items: z.array(APIQuestion)
});
z.object({
  paperTitle: z.string(),
  groups: z.array(z.object({
    title: z.string(),
    // id可能为空，id为空是新分组
    id: z.number().optional(),
    sections: z.array(z.object({
      qa_id: z.number()
    }))
  }))
});
z.object({
  formulation: Formulation
});
z.object({
  formulation: Formulation
});
z.object({});
z.object({
  formulation: Formulation
});
z.null();
z.object({
  // 学科
  subject: SubjectEnum,
  // 组卷类型
  type: FormulationType.optional(),
  // 状态
  state: FormulationState.optional(),
  // 创建时间筛选，开始时间戳 精确到秒(10位)
  create_time_begin: z.coerce.number().optional(),
  // 创建时间筛选，结束时间戳 精确到秒(10位)
  create_time_end: z.coerce.number().optional(),
  limit: z.coerce.number(),
  skip: z.coerce.number()
});
z.object({
  items: z.array(Formulation),
  total: z.number()
});
z.object({
  formulation: Formulation
});
z.object({
  formulation: Formulation
});
let id = (/* @__PURE__ */ new Date()).getTime();
function genFormulation(p) {
  const type = p.type || "agent";
  const state = p.state || "initial";
  let agent = void 0;
  let manual = void 0;
  let agent_formulation_settings = void 0;
  if (type === "agent") {
    agent = {
      // state: "running",
      description: "AI is generating questions...",
      action: "generating_questions",
      qa_ids: [],
      analysis: {
        qa: []
      },
      steps: [
        {
          state: "completed",
          name: "task1",
          end_time: (/* @__PURE__ */ new Date()).toString(),
          start_time: (/* @__PURE__ */ new Date()).toString()
        },
        {
          state: "running",
          name: "task2",
          end_time: (/* @__PURE__ */ new Date()).toString(),
          start_time: (/* @__PURE__ */ new Date()).toString()
        }
      ]
    };
    agent_formulation_settings = {
      difficulty: "normal",
      book_section_ids: [],
      kpoints: [],
      questions: []
    };
  } else {
    manual = {
      qa_ids: []
    };
  }
  const difficultys = ["容易", "很难"];
  return {
    state,
    subject: "EN",
    title: "测试",
    created_at: (/* @__PURE__ */ new Date()).toISOString(),
    id: p.id || id++,
    qa_count: state == "initial" ? 0 : Math.floor(Math.random() * 100),
    no: `no-${id++}`,
    difficulty: difficultys[Math.floor(Math.random() * difficultys.length)] || "容易",
    estimated_minutes: Math.floor(Math.random() * 100),
    tags: ["章节知识点", "分类知识点"],
    kpoints: state == "initial" ? [] : [
      { name: "formal", id: "0", parent_id: "0" },
      { name: "negative", id: "0", parent_id: "0" }
    ],
    type,
    manual,
    agent,
    agent_formulation_settings,
    paper: void 0,
    paper_analysis: void 0,
    ...p
  };
}
const allFormulations = [
  genFormulation({ type: "agent", state: "initial", title: "AI组卷-初始中" }),
  genFormulation({ type: "agent", state: "draft", title: "AI组卷-草稿" }),
  genFormulation({ type: "agent", state: "published", title: "AI组卷-已发布" }),
  genFormulation({ type: "agent", state: "archived", title: "AI组卷-已归档" }),
  genFormulation({ type: "manual", state: "draft", title: "手动组卷-草稿" }),
  genFormulation({ type: "manual", state: "published", title: "手动组卷-已发布" }),
  genFormulation({ type: "manual", state: "archived", title: "手动组卷-已归档" })
];
const types = ["agent", "manual"];
const states = ["initial", "draft", "published", "archived"];
for (let i = allFormulations.length; i < 40; i++) {
  let type = types[Math.floor(Math.random() * types.length)];
  let state = states[Math.floor(Math.random() * states.length)];
  if (type === "manual" && state === "initial") {
    state = "draft";
  }
  allFormulations.push(genFormulation({
    type,
    state,
    title: `${type === "agent" ? "AI组卷" : "手动组卷"}-${i + 1}`
  }));
}
const LessionPeroidLevel = z.enum(["A", "B", "C"]);
const LessionPeriod = z.object({
  id: z.number(),
  // 等级
  level_name: LessionPeroidLevel,
  // 课本名称
  tb_version_name: z.string(),
  // 课本章节名称
  tb_catalogs_name: z.string(),
  // 课时名称
  name: z.string(),
  // 课本ID
  tb_version_id: z.number(),
  // 课本章节ID
  tb_catalogs_id: z.number()
});
const TestPaperItem = z.object({
  id: z.number(),
  state: z.string(),
  subject: SubjectEnum,
  year: z.number(),
  title: z.string(),
  created_at: z.string(),
  updated_at: z.string()
});
z.object({});
z.object({
  subject: SubjectEnum,
  level_name: LessionPeroidLevel,
  skip: z.number(),
  limit: z.number()
});
z.array(LessionPeriod);
z.object({
  year: z.number(),
  subject: SubjectEnum,
  course_id: z.number(),
  textbook_id: z.number(),
  textbook_catalog_id: z.number(),
  imgs: z.array(z.string())
});
z.object({
  qa_id: z.number()
});
z.object({
  subject: SubjectEnum,
  year: z.number().optional(),
  textbook_catalog_id: z.number().optional(),
  skip: z.number(),
  limit: z.number()
});
z.object({
  limit: z.number(),
  skip: z.number(),
  total: z.number(),
  items: z.array(TestPaperItem)
});
z.object({
  id: z.string()
});
const TestPaperQA = z.object({
  id: z.number(),
  // OCR 试题识别的数据
  data: z.object({
    images: z.array(z.object({ src: z.string() })),
    content: z.string()
  }),
  qa_id: z.number(),
  // 属于哪张卷子(去掉了，只使用 qidx排序)
  // pidx: z.number(),
  // 试题顺序
  qidx: z.number(),
  // 状态
  // #0" INITIAL"
  // #1" REVIEW"
  // #2" OCR_FAILED"
  // #3" OK"
  // 去掉了，只需要通过判断qa_id是否为0来判断是否已关联了
  // state: z.enum(['INITIAL', 'REVIEW', 'OCR_FAILED', 'OK']),
  created_at: z.string(),
  updated_at: z.string()
});
z.object({
  id: z.number(),
  // 年份
  year: z.number(),
  // 上传的图片列表
  imgs: z.array(z.string()),
  // 状态：显示用
  state: z.string(),
  // 创建时间
  created_at: z.string(),
  // 更新时间
  updated_at: z.string(),
  extra_info: z.object({
    course_name: z.string(),
    textbook_catalog_name: z.string(),
    textbook_catalog_parent_name: z.string(),
    textbook_name: z.string()
  }),
  // 试卷下面识别出来的的试题列表
  qas: z.array(TestPaperQA)
});
z.array(z.object({
  qa_id: z.number(),
  distance: z.number()
}));
var ErrCode = /* @__PURE__ */ ((ErrCode2) => {
  ErrCode2[ErrCode2["UN_AUTH"] = 1] = "UN_AUTH";
  ErrCode2[ErrCode2["BUSINESS_ERR"] = 10] = "BUSINESS_ERR";
  return ErrCode2;
})(ErrCode || {});
class Auth {
  // 获取 token
  async getToken() {
    const { auth } = await chrome.storage.local.get({ auth: { token: "" } });
    return auth.token;
  }
  // 设置 token
  setToken(token) {
    return chrome.storage.local.set({ auth: { token } });
  }
}
class AutoUpdater {
  constructor() {
    __publicField(this, "updateUrl", "https://hh-chrome-extension.letuzhixing.com/api/check-update");
    __publicField(this, "currentVersion", chrome.runtime.getManifest().version);
    __publicField(this, "cacheTimeout", 30 * 60 * 1e3);
  }
  // 30分钟缓存
  async checkForUpdates() {
    try {
      const env = await this.getCurrentEnv();
      console.log(`🔍 检查更新: 当前版本 ${this.currentVersion}, 环境 ${env}`);
      const response = await fetch(`${this.updateUrl}?env=${env}&current=${this.currentVersion}`);
      if (!response.ok) {
        throw new Error(`更新检查请求失败: ${response.status}`);
      }
      const updateInfo = await response.json();
      console.log("更新检查结果:", updateInfo);
      if (updateInfo.hasUpdate) {
        await this.notifyUserOfUpdate(updateInfo);
        console.log(`📢 发现新版本 ${updateInfo.version}, 已显示通知`);
      } else {
        console.log("✅ 当前已是最新版本:", updateInfo.message);
      }
    } catch (error) {
      console.error("❌ 更新检查失败:", error);
    }
  }
  async getCurrentEnv() {
    const { env } = await chrome.storage.local.get(["env"]);
    if (env) {
      return env;
    }
    const mode = "development";
    const envMap = {
      "development": "dev",
      "test": "test",
      "staging": "staging",
      "production": "prod"
    };
    const detectedEnv = envMap[mode] || "prod";
    await chrome.storage.local.set({ env: detectedEnv });
    console.log(`🔧 自动检测环境: ${mode} -> ${detectedEnv}`);
    return detectedEnv;
  }
  isNewerVersion(remoteVersion, currentVersion) {
    const remote = remoteVersion.split(".").map(Number);
    const current2 = currentVersion.split(".").map(Number);
    for (let i = 0; i < Math.max(remote.length, current2.length); i++) {
      const r = remote[i] || 0;
      const c = current2[i] || 0;
      if (r > c) return true;
      if (r < c) return false;
    }
    return false;
  }
  async notifyUserOfUpdate(updateInfo) {
    const { lastNotifiedVersion, lastNotificationTime } = await chrome.storage.local.get([
      "lastNotifiedVersion",
      "lastNotificationTime"
    ]);
    const now = Date.now();
    const oneHour = 60 * 60 * 1e3;
    if (lastNotifiedVersion === updateInfo.version && lastNotificationTime && now - lastNotificationTime < oneHour) {
      console.log(`⏭️ 跳过重复通知，版本 ${updateInfo.version} 在1小时内已通知过`);
      return;
    }
    const notificationId = `update-${Date.now()}`;
    const envName = this.getEnvDisplayName(updateInfo.env);
    await chrome.notifications.create(notificationId, {
      type: "basic",
      iconUrl: "/icons/icon48.png",
      title: "🚀 HH Quiz Scraper 有新版本",
      message: `发现${envName} v${updateInfo.version}，当前 v${updateInfo.currentVersion}`,
      buttons: [{ title: "立即更新" }, { title: "稍后提醒" }],
      requireInteraction: true
      // 需要用户交互才消失
    });
    await chrome.storage.local.set({
      pendingUpdate: updateInfo,
      lastUpdateCheck: now,
      lastNotifiedVersion: updateInfo.version,
      lastNotificationTime: now
    });
    console.log(`📢 显示更新通知: ${updateInfo.version} (${envName})`);
  }
  getEnvDisplayName(env) {
    const envNames = {
      "prod": "生产版",
      "test": "测试版",
      "staging": "预发版",
      "dev": "开发版"
    };
    return envNames[env] || "未知版本";
  }
  async handleUpdateClick() {
    const { pendingUpdate, blockingUpdateInfo } = await chrome.storage.local.get(["pendingUpdate", "blockingUpdateInfo"]);
    const updateInfo = blockingUpdateInfo || pendingUpdate;
    if (updateInfo) {
      const downloadUrl = updateInfo.downloadUrl || "https://hh-chrome-extension.letuzhixing.com/";
      await chrome.tabs.create({
        url: downloadUrl
      });
      console.log(`🔗 用户确认后打开更新页面: ${downloadUrl}`);
      await chrome.storage.local.set({
        lastUpdateClickTime: Date.now(),
        pendingUpdateRedirect: false
        // 清除待处理的跳转标记
      });
      const successNotificationId = `update-redirect-${Date.now()}`;
      await chrome.notifications.create(successNotificationId, {
        type: "basic",
        iconUrl: "/icons/icon48.png",
        title: "🔗 已打开更新页面",
        message: "请在新打开的页面中下载最新版本插件",
        requireInteraction: false
      });
      setTimeout(() => {
        chrome.notifications.clear(successNotificationId);
      }, 3e3);
    } else {
      console.warn("⚠️ 没有找到待更新信息");
    }
  }
  startPeriodicCheck() {
    const checkInterval = 6 * 60 * 60 * 1e3;
    setInterval(() => {
      console.log("⏰ 定时检查更新中...");
      this.checkForUpdates();
    }, checkInterval);
    setTimeout(() => {
      console.log("🚀 启动首次更新检查...");
      this.checkForUpdates();
    }, 5e3);
    console.log(`✅ 自动更新检查已启动，每${checkInterval / (60 * 60 * 1e3)}小时检查一次`);
  }
  // 手动检查更新的方法，供popup调用
  async manualCheckForUpdates() {
    try {
      return {
        hasUpdate: false,
        message: "当前已是最新版本"
      };
    } catch (error) {
      console.error("手动检查更新失败:", error);
      return {
        hasUpdate: false,
        message: "检查更新失败，请稍后重试"
      };
    }
  }
  // 版本检查守护函数 - 在执行操作前检查更新（暂时禁用缓存用于调试）
  async checkVersionBeforeOperation() {
    try {
      console.log(`🔍 [VERSION CHECK] Starting version check. Current version: ${this.currentVersion}`);
      console.log("⚠️ [DEBUG] 缓存已暂时禁用，每次都进行版本检查");
      const env = await this.getCurrentEnv();
      console.log(`🔍 [VERSION CHECK] Executing version check (cache bypassed). Environment: ${env}`);
      const requestUrl = `${this.updateUrl}?env=${env}&current=${this.currentVersion}&_t=${Date.now()}`;
      console.log(`🔍 [VERSION CHECK] Request URL: ${requestUrl}`);
      const response = await fetch(requestUrl);
      if (!response.ok) {
        console.warn(`⚠️ [VERSION CHECK] Version check failed (${response.status}), 为安全起见阻止操作`);
        return { canProceed: false };
      }
      const updateInfo = await response.json();
      console.log(`📊 [VERSION CHECK] API Response:`, updateInfo);
      if (updateInfo.hasUpdate) {
        console.log(`⚠️ [VERSION CHECK] Found new version ${updateInfo.version}, blocking operation`);
        await this.showUpdateBlockingDialog(updateInfo);
        return { canProceed: false, updateInfo };
      } else {
        console.log("✅ [VERSION CHECK] Version check passed, allowing operation to continue");
        return { canProceed: true };
      }
    } catch (error) {
      console.error("❌ [VERSION CHECK] 版本检查出错，为安全起见阻止操作:", error);
      return { canProceed: false };
    }
  }
  // 获取缓存的版本检查结果
  async getCachedVersionCheck() {
    try {
      const { versionCheckCache } = await chrome.storage.local.get(["versionCheckCache"]);
      if (!versionCheckCache) {
        return null;
      }
      const now = Date.now();
      const cacheAge = now - versionCheckCache.timestamp;
      if (cacheAge > this.cacheTimeout) {
        console.log("📋 版本检查缓存已过期");
        await chrome.storage.local.remove(["versionCheckCache"]);
        return null;
      }
      return {
        hasUpdate: versionCheckCache.hasUpdate,
        updateInfo: versionCheckCache.updateInfo
      };
    } catch (error) {
      console.error("获取版本检查缓存失败:", error);
      return null;
    }
  }
  // 缓存版本检查结果
  async cacheVersionCheck(updateInfo) {
    try {
      const cacheData = {
        hasUpdate: updateInfo.hasUpdate,
        updateInfo,
        timestamp: Date.now()
      };
      await chrome.storage.local.set({ versionCheckCache: cacheData });
      console.log(`📋 版本检查结果已缓存 (${this.cacheTimeout / (60 * 1e3)}分钟)`);
    } catch (error) {
      console.error("缓存版本检查结果失败:", error);
    }
  }
  // 显示阻止操作的更新对话框
  async showUpdateBlockingDialog(updateInfo) {
    const envName = this.getEnvDisplayName(updateInfo.env);
    try {
      const notificationId = `update-blocking-${Date.now()}`;
      await chrome.notifications.create(notificationId, {
        type: "basic",
        iconUrl: "/icons/icon48.png",
        title: "🚫 需要更新插件",
        message: `发现${envName} v${updateInfo.version}，请先更新后再使用。是否现在去更新？`,
        buttons: [{ title: "立即更新" }, { title: "稍后提醒" }],
        requireInteraction: true
      });
      await chrome.storage.local.set({
        operationBlocked: true,
        blockingUpdateInfo: updateInfo,
        blockingTime: Date.now(),
        pendingUpdateRedirect: true
        // 标记有待处理的更新跳转
      });
      console.log("🔔 已显示更新确认通知，等待用户选择");
    } catch (error) {
      console.error("显示更新对话框失败:", error);
    }
  }
}
globalThis.window = {};
main();
function delay(time) {
  return new Promise((resolve) => {
    setTimeout(() => resolve(), time);
  });
}
function main() {
  const hhToken = new HHToken("");
  const hhapi = new HHAPI({ baseURL: "https://devhhapi.letuzhixing.com/api" }, hhToken);
  const quizapi = new QuizAPI({ baseURL: "https://devhhapi.letuzhixing.com/api" }, hhToken);
  const auth = new Auth();
  const updater = new AutoUpdater();
  updater.startPeriodicCheck();
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    handleMessage(message, sender, sendResponse);
    return true;
  });
  chrome.notifications.onClicked.addListener((notificationId) => {
    if (notificationId.startsWith("update-")) {
      updater.handleUpdateClick();
    }
  });
  chrome.notifications.onButtonClicked.addListener(async (notificationId, buttonIndex) => {
    if (notificationId.startsWith("update-")) {
      if (buttonIndex === 0) {
        console.log("👤 用户选择立即更新");
        await updater.handleUpdateClick();
      } else if (buttonIndex === 1) {
        console.log("👤 用户选择稍后提醒");
        chrome.notifications.clear(notificationId);
      }
    }
  });
  async function handleMessage(message, _sender, sendResponse) {
    await delay(10);
    const action = message.action;
    switch (action) {
      case "fetch_image": {
        const versionCheck = await updater.checkVersionBeforeOperation();
        if (!versionCheck.canProceed) {
          sendResponse({
            success: false,
            error: { code: ErrCode.BUSINESS_ERR, msg: "需要更新插件版本才能继续使用" }
          });
          return;
        }
        checkAndSetAuth();
        handleFetchImageMessage(message, sendResponse);
        break;
      }
      case "add_quiz": {
        console.log("🔍 [VERSION CHECK] add_quiz action triggered");
        const versionCheck = await updater.checkVersionBeforeOperation();
        console.log("🔍 [VERSION CHECK] add_quiz result:", versionCheck);
        if (!versionCheck.canProceed) {
          console.log("🚫 [VERSION CHECK] add_quiz blocked due to update requirement");
          sendResponse({
            success: false,
            error: { code: ErrCode.BUSINESS_ERR, msg: "需要更新插件版本才能继续使用" }
          });
          return;
        }
        console.log("✅ [VERSION CHECK] add_quiz proceeding...");
        checkAndSetAuth();
        handleAddQuizMessage(message, sendResponse);
        break;
      }
      case "add_quiz_from_xkw": {
        const versionCheck = await updater.checkVersionBeforeOperation();
        if (!versionCheck.canProceed) {
          sendResponse({
            success: false,
            error: { code: ErrCode.BUSINESS_ERR, msg: "需要更新插件版本才能继续使用" }
          });
          return;
        }
        break;
      }
      case "is_logged_in": {
        handleIsLoggedInMessage(message, sendResponse);
        break;
      }
      case "set_token": {
        console.log("🔍 [VERSION CHECK] set_token action triggered");
        const versionCheck = await updater.checkVersionBeforeOperation();
        console.log("🔍 [VERSION CHECK] set_token result:", versionCheck);
        if (!versionCheck.canProceed) {
          console.log("🚫 [VERSION CHECK] set_token blocked due to update requirement");
          sendResponse({
            success: false,
            error: { code: ErrCode.BUSINESS_ERR, msg: "需要更新插件版本才能继续使用" }
          });
          return;
        }
        console.log("✅ [VERSION CHECK] set_token proceeding...");
        handleSetTokenMessage(message, sendResponse);
        break;
      }
      case "open_tab": {
        handleOpenTabMessage(message, sendResponse);
        break;
      }
      case "close_current_tab": {
        handleCloseCurrentTab(message, _sender, sendResponse);
        break;
      }
      case "check_update": {
        updater.manualCheckForUpdates().then((result) => {
          sendResponse({
            success: true,
            data: result
          });
        }).catch((error) => {
          sendResponse({
            success: false,
            error: { code: ErrCode.BUSINESS_ERR, msg: error.message }
          });
        });
        break;
      }
    }
    async function handleCloseCurrentTab(message2, sender, sendResponse2) {
      try {
        console.debug(message2);
        if (sender.tab && sender.tab.id) {
          await chrome.tabs.remove(sender.tab.id);
          sendResponse2({ success: true });
        } else {
          throw new Error("不能关闭tab");
        }
      } catch (ex) {
        sendResponse2({ success: false, error: { code: ErrCode.BUSINESS_ERR, msg: "" } });
      }
    }
    async function handleOpenTabMessage(message2, sendResponse2) {
      try {
        await chrome.tabs.create({
          url: message2.data.url
        });
        sendResponse2({ success: true });
      } catch (e) {
        sendResponse2({ success: false, error: { code: ErrCode.BUSINESS_ERR, msg: "" } });
      }
    }
    async function checkAndSetAuth() {
      if (!hhToken.getToken()) {
        const token = await auth.getToken();
        if (token) {
          hhToken.setToken(token);
        } else {
          sendResponse({ success: false, error: { code: ErrCode.UN_AUTH, msg: "" } });
          return;
        }
      }
    }
  }
  async function handleIsLoggedInMessage(message, sendResponse) {
    console.log(message);
    if (hhToken.getToken()) {
      sendResponse({ success: true, data: { isLoggedIn: true } });
    } else {
      sendResponse({ success: true, data: { isLoggedIn: false } });
    }
  }
  async function handleSetTokenMessage(message, sendResponse) {
    try {
      const token = message.data.token;
      await auth.setToken(token);
      hhToken.setToken(token);
      sendResponse({ success: true });
    } catch (ex) {
      sendResponse({ success: false, error: { code: ErrCode.BUSINESS_ERR, msg: "设置权限失败" } });
    }
  }
  async function handleFetchImageMessage(message, sendResponse) {
    try {
      const url = message.data.url;
      const base64 = await _fetchImg(url);
      sendResponse({ success: true, data: { image: base64 } });
    } catch (ex) {
      console.log(ex);
      sendResponse({ success: false, error: { code: ErrCode.BUSINESS_ERR, msg: "获取图片信息失败" } });
    }
  }
  async function handleAddQuizMessage(message, sendResponse) {
    try {
      const subjectCN = message.data.subject;
      const subjectItem = subjectOptions.find((x) => x.label == subjectCN);
      if (!subjectItem) {
        throw new Error("不支持的科目");
      }
      const { data: { xkw_qa, quizImageDataUrl } } = message;
      const xkwQuiz = await quizapi.getXKWQuizByXKWQuizId({ xkwQuizId: xkw_qa.qaId });
      if (xkwQuiz.isOk()) {
        if (xkwQuiz.value) {
          throw new Error("该题已录入");
        }
      }
      const files = [];
      for (const [i, url] of quizImageDataUrl.entries()) {
        const file = _base64ToFile(url, `quiz${i}.png`);
        const uploadRes = await hhapi.uploadMedia(file);
        console.log("uploadRes: ", uploadRes);
        if (uploadRes.isErr()) {
          throw new Error("上传图片失败");
        } else {
          files.push({
            type: FileType.Image,
            coverUrl: "",
            originalFile: file,
            fileUrl: uploadRes.value.fileUrl,
            id: genId()
          });
        }
      }
      const bodyFiles = [];
      const bodyImgDataUrl = message.data.quizBodyImageDataUrl[0];
      if (bodyImgDataUrl) {
        const file = _base64ToFile(bodyImgDataUrl, `quiz_body.png`);
        const uploadRes = await hhapi.uploadMedia(file);
        if (uploadRes.isErr()) {
          throw new Error("上传图片失败");
        } else {
          bodyFiles.push({
            type: FileType.Image,
            coverUrl: "",
            originalFile: file,
            fileUrl: uploadRes.value.fileUrl,
            id: genId()
          });
        }
      }
      const difficulty = resolveDifficulty(xkw_qa.difficulty);
      const quizConfig = (await hhapi.getExamPaperConfig()).unwrap();
      let extra = {
        year: 0,
        difficulty_name: "一般",
        apply_scene: "",
        district: "",
        grade_name: "",
        xkw_kpoints: xkw_qa.knowledges.map((x) => ({ id: Number(x.id), name: x.name })),
        xkw_type_name: xkw_qa.type
      };
      try {
        const m = resolveMeta(xkw_qa.meta, quizConfig);
        if (m) {
          extra.year = Number(m.year) || 2025;
          extra.difficulty_name = difficulty || "一般";
          extra.apply_scene = m.scene;
          extra.district = m.district;
          extra.grade_name = m.grade;
        }
      } finally {
      }
      if (message.data.addQuizRecord) {
        const res = await quizapi.addQuizRecord({
          subject: subjectItem.value,
          action: "recognize",
          fromId: xkw_qa.qaId,
          fromType: "qaxkw_info",
          prompt: {
            files,
            text: "",
            bodyFile: bodyFiles[0]
          },
          extra
        });
        if (res.isErr()) {
          throw res.error;
        }
      }
      const addXKWRes = await quizapi.addXKWQuiz({
        difficulty: xkw_qa.difficulty,
        flow_input_image_urls: files.map((x) => x.fileUrl),
        flow_input_text: "",
        htmlSource: xkw_qa.htmlSource,
        kpoints_list: xkw_qa.knowledges,
        subject: subjectItem.value,
        qa_content_image_urls: bodyFiles.map((f) => f.fileUrl),
        type: xkw_qa.type,
        xkw_qa_id: xkw_qa.qaId,
        extra
      });
      if (addXKWRes.isErr()) {
        throw addXKWRes.error;
      }
      sendResponse({ success: true });
    } catch (ex) {
      const msg = ex.message || "";
      sendResponse({ success: false, error: { code: ErrCode.BUSINESS_ERR, msg } });
    }
  }
}
function resolveMeta(meta, quizConfig) {
  const regex = /^(\d{4}|\d{2}-\d{2})(.*?)·(.*?)·(.*)$/;
  const match = meta.match(regex);
  if (!match) {
    return null;
  }
  let [, year, grade, district, scene] = match;
  if (year) {
    let _year = year;
    if (year.includes("-")) {
      const arr = year.split("-");
      _year = arr[0] || "25";
      _year = "20" + _year;
    }
    year = _year;
  }
  if (district) {
    district = getDistrictCode(district, quizConfig);
  }
  return {
    scene: scene || "",
    grade: grade || "",
    district: district || "",
    year: year || "0"
  };
}
function getDistrictCode(name, config) {
  const areas2 = config.cities;
  const cities2 = config.cities;
  const provices2 = config.provices;
  const p = provices2.find((x) => x.name.includes(name));
  if (p) {
    return p.code;
  }
  const c = cities2.find((x) => x.name.includes(name));
  if (c) {
    return c.code;
  }
  const area = areas2.find((x) => x.name.includes(name));
  if (area) {
    return area.code;
  }
  return "";
}
function resolveDifficulty(val) {
  let r = val;
  const matchs = val.match(/(.*?)\(.*\)/);
  if (matchs && matchs[1]) {
    r = matchs[1];
  }
  const map = {
    "容易": "简单",
    "较易": "较易",
    "适中": "一般",
    "较难": "较难",
    "困难": "困难"
  };
  if (map[r]) {
    return map[r];
  } else {
    return "一般";
  }
}
async function _fetchImg(src) {
  const res = await fetch(src);
  const buffer = await res.arrayBuffer();
  const base64 = await _arrayBufferToBase64(buffer);
  return base64;
}
function _arrayBufferToBase64(buffer) {
  return new Promise((resolve, reject) => {
    const blob = new Blob([buffer]);
    const reader = new FileReader();
    reader.onload = () => {
      const result = reader.result;
      const base64 = result.split(",")[1];
      if (base64) {
        resolve(base64);
      } else {
        reject(new Error("Failed to convert ArrayBuffer to Base64"));
      }
    };
    reader.onerror = reject;
    reader.readAsDataURL(blob);
  });
}
function _base64ToFile(base64, filename) {
  const arr = base64.split(",");
  if (arr.length < 2) {
    throw new Error("Invalid base64 string");
  }
  const mime = arr[0].match(/:(.*?);/)[1];
  const bstr = atob(arr[1]);
  let n = bstr.length;
  const u8arr = new Uint8Array(n);
  while (n--) {
    u8arr[n] = bstr.charCodeAt(n);
  }
  return new File([u8arr], filename, { type: mime });
}
